-- Top-level entity for the POTS project. Note: KEY(0) resets the
-- ADV7181, while KEY(2) resets the I2C controller. When first starting
-- the board, it is necessary to press KEY(0) and then KEY(2) to reset both.
-- This sometimes needs to be done multiple times before the ADV7181
-- produces valid data.

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity pots is
  port (
    -- Clocks
   
    CLOCK_27,                                      -- 27 MHz
    CLOCK_50 : in std_logic;                       -- 50 MHz

    -- SDRAM
   
    DRAM_DQ : inout std_logic_vector(15 downto 0); -- Data Bus
    DRAM_ADDR : out std_logic_vector(11 downto 0); -- Address Bus    
    DRAM_LDQM,                                     -- Low-byte Data Mask 
    DRAM_UDQM,                                     -- High-byte Data Mask
    DRAM_WE_N,                                     -- Write Enable
    DRAM_CAS_N,                                    -- Column Address Strobe
    DRAM_RAS_N,                                    -- Row Address Strobe
    DRAM_CS_N,                                     -- Chip Select
    DRAM_BA_0,                                     -- Bank Address 0
    DRAM_BA_1,                                     -- Bank Address 0
    DRAM_CLK,                                      -- Clock
    DRAM_CKE : out std_logic;                      -- Clock Enable

    -- SRAM
    
    SRAM_DQ : inout std_logic_vector(15 downto 0);         -- Data bus 
    SRAM_ADDR : out std_logic_vector(17 downto 0);         -- Address bus 
    SRAM_UB_N,                                     -- High-byte Data Mask 
    SRAM_LB_N,                                     -- Low-byte Data Mask 
    SRAM_WE_N,                                     -- Write Enable
    SRAM_CE_N,                                     -- Chip Enable
    SRAM_OE_N : out std_logic;                     -- Output Enable

    -- VGA output
    
    VGA_CLK,                            -- Clock
    VGA_HS,                             -- H_SYNC
    VGA_VS,                             -- V_SYNC
    VGA_BLANK,                          -- BLANK
    VGA_SYNC : out std_logic;           -- SYNC
    VGA_R,                              -- Red[9:0]
    VGA_G,                              -- Green[9:0]
    VGA_B : out std_logic_vector(9 downto 0);   -- Blue[9:0]

    -- TV input
    
    TD_DATA  : in unsigned(7 downto 0);  -- Data bus
    TD_HS,                               -- H_SYNC
    TD_VS    : in std_logic;             -- V_SYNC
    TD_RESET : out std_logic;             -- Reset

    -- Buttons and switches
    
    KEY : in std_logic_vector(3 downto 0);         -- Push buttons
    SW : in std_logic_vector(17 downto 0);         -- DPDT switches

    -- I2C bus

    I2C_SDAT : inout std_logic; -- I2C Data
    I2C_SCLK : out   std_logic;  -- I2C Clock

    -- UART

    UART_TXD : out std_logic;
    UART_RXD : in std_logic
  );
end pots;

architecture datapath of pots is
  
component nios_system is
    port (
      -- global signals
      signal clk     : in std_logic;
	  signal reset_n : in std_logic;	

      -- SDRAM
      signal zs_addr_from_the_sdram : out std_logic_vector(11 downto 0);
      signal zs_ba_from_the_sdram : out std_logic_vector (1 downto 0);
      signal zs_cas_n_from_the_sdram : out std_logic;
      signal zs_cke_from_the_sdram : out std_logic;
      signal zs_cs_n_from_the_sdram : out std_logic;
      signal zs_dq_to_and_from_the_sdram : inout std_logic_vector(15 downto 0);
      signal zs_dqm_from_the_sdram : out std_logic_vector (1 downto 0);
      signal zs_ras_n_from_the_sdram : out std_logic;
      signal zs_we_n_from_the_sdram : out std_logic;

      -- VGA framebuffer
      signal sram_addr_from_the_vga      : out std_logic_vector (17 downto 0);
      signal sram_ce_n_from_the_vga      : out std_logic;
      signal sram_dq_to_and_from_the_vga : inout std_logic_vector (15 downto 0);
      signal sram_lb_n_from_the_vga : out std_logic;
      signal sram_oe_n_from_the_vga : out std_logic;
      signal sram_ub_n_from_the_vga : out std_logic;
      signal sram_we_n_from_the_vga : out std_logic;
      signal vga_blank_from_the_vga : out std_logic;
      signal vga_b_from_the_vga : out std_logic_vector (9 downto 0);
      signal vga_clk_from_the_vga : out std_logic;
      signal vga_g_from_the_vga : out std_logic_vector (9 downto 0);
      signal vga_hs_from_the_vga : out std_logic;
      signal vga_r_from_the_vga : out std_logic_vector (9 downto 0);
      signal vga_sync_from_the_vga : out std_logic;
      signal vga_vs_from_the_vga : out std_logic;

      -- TV input
      signal clk27_to_the_tv_in   : in std_logic;
      signal td_data_to_the_tv_in : in unsigned(7 downto 0);
      signal td_hs_to_the_tv_in : in std_logic;
      signal td_vs_to_the_tv_in : in std_logic;

      -- UART
      signal rxd_to_the_uart : in std_logic;
      signal txd_from_the_uart : out std_logic
    );
  end component;

  component sdram_pll
    port (
      inclk0 : in std_logic;
      c0 : out std_logic;
      c1 : out std_logic
    );
  end component;

  component de2_i2c_av_config is
    port (
      iCLK : in std_logic;
      iRST_N : in std_logic;
      I2C_SCLK : out std_logic;
      I2C_SDAT : inout std_logic
    );
  end component;      

  signal ba  : std_logic_vector(1 downto 0);
  signal dqm : std_logic_vector(1 downto 0);

  signal pll_c1 : std_logic;

  signal sram_addr_conv : std_logic_vector(17 downto 0);
  signal sram_dq_conv   : std_logic_vector(15 downto 0);

  signal vga_r_conv : std_logic_vector(9 downto 0);
  signal vga_g_conv : std_logic_vector(9 downto 0);
  signal vga_b_conv : std_logic_vector(9 downto 0);

begin
  TD_RESET <= KEY(0);
  
  dram_ba_1 <= ba(1);
  dram_ba_0 <= ba(0);
  dram_udqm <= dqm(1); 
  dram_ldqm <= dqm(0);
  
  nios: nios_system port map (
    clk => pll_c1,
    reset_n => KEY(1),
    
    zs_addr_from_the_sdram => DRAM_ADDR,
    zs_ba_from_the_sdram => BA,
    zs_cas_n_from_the_sdram => DRAM_CAS_N,
    zs_cke_from_the_sdram => DRAM_CKE,
    zs_cs_n_from_the_sdram => DRAM_CS_N,
    zs_dq_to_and_from_the_sdram => DRAM_DQ,
    zs_dqm_from_the_sdram => DQM,
    zs_ras_n_from_the_sdram => DRAM_RAS_N,
    zs_we_n_from_the_sdram => DRAM_WE_N,
    
    sram_addr_from_the_vga => SRAM_ADDR,
    sram_ce_n_from_the_vga => SRAM_CE_N,
    sram_dq_to_and_from_the_vga => SRAM_DQ,
    sram_lb_n_from_the_vga => SRAM_LB_N,
    sram_oe_n_from_the_vga => SRAM_OE_N,
    sram_ub_n_from_the_vga => SRAM_UB_N,
    sram_we_n_from_the_vga => SRAM_WE_N,
    
    vga_blank_from_the_vga => VGA_BLANK,
    vga_b_from_the_vga => VGA_B,
    vga_clk_from_the_vga => VGA_CLK,
    vga_g_from_the_vga => VGA_G,
    vga_hs_from_the_vga => VGA_HS,
    vga_r_from_the_vga => VGA_R,
    vga_sync_from_the_vga => VGA_SYNC,
    vga_vs_from_the_vga => VGA_VS,
    
    clk27_to_the_tv_in => CLOCK_27,
    td_data_to_the_tv_in => TD_DATA,
    td_hs_to_the_tv_in => TD_HS,
    td_vs_to_the_tv_in => TD_VS,

    rxd_to_the_uart => uart_rxd,
    txd_from_the_uart => uart_txd
  );
  
  neg_3ns : sdram_pll port map (CLOCK_50, DRAM_CLK, pll_c1);

  i2c: de2_i2c_av_config port map (
    iCLK => CLOCK_50,
    iRST_N => KEY(2),
    I2C_SCLK => I2C_SCLK,
    I2C_SDAT => I2C_SDAT
  );
end datapath;
      
      

    
