#include <stdio.h>
#include <stdlib.h>
#include <system.h>
#include <sys/alt_dma.h>
#include <io.h>
#include "img_tracking.h"

#define VMAX 400 // maximum velocity allowed by the robot

#define CROSSCOLOR 0x0000

#define IORD_TV_IN_FIELD(base)  \
    IORD_16DIRECT(base, 1024)
#define IORD_TV_IN_HCOUNT(base)	\
    IORD_16DIRECT(base, 2048)
#define IORD_TV_IN_VCOUNT(base)	\
    IORD_16DIRECT(base, 3072)

#define HRES 320
#define VRES 200

static volatile int rx_done = 0;
static volatile int tx_done = 0;

FILE *serial;

static void snd_done(void *handle)
{
    tx_done = 1;
}

static void rcv_done(void *handle, void *data)
{
    rx_done = 1;
}

static void draw_crosshair(short *buf, int x, int y) {
   IOWR_SDRAM(buf+y*HRES+x, 0xffff);
    if (y < VRES-1) {
       IOWR_SDRAM(buf+(y+1)*HRES+x, CROSSCOLOR);  
    }
    if (y > 1) {
     IOWR_SDRAM(buf+(y-1)*HRES+x, CROSSCOLOR);
    }
    if (x < HRES-1) {
     IOWR_SDRAM(buf+y*HRES+(x+1), CROSSCOLOR);
    }
    if (x > 1) {
       IOWR_SDRAM(buf+y*HRES+(x-1), CROSSCOLOR);
    }
}


static inline void initSerial()
{ 
    serial = fopen("/dev/uart", "r+");
    if (!serial) {
        perror("fopen");
        exit(1);   
    }
    printf("Serial is opened!\n");
}

static inline void writeSerial(unsigned char *buf, int bytes)
{
    int written;

    if (!serial) {
        printf("Serial port not opened, returning...\n");
        return;
    }

    written = fwrite(buf, 1, bytes, serial);
    if (written == -1) {
        perror("write");   
    }
}

static inline void startRobot()
{
    unsigned char data[2];
    data[0] = (unsigned char) 128;
    data[1] = (unsigned char) 131;
    writeSerial(data, 2);
}

static inline void turnRight(int speed)
{
    unsigned char data[5];
    
    if (speed < 1 || speed > VMAX)
        return;
    
    data[0] = 145;  //Direct Drive command
    data[1] = (unsigned char) ((-speed) >> 8 & 0x00FF);  //[Right velocity high byte] 
    data[2] = (unsigned char) ((-speed) & 0x00FF);    //[Right velocity low byte]
    data[3] = (unsigned char) ((speed) >> 8 & 0x00FF);  //[Left velocity high byte]
    data[4] = (unsigned char) ((speed) & 0x00FF);    //[Left velocity low byte]
    
    writeSerial(data, 5);
}


static inline void turnLeft(int speed)
{
    unsigned char data[5];
    
    if (speed < 1 || speed > VMAX)
        return;
    
    data[0] = 145;  //Direct Drive command
    data[1] = (unsigned char) ((speed) >> 8 & 0x00FF);  //[Right velocity high byte] 
    data[2] = (unsigned char) ((speed) & 0x00FF);    //[Right velocity low byte]
    data[3] = (unsigned char) ((-speed) >> 8 & 0x00FF);  //[Left velocity high byte]
    data[4] = (unsigned char) ((-speed) & 0x00FF);    //[Left velocity low byte]
    
    writeSerial(data, 5);
}

static inline void stopRobot()
{
    unsigned char data[5];
    
    data[0] = 145;  //Direct Drive command
    data[1] = (unsigned char) 0;  //[Right velocity high byte] 
    data[2] = (unsigned char) 0;    //[Right velocity low byte]
    data[3] = (unsigned char) 0;  //[Left velocity high byte]
    data[4] = (unsigned char) 0;    //[Left velocity low byte]
    
    writeSerial(data, 5);
}

int main()
{
  int i, j = 0, rc, ret, speed;
  unsigned short hc, vc, vc2;

  int row=-1, col=-1;
  
  alt_dma_txchan txchan;
  alt_dma_rxchan rxchan;
  
  printf("started running!\n");

  initSerial();
  startRobot();
    
  printf("Initialized robot!\n");
  
  short *buf = malloc(sizeof(short) * (HRES*VRES));
  if (buf == NULL) {
    printf("Could not alloc!\n");
    exit(1); 
  }
  
  // clear buffer at first
  for (i=0; i < HRES*VRES; ++i) {
    IOWR_SDRAM(buf + i, 0);
  }
   
  // set up DMA
  if ((txchan = alt_dma_txchan_open("/dev/dma")) == NULL) {
    printf("Failed to open transmit channel.\n");
    exit(1); 
  }
  
  if ((rxchan = alt_dma_rxchan_open("/dev/dma")) == NULL) {
    printf("Failed to open receive channel.\n");
    exit(1); 
  }
  
  ret = alt_dma_txchan_ioctl(txchan, ALT_DMA_SET_MODE_16, NULL);
  if (ret < 0) {
    printf("IOCTL failed, could not set mode 16.\n");
    exit(1); 
  }
  
  ret = alt_dma_rxchan_ioctl(rxchan, ALT_DMA_SET_MODE_16, NULL);
  if (ret < 0) {
    printf("IOCTL failed, could not set mode 16.\n");
    exit(1); 
  }
  
  ret = alt_dma_rxchan_ioctl(rxchan, ALT_DMA_RX_ONLY_OFF, NULL);
  
  if (ret < 0) {
    printf("IOCTL failed, could not set default mode.\n");
    exit(1); 
  }
  ret = alt_dma_txchan_ioctl(txchan, ALT_DMA_TX_ONLY_OFF, NULL);
  if (ret < 0) {
    printf("IOCTL failed, could not set default mode.\n");
    exit(1); 
  }
  
  // main program loop
  while(1) {
  rx_done = 0;
  
  // transfer one dummy line so that the next transfer doesn't
  // result in a delay
  alt_dma_txchan_send(txchan, (void *)(TV_IN_BASE + 176), 
		      HRES*2, snd_done, NULL);
  alt_dma_rxchan_prepare(rxchan, buf, HRES*2, rcv_done, NULL);
  while (rx_done == 0);
  rx_done = 0;
  // wait for vsync
  while (IORD_TV_IN_HCOUNT(TV_IN_BASE) > 100 || 
	 (IORD_TV_IN_FIELD(TV_IN_BASE) & 0x0004) == 1 ||
         IORD_TV_IN_VCOUNT(TV_IN_BASE) != 0);
  for(i = 0; i < VRES; i++)
    {
      // wait for hsync
       do {
        hc = IORD_TV_IN_HCOUNT(TV_IN_BASE);
       }
       while (hc < 200 || hc > 300);
 
       alt_dma_txchan_send(txchan, (void *)(TV_IN_BASE + 176),
			   HRES*2, snd_done, NULL);
       alt_dma_rxchan_prepare(rxchan, buf+HRES*i, HRES*2, rcv_done, NULL);
       
       
       
       while (rx_done == 0);
       rx_done = 0;
    } 
    rx_done = 0;

  
    track_obj4(buf, row, col, &row, &col);
    draw_crosshair(buf, col, row);
    
    // set the speed of the robot based on how far the center of the
    // object is from the center of the image
    speed = col - 160;
    if (speed < 0)
        speed = -speed;
    speed += 40;
    if (speed > 80) speed = 80;
    
    // turn the robot
    if (col < 145) {
        turnLeft(speed);
    }
    else if (col > 175) {
        turnRight(speed);
    }
    else {
        stopRobot();
    }
    
    // transfer from buffer to video buffer in SRAM
    alt_dma_txchan_send(txchan, buf, HRES*VRES*2, snd_done, NULL);
    alt_dma_rxchan_prepare(rxchan, (void*)(VGA_BASE), HRES*VRES*2, 
			   rcv_done, NULL);
    while (!tx_done);
    tx_done = 0;
  }
  
  free(buf);
  return 0;
}
