library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity adv_interface is

  port (
    clk27     : in std_logic;
    reset     : in std_logic;
    
    -- data is output two pixels (= 32 bits) at a time
    out_data  : out unsigned(31 downto 0);
    -- this signal is asserted to indicate that the data in
    -- out_data will be valid two rising edges of td_clk27 later
    data_good : out std_logic;
    field_out   : out std_logic;
    hcount : out unsigned(10 downto 0);
    vcount: out unsigned(9 downto 0);   
    -- ADV7181 decoder signals
    td_data   : in unsigned(7 downto 0);
    td_hs     : in std_logic;	
    td_vs     : in std_logic
  );

end adv_interface;

architecture rtl of adv_interface is


  -- data is temporarily put here before being copied to out_data
  signal data_buffer   : unsigned(31 downto 0) := (others => '0');

  -- keeps track of the horizontal raster position
  signal h_counter     : unsigned(10 downto 0);
  -- keeps track of the current line
  signal v_counter     : unsigned(9 downto 0);
    -- 0 if this is the first interlaced field, 1 if this is the second
  signal field         : std_logic := '0';

  -- high whenever horizontal/vertical sync is pulled low; low whenever
  -- horizontal/vertical sync is pulled high again. Ensures that certain
  -- actions are done only once on the falling edge of horiz/vertical sync.
  signal hit_hsync : std_logic := '0';
  signal hit_vsync : std_logic := '0';
begin

  -- continually copy data from ADV7181 into buffer and output it;
  -- also, update the horizontal and vertical counters as well as
  -- the current field
  GetData : process (clk27)
  begin
    if rising_edge(clk27) then
      if reset = '1' then
        data_buffer <= (others => '0');
        out_data <= (others => '0');
        data_good <= '0';
        h_counter <= (others => '0');
        v_counter <= (others => '0');
        hit_hsync <= '0';
        hit_vsync <= '0';
        field <= '0';
      else
        if h_counter(1 downto 0) = "00" then
          data_buffer(31 downto 24) <= td_data;
          data_good <= '1';
        elsif h_counter(1 downto 0) = "01" then
          data_buffer(23 downto 16) <= td_data;
        elsif h_counter(1 downto 0) = "10" then
          data_buffer(15 downto 8) <= td_data;
          data_good <= '0';
        else
          data_buffer(7 downto 0) <= td_data;
          out_data <= data_buffer;
        end if;
        h_counter <= h_counter + "00000000001";
        -- Hit hsync? Record that it happened, reset horizontal
        -- counter and increment vertical counter  
        if td_hs = '0' and hit_hsync = '0' then
          hit_hsync <= '1';
          h_counter <= "00000000000";
          v_counter <= v_counter + "0000000001";
        elsif td_hs = '1' then
          hit_hsync <= '0';
        end if;
        -- Hit vsync? Record that it happened, reset vertical counter,
        -- and change the field
        if td_vs = '0' and hit_vsync = '0' then
          hit_vsync <= '1';
          v_counter <= (others => '0');
          field <= not field;
        end if;
        if td_vs = '1' then
          hit_vsync <= '0';
        end if;
      end if;
    end if;
  end process GetData;

  hcount <= h_counter;
  vcount <= v_counter;
  field_out <= field;

end rtl;

