-------------------------------------------------------------------------------
-- Simple VGA raster display
--
-- Modified version of lab 3 code
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is

  port (

-- Avalon Bus Signals

    reset       : in  std_logic;
    clk         : in  std_logic;        -- Should be 25.125 MHz
    chipselect  : in  std_logic;
    write, read : in  std_logic;
    address     : in  std_logic_vector(17 downto 0);
    readdata    : out std_logic_vector(15 downto 0);
    writedata   : in  std_logic_vector(15 downto 0);
    byteenable  : in  std_logic_vector(1 downto 0);
    waitrequest : out std_logic;
-- SRAM Control Signals

    SRAM_DQ              : inout std_logic_vector(15 downto 0);
    SRAM_ADDR            : out   std_logic_vector(17 downto 0);
    SRAM_UB_N, SRAM_LB_N : out   std_logic;
    SRAM_WE_N, SRAM_CE_N : out   std_logic;
    SRAM_OE_N            : out   std_logic;

-- VGA control signals

    VGA_CLK,                             -- Clock
    VGA_HS,                              -- H_SYNC
    VGA_VS,                              -- V_SYNC
    VGA_BLANK,                           -- BLANK
    VGA_SYNC : out std_logic;            -- SYNC
    VGA_R,                               -- Red[9:0]
    VGA_G,                               -- Green[9:0]
    VGA_B    : out unsigned(9 downto 0)  -- Blue[9:0]
    );

end de2_vga_raster;

architecture rtl of de2_vga_raster is

  component sram_mux
    port (
      sel                  : in  std_logic;
      SRAM_DQ_IN_from_av1  : in  std_logic_vector(15 downto 0);
      SRAM_DQ_OUT_from_av1 : out std_logic_vector(15 downto 0);
      SRAM_ADDR_from_av1   : in  std_logic_vector(17 downto 0);
      SRAM_UB_N_from_av1   : in  std_logic;
      SRAM_LB_N_from_av1   : in  std_logic;
      SRAM_WE_N_from_av1   : in  std_logic;
      SRAM_CE_N_from_av1   : in  std_logic;
      SRAM_OE_N_from_av1   : in  std_logic;

      SRAM_DQ_IN_from_vga  : in  std_logic_vector(15 downto 0);
      SRAM_DQ_OUT_from_vga : out std_logic_vector(15 downto 0);
      SRAM_ADDR_from_vga   : in  std_logic_vector(17 downto 0);
      SRAM_UB_N_from_vga   : in  std_logic;
      SRAM_LB_N_from_vga   : in  std_logic;
      SRAM_WE_N_from_vga   : in  std_logic;
      SRAM_CE_N_from_vga   : in  std_logic;
      SRAM_OE_N_from_vga   : in  std_logic;

      SRAM_DQ   : inout std_logic_vector(15 downto 0);
      SRAM_ADDR : out   std_logic_vector(17 downto 0);
      SRAM_UB_N : out   std_logic;
      SRAM_LB_N : out   std_logic;
      SRAM_WE_N : out   std_logic;
      SRAM_CE_N : out   std_logic;
      SRAM_OE_N : out   std_logic
      );
  end component;






  -- Video parameters

  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;

  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  constant RECTANGLE_HSTART : integer := 100;
  constant RECTANGLE_HEND   : integer := 540;
  constant RECTANGLE_VSTART : integer := 100;
  constant RECTANGLE_VEND   : integer := 380;
  constant HCENTER          : integer := 320;
  constant Hleft            : integer := 160;
  constant Hright           : integer := 480;
  constant VCENTER          : integer := 240;
  constant Vtop             : integer := 120;
  constant Vbot             : integer := 360;

  -- Signals for the video controller
  signal Hcount                : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount                : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals

  signal image_h, image_v, image_v_total, image_h_total, image_p : std_logic;  -- rectangle area
  signal clk25                                                   : std_logic := '0';

  -- Signals 
  signal IMAGE_BASE : unsigned(17 downto 0) := "000000000000000000";  --stores base address of image in SRAM
  signal SIZE       : unsigned(15 downto 0) := X"007F";  --size of image i.e. 200x200
  signal RED        : unsigned(9 downto 0)  := "0000000000";
  signal GREEN      : unsigned(9 downto 0)  := "0000000000";
  signal BLUE       : unsigned(9 downto 0)  := "0000000000";

  ------------------- Create Control Registers  -----------------
  -- cntl_regs( 0 to 2 ) = address of image in SRAM
  -- cntl_regs( 3 to 4 ) = image size i.e. 200x200 = 200
  -- cntl_regs( 5 ) = status register 
  -- status register = invalid_W/valid_W|invalid_R/valid_R|0|0|0|0|0|GO/DONE|   
  type regs_type is array(0 to 7) of std_logic_vector(7 downto 0);
  signal cntl_regs : regs_type;
  signal addr_cnt  : unsigned(17 downto 0) := "000000000000000000";
  signal CE_STATE  : std_logic             := '1';

  signal SRAM_DQ_IN_from_vga  : std_logic_vector(15 downto 0);
  signal SRAM_DQ_OUT_from_vga : std_logic_vector(15 downto 0);
  signal SRAM_ADDR_from_vga   : std_logic_vector(17 downto 0);
  signal SRAM_UB_N_from_vga   : std_logic;
  signal SRAM_LB_N_from_vga   : std_logic;
  signal SRAM_WE_N_from_vga   : std_logic;
  signal SRAM_CE_N_from_vga   : std_logic;
  signal SRAM_OE_N_from_vga   : std_logic;
  signal sel                  : std_logic;
begin

  -- static signals 
  SRAM_OE_N_from_vga <= '0';
  SRAM_WE_N_from_vga <= '1';
  SRAM_UB_N_from_vga <= '0';
  SRAM_LB_N_from_vga <= '0';
  SRAM_CE_N_from_vga <= '0';

  -- VGA uses SRAM as FRAME BUFFER
  sram_mux0 : sram_mux port map (
    sel                  => sel,
    SRAM_DQ_IN_from_av1  => writedata,
    SRAM_DQ_OUT_from_av1 => readdata,
    SRAM_ADDR_from_av1   => address,
    SRAM_UB_N_from_av1   => not byteenable(1),
    SRAM_LB_N_from_av1   => not byteenable(0),
    SRAM_WE_N_from_av1   => not write,
    SRAM_CE_N_from_av1   => not chipselect,
    SRAM_OE_N_from_av1   => not read,

    SRAM_DQ_IN_from_vga  => SRAM_DQ_IN_from_vga,
    SRAM_DQ_OUT_from_vga => SRAM_DQ_OUT_from_vga,
    SRAM_ADDR_from_vga   => SRAM_ADDR_from_vga,
    SRAM_UB_N_from_vga   => SRAM_UB_N_from_vga,
    SRAM_LB_N_from_vga   => SRAM_LB_N_from_vga,
    SRAM_WE_N_from_vga   => SRAM_WE_N_from_vga,
    SRAM_CE_N_from_vga   => SRAM_CE_N_from_vga,
    SRAM_OE_N_from_vga   => SRAM_OE_N_from_vga,

    SRAM_DQ   => SRAM_DQ,
    SRAM_ADDR => SRAM_ADDR,
    SRAM_UB_N => SRAM_UB_N,
    SRAM_LB_N => SRAM_LB_N,
    SRAM_WE_N => SRAM_WE_N,
    SRAM_CE_N => SRAM_CE_N,
    SRAM_OE_N => SRAM_OE_N
    );



  process (clk)
  begin
    if rising_edge(clk) then
      clk25 <= not clk25;
    end if;
  end process;

  -- Horizontal and vertical counters

  HCounter : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';

  VCounter : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        Vcount   <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;

  HBlankGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;
    end if;
  end process HBlankGen;

  VSyncGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        vga_vsync   <= '1';
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;
    end if;
  end process VSyncGen;

  VBlankGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        vga_vblank   <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

  -- Rectangle generator

  -- Vertical edges of the four squares that hold images
  PosHGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' or Hcount = HSYNC + HBACK_PORCH + Hleft - (SIZE srl 1) - 1 then
        image_h_total <= '1';
        image_h       <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH + Hleft + (SIZE srl 1) + 1 then
        image_h       <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + Hright - (SIZE srl 1) - 1 then
        image_h       <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH + Hright + (SIZE srl 1) + 1 then
        image_h       <= '0';
        image_h_total <= '0';
      end if;
    end if;
  end process PosHGen;

  -- Horizontal edges
  PosVGen : process (clk25)
  begin
    if rising_edge(clk25) then
      if reset = '1' then
        image_v         <= '0';
        image_v_total   <= '0';
      elsif EndOfLine = '1' then
        if Vcount = (VSYNC + VBACK_PORCH - 1) + Vtop - (SIZE srl 1) - 1 then
          image_v_total <= '1';
          image_v       <= '1';
        elsif Vcount = (VSYNC + VBACK_PORCH - 1) + Vtop + (SIZE srl 1) + 1 then
          image_v       <= '0';
        elsif Vcount = (VSYNC + VBACK_PORCH - 1) + Vbot - (SIZE srl 1) - 1 then
          image_v       <= '1';
        elsif Vcount = (VSYNC + VBACK_PORCH - 1) + Vbot + (SIZE srl 1) + 1 then
          image_v_total <= '0';
          image_v       <= '0';
        end if;

      end if;
    end if;
  end process PosVGen;

  image_p     <= image_h and image_v;
  sel         <= image_p;
  waitrequest <= image_p;
  -- Get data from SRAM

  PixelData : process ( clk25, reset )
  begin
    if reset = '1' then
      RED                    <= "0000000000";
      GREEN                  <= "0000000000";
      BLUE                   <= "0000000000";
    else
      if rising_edge(clk25) then
        if (image_v_total or image_h_total) = '0' then
          addr_cnt           <= "000000000000000000";
        elsif image_p = '1' then
	  -- retrieve the pixels from SRAM
          SRAM_ADDR_from_vga <= std_logic_vector(IMAGE_BASE + addr_cnt);
          RED                <= unsigned("1111111111" and SRAM_DQ_OUT_from_vga(14 downto 10)) sll 5;
          GREEN              <= unsigned("1111111111" and SRAM_DQ_OUT_from_vga(9 downto 5)) sll 5;
          BLUE               <= unsigned("1111111111" and SRAM_DQ_OUT_from_vga(4 downto 0)) sll 5;
          addr_cnt           <= addr_cnt + 1;
        else
          addr_cnt           <= addr_cnt;
        end if;
      end if;
    end if;
  end process PixelData;

  -- Registered video signals going to the video DAC

  VideoOut : process (clk25, reset)
  begin
    if reset = '1' then
      VGA_R   <= "0000000000";
      VGA_G   <= "0000000000";
      VGA_B   <= "0000000000";
    elsif clk25'event and clk25 = '1' then
      if image_p = '1' then
        VGA_R <= RED;
        VGA_G <= GREEN;
        VGA_B <= BLUE;
      elsif vga_hblank = '0' and vga_vblank = '0' then
        VGA_R <= "0000000000";
        VGA_G <= "1111111111";
        VGA_B <= "0000000000";
      else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk25;
  VGA_HS  <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
