/*
 * CSEE 4840 Lab 2: Ethernet packet send and receive
 *
 * Stephen A. Edwards et al.
 *
 */

#include "basic_io.h"
#include "DM9000A.h"
#include <alt_types.h>
#include "alt_up_ps2_port.h"
#include "ps2_keyboard.h"
#include "ps2_mouse.h"
#include "LCD.h"
#include "VGA.h"

#define MAX_MSG_LENGTH 128

// Ethernet MAC address.  Choose the last three bytes yourself
unsigned char mac_address[6] = { 0x01, 0x60, 0x6E, 0x11, 0x02, 0x0F  };

unsigned int interrupt_number;

unsigned int receive_buffer_length;
unsigned char receive_buffer[1600];

KB_CODE_TYPE decode_mode;

#define UDP_PACKET_PAYLOAD_OFFSET 42
#define UDP_PACKET_LENGTH_OFFSET 38

#define UDP_PACKET_PAYLOAD (transmit_buffer + UDP_PACKET_PAYLOAD_OFFSET)

unsigned char transmit_buffer[] = {
  // Ethernet MAC header
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, // Destination MAC address
  0x01, 0x60, 0x6E, 0x11, 0x02, 0x0F, // Source MAC address
  0x08, 0x00,                         // Packet Type: 0x800 = IP
                          
  // IP Header
  0x45,                // version (IPv4), header length = 20 bytes
  0x00,                // differentiated services field
  0x00,0x9C,           // total length: 20 bytes for IP header +
                       // 8 bytes for UDP header + 128 bytes for payload
  0x3d, 0x35,          // packet ID
  0x00,                // flags
  0x00,                // fragment offset
  0x80,                // time-to-live
  0x11,                // protocol: 11 = UDP
  0xa3,0x43,           // header checksum: incorrect
  0xc0,0xa8,0x01,0x01, // source IP address
  0xc0,0xa8,0x01,0xff, // destination IP address
                          
  // UDP Header
  0x67,0xd9, // source port port (26585: garbage)
  0x27,0x2b, // destination port (10027: garbage)
  0x00,0x88, // length (136: 8 for UDP header + 128 for data)
  0x00,0x00, // checksum: 0 = none
                          
  // UDP payload
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x73, 0x67
};   

static void ethernet_interrupt_handler() {
  unsigned int receive_status;
  int i;
  
  receive_status = ReceivePacket(receive_buffer, &receive_buffer_length);

  if (receive_status == DMFE_SUCCESS) {

#if 1
    printf("\n\nReceive Packet Length = %d", receive_buffer_length);
    for(i=0;i<receive_buffer_length;i++) {
      if (i%8==0) printf("\n");
      printf("0x%.2X,", receive_buffer[i]);
    }
    printf("\n");
#endif

    if (receive_buffer_length >= 14) {
      //  A real Ethernet packet
      if (receive_buffer[12] == 8 && receive_buffer[13] == 0 &&
	  receive_buffer_length >= 34) {
	// An IP packet
	if (receive_buffer[23] == 0x11) {
	  // A UDP packet
	  if (receive_buffer_length >= UDP_PACKET_PAYLOAD_OFFSET) {
	    printf("Received: %s\n",
		   receive_buffer + UDP_PACKET_PAYLOAD_OFFSET);
	  }
	} else {
	  printf("Received non-UDP packet\n");
	}
      } else {
	printf("Received non-IP packet\n");
      }
    } else {
      printf("Malformed Ethernet packet\n");
    }

  } else {
    printf("Error receiving packet\n");
  }

  /* Display the number of interrupts on the LEDs */
  interrupt_number++;
  outport(SEG7_DISPLAY_BASE, interrupt_number);

  /* Clear the DM9000A ISR: PRS, PTS, ROS, ROOS 4 bits, by RW/C1 */
  dm9000a_iow(ISR, 0x3F);
			  
  /* Re-enable DM9000A interrupts */
  dm9000a_iow(IMR, INTR_set);
}

int main()
{
  int curMsgChar = 0;
  alt_u8 key = 0;
  int status = 0;
  unsigned int packet_length;

  unsigned int row, col;

  VGA_Ctrl_Reg vga_ctrl_set;
  
  vga_ctrl_set.VGA_Ctrl_Flags.RED_ON    = 1;
  vga_ctrl_set.VGA_Ctrl_Flags.GREEN_ON  = 1;
  vga_ctrl_set.VGA_Ctrl_Flags.BLUE_ON   = 1;
  vga_ctrl_set.VGA_Ctrl_Flags.CURSOR_ON = 0;
  
  Vga_Write_Ctrl(VGA_0_BASE, vga_ctrl_set.Value);
  Set_Pixel_On_Color(1023,1023,1023);
  Set_Pixel_Off_Color(0,0,0);
  Set_Cursor_Color(0,1023,0);

  // Initialize the LCD and display a welcome message
  LCD_Init();
  LCD_Show_Text("4840 Lab 2");

  // Clear the LEDs to zero (will display interrupt count)
  outport(SEG7_DISPLAY_BASE, 0);

  // Print a friendly welcome message
  printf("4840 Lab 2 started\n");

  // Initalize the DM9000 and the Ethernet interrupt handler
  DM9000_init(mac_address);
  interrupt_number = 0;
  alt_irq_register(DM9000A_IRQ, NULL, (void*)ethernet_interrupt_handler); 
 
  // Initialize the keyboard
  printf("Please wait three seconds to initialize keyboard\n");
  clear_FIFO();
  switch (get_mode()) {
  case PS2_KEYBOARD:
    break;
  case PS2_MOUSE:
    printf("Error: Mouse detected on PS/2 port\n");
    goto ErrorExit;
  default:
    printf("Error: Unrecognized or no device on PS/2 port\n");
    goto ErrorExit;
  }

  printf("Ready to send messages\n");

  /*
  for (row = 0 ; row < 30 ; row++)
    for (col = 0 ; col < 80 ; col++)
      put_vga_char(col + row * 80, col, row);  
  */

  put_vga_string("Hello World!", 0, 0);

  // Clear the payload
  for (curMsgChar=MAX_MSG_LENGTH-1; curMsgChar>0; curMsgChar--) {
    UDP_PACKET_PAYLOAD[curMsgChar] = 0;
  }

  for (;;) { 
    // wait for the user's input and get the make code
    status = read_make_code(&decode_mode, &key);
    if (status == PS2_SUCCESS) {
      // print out the result
      switch ( decode_mode ) {
      case KB_ASCII_MAKE_CODE :
	printf("%c", key );
	
	if (curMsgChar < MAX_MSG_LENGTH) { 
	  UDP_PACKET_PAYLOAD[curMsgChar] = key;
	  curMsgChar++; 
	}
	break ;
      case KB_LONG_BINARY_MAKE_CODE :
	printf("%s", " LONG ");
	// fall through
      case KB_BINARY_MAKE_CODE :
	switch (key) {
	case  0x5a: //enter key: send the msg
	  printf("Msg to Send: "); 
	  UDP_PACKET_PAYLOAD[curMsgChar++] = 0; // Terminate the string
      packet_length = UDP_PACKET_PAYLOAD_OFFSET + curMsgChar;
      transmit_buffer[UDP_PACKET_LENGTH_OFFSET] = packet_length >> 8;
      transmit_buffer[UDP_PACKET_LENGTH_OFFSET + 1] = packet_length & 0xff;
	  if (TransmitPacket(transmit_buffer, UDP_PACKET_PAYLOAD_OFFSET + curMsgChar + 1)==DMFE_SUCCESS) { 
	    printf("\nMessage sent successfully\n");
	  } else {
	    printf("\nMessage sending failed\n"); 
	  }
	  // reset data
	  for (curMsgChar=MAX_MSG_LENGTH-1; curMsgChar>0; curMsgChar--) { 
	    UDP_PACKET_PAYLOAD[curMsgChar] = 0;
	  }
	  printf("Msg to Send: "); 
	  break; 
	case 0x29: //space key
	  UDP_PACKET_PAYLOAD[curMsgChar++] = ' ';
	  break;
	default:
	  printf(" MAKE CODE :\t%X\n", key ); //print other unknown breakcode
	}
  		      
	break ;
      case KB_BREAK_CODE :
	// do nothing
      default :
	break ;
      }
    }
    else {
      printf(" Keyboard error ....\n");
    }
  }

  printf("Program terminated normally\n");
  return 0;
    	
 ErrorExit:
  printf("Program terminated with an error condition\n");

  return 1;
}
