-------------------------------------------------------------------------------
--
--  Sokoban-like Video Game for the Digilent Spartan 3 starter board
--
-- 
--   Shih-chun-yu (Halu) 
-- 
-- 
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity opb_s3board_vga is
  
  generic (
    C_OPB_AWIDTH : integer                   := 32;
    C_OPB_DWIDTH : integer                   := 32;
    C_BASEADDR   : std_logic_vector(31 downto 0) := X"FEFF0000";
    C_HIGHADDR   : std_logic_vector(31 downto 0) := X"FEFF0FFF");

  port (
    OPB_Clk        : in std_logic;
    OPB_Rst        : in std_logic;

    -- OPB signals
    OPB_ABus       : in std_logic_vector (31 downto 0);
    OPB_BE         : in std_logic_vector (3 downto 0);
    OPB_DBus       : in std_logic_vector (31 downto 0);
    OPB_RNW        : in std_logic;
    OPB_select     : in std_logic;
    OPB_seqAddr    : in std_logic;

    VGA_DBus       : out std_logic_vector (31 downto 0);
    VGA_errAck     : out std_logic;
    VGA_retry      : out std_logic;
    VGA_toutSup    : out std_logic;
    VGA_xferAck    : out std_logic;

    Pixel_Clock_2x : in std_logic;

    -- Video signals
    VIDOUT_CLK     : out std_logic;
    VIDOUT_RED     : out std_logic;
    VIDOUT_GREEN   : out std_logic;
    VIDOUT_BLUE    : out std_logic;
    VIDOUT_HSYNC : out std_logic;
    VIDOUT_VSYNC : out std_logic);

end opb_s3board_vga;

architecture Behavioral of opb_s3board_vga is
  
  constant BASEADDR : std_logic_vector(31 downto 0) := X"FEFF0000";
 
  -- Video parameters
  
  constant HTOTAL : integer := 800;
  constant HSYNC : integer := 96;
  constant HBACK_PORCH : integer := 48;
  constant HACTIVE : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL : integer := 525;
  constant VSYNC : integer := 2;
  constant VBACK_PORCH : integer := 33;
  constant VACTIVE : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  -- 512 X 8 dual-ported Xilinx block RAM
  component RAMB4_S8_S8
    port (
      DOA   : out std_logic_vector (7 downto 0);
      ADDRA : in std_logic_vector (8 downto 0);
      CLKA  : in std_logic;
      DIA   : in std_logic_vector (7 downto 0);
      ENA   : in std_logic;
      RSTA  : in std_logic;
      WEA   : in std_logic;
      DOB   : out std_logic_vector (7 downto 0);
      ADDRB : in std_logic_vector (8 downto 0);
      CLKB  : in std_logic;
      DIB   : in std_logic_vector (7 downto 0);
      ENB   : in std_logic;
      RSTB  : in std_logic;
      WEB   : in std_logic);
  end component;

  -- Signals latched from the OPB
  signal ABus : std_logic_vector (31 downto 0);
  signal DBus : std_logic_vector (31 downto 0);
  signal RNW  : std_logic;
  signal select_delayed : std_logic;

  -- Latched output signals for the OPB
  signal DBus_out : std_logic_vector (31 downto 0);

  -- Signals for the OPB-mapped RAM
  signal ChipSelect : std_logic;            -- Address decode
  signal MemCycle1, MemCycle2 : std_logic;  -- State bits
  signal RamPageAddress : std_logic_vector(1 downto 0); 
  signal RamSelect : std_logic_vector (2 downto 0);
  signal RST, WE : std_logic_vector (2 downto 0);
  signal DOUT0, DOUT1, DOUT2 : std_logic_vector(7 downto 0);
 
  signal ReadData : std_logic_vector(7 downto 0);

  -- Signals for the video controller
  signal Pixel_Clock : std_logic;       -- 25 MHz clock divided from 50 MHz
  signal LoadNShift : std_logic;        -- Shift register control
  signal FontData : std_logic_vector(7 downto 0);   -- Input to shift register
  signal ShiftData : std_logic_vector(7 downto 0);  -- Shift register data
  signal VideoData : std_logic;         -- Serial out ANDed with blanking

  signal Hcount : std_logic_vector(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : std_logic_vector(9 downto 0);  -- Vertical position (0-524)
  signal HBLANK_N, VBLANK_N : std_logic;   -- Blanking signals
 
  signal FontLoad, LoadChar : std_logic;  -- Font/Character RAM read triggers
  signal FontAddr : std_logic_vector(9 downto 0);
   signal FontAddrDelayed1 : std_logic_vector(9 downto 0); 
    signal FontAddrDelayed2 : std_logic_vector(9 downto 0); 
   signal FontAddrDelayed : std_logic_vector(9 downto 0);   
  signal CharRamPage : std_logic;
  signal CharRamSelect_N : std_logic;
  signal FontRamPage : std_logic;
  signal FontRamSelect_N : std_logic_vector(1 downto 0);
  signal CharAddr : std_logic_vector(9 downto 0);
  signal CharColumn : std_logic_vector(9 downto 0);
  signal CharRow : std_logic_vector(9 downto 0);
  signal Column : std_logic_vector(4 downto 0); -- 0-19
  signal Row : std_logic_vector(3 downto 0);   -- 0-14
  signal EndOfLine, EndOfField : std_logic;

  signal DOUTB0, DOUTB1, DOUTB2 : std_logic_vector(7 downto 0);
  
  ---color control-----
  signal ColorFormat : std_logic_vector(2 downto 0);   ---select color of picture
  signal PicColor : std_logic_vector(2 downto 0);      ---the color of selecting picture
  signal ColorHCount : std_logic_vector(9 downto 0);  ---count in the 640*480 block (0-639)
  signal ColorVCount : std_logic_vector(9 downto 0);  ---count in the 640*480 block (0-479)
  signal PicHCount : std_logic_vector(4 downto 0);  --count PicHClock in the 640*480 block (0-39)
  signal PicVCount : std_logic_vector(4 downto 0);  --count PicVClock in the 640*480 block (0-29)
  signal PicHClock : std_logic;                       ---change every 8 ColorHCount in the 640*480 block 
  signal PicVClock : std_logic;                       ---change once every 8 ColorHCount in the 640*480 block 
  signal PicHBLANK_N : std_logic;  ---blank picture
  signal PicVBLANK_N : std_logic;  ---blank picture
  signal EndPicOfLine, EndPicOfField : std_logic;


  attribute INIT_00 : string;
  attribute INIT_01 : string;
  attribute INIT_02 : string;
  attribute INIT_03 : string;
  attribute INIT_04 : string;
  attribute INIT_05 : string;
  attribute INIT_06 : string;
  attribute INIT_07 : string;
  attribute INIT_08 : string;
  attribute INIT_09 : string;
  attribute INIT_0a : string;
  attribute INIT_0b : string;
  attribute INIT_0c : string;
  attribute INIT_0d : string;
  attribute INIT_0e : string;
  attribute INIT_0f : string;

    ------------------start memory map (backgroup)-------------------------
        attribute INIT_00 of RAMB4_S8_S8_0 : label is           
           "0000000000000000000000000000000000000000000000000007060504030201";
        attribute INIT_01 of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_02 of RAMB4_S8_S8_0 : label is
           "0000000000000100000401000000000000000000000000000000010101010100";
        attribute INIT_03 of RAMB4_S8_S8_0 : label is
           "0200010000000000000000000000010101000102020001000000000000000000";
        attribute INIT_04 of RAMB4_S8_S8_0 : label is
           "0000000000000103010101000101010000000000000000000000010301000100";
        attribute INIT_05 of RAMB4_S8_S8_0 : label is
           "0001000000010000000000000000000000000103000000000101000000000000";
        attribute INIT_06 of RAMB4_S8_S8_0 : label is
           "0000000000000000000001010101000000010000000000000000000000000100";
        attribute INIT_07 of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000001010101010000";
        attribute INIT_08 of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_09 of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0a of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0b of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0c of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0d of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0e of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
        attribute INIT_0f of RAMB4_S8_S8_0 : label is
           "0000000000000000000000000000000000000000000000000000000000000000";
   
   
  --------------end memory map--------------------------------------                        
   
  ------------- start ram for font of 32*32bits ------------------
  
         
         ----------start street-------------------------    
   attribute INIT_00 of RAMB4_S8_S8_1 : label is 
    "0000000000000000000000000000000000000000000000000000000000000000";     
   attribute INIT_01 of RAMB4_S8_S8_1 : label is 
    "0000000000000000000000000000000000000000000000000000000000000000";                                           
   attribute INIT_02 of RAMB4_S8_S8_1 : label is                                            
    "0000000000000000000000000000000000000000000000000000000000000000";
   attribute INIT_03 of RAMB4_S8_S8_1 : label is                                            
    "0000000000000000000000000000000000000000000000000000000000000000";  
         ----------end street-------------------------          
         
           ------starta a wall  ------------------ 
   attribute INIT_04 of RAMB4_S8_S8_1 : label is
    "000000f0000000f0000000f0000000f0000000f0000000f0ffffffffffffffff";                              
   attribute INIT_05 of RAMB4_S8_S8_1 : label is 
    "ffffffffffffffff000000f0000000f0000000f0000000f0000000f0000000f0";                                           
   attribute INIT_06 of RAMB4_S8_S8_1 : label is
    "00c0030000c0030000c0030000c0030000c0030000c00300ffffffffffffffff";                                            
   attribute INIT_07 of RAMB4_S8_S8_1 : label is  
    "ffffffffffffffff00c0030000c0030000c0030000c0030000c0030000c00300";
         -------end a wall----------------------
       
       ------starta a box------------------                
   attribute INIT_08 of RAMB4_S8_S8_1 : label is
      "2902409419018098890000915F0000FAF1FFFF8F150000A811000088FFFFFFFF";
   attribute INIT_09 of RAMB4_S8_S8_1 : label is
      "0960069009900990098811900944229009224490091188908908109149042092";
   attribute INIT_0a of RAMB4_S8_S8_1 : label is
      "4904209289081091091188900922449009442290098811900990099009600690";
   attribute INIT_0b of RAMB4_S8_S8_1 : label is
      "FFFFFFFF11000088150000A8F1FFFF8F5F0000FA890000911901809829024094"; 
       -------end a box----------------------
       
       ------starta a target  ------------------
   attribute INIT_0c of RAMB4_S8_S8_1 : label is
       "00E07F0000C07F0000807F0000007F0000007E0000007C000000780000007000" ;
   attribute INIT_0d of RAMB4_S8_S8_1 : label is
       "E0FF7F00C0FF7F0080FF7F0000FF7F0000FE7F0000FC7F0000F87F0000F07F00";
   attribute INIT_0e of RAMB4_S8_S8_1 : label is
       "0000700000007000000070000000700000007000000070000000700000007000";
   attribute INIT_0f of RAMB4_S8_S8_1 : label is
       "0000000000000000000000000000700000007000000070000000700000007000";
       ------end a target-----------------------
   
                                
         ------starta a pig  ------------------                       
   attribute INIT_00 of RAMB4_S8_S8_2 : label is  
       "00E4270000142800001E78000000000000000000000000000000000000000000" ; 
   attribute INIT_01 of RAMB4_S8_S8_2 : label is 
       "206246042004200440F81F024000000280600601000180000002400000020400";
   attribute INIT_02 of RAMB4_S8_S8_2 : label is
       "00FC3F000002400000C18300802004018000000140F81F024004200220624604"; 
   attribute INIT_03 of RAMB4_S8_S8_2 : label is 
       "0000000000000000000000000000000000000000000000000000000000700E00";
        -------end a pig------------------------ 
        
        ------start a pig (move right)----------
   attribute INIT_04 of RAMB4_S8_S8_2 : label is 
       "ffffffffffffffffffffffffffffffff00000000000000000000000000000000";  
   attribute INIT_05 of RAMB4_S8_S8_2 : label is   
       "00000000000000000000000000000000ffffffffffffffffffffffffffffffff"; 
   attribute INIT_06 of RAMB4_S8_S8_2 : label is
       "ffffffffffffffffffffffffffffffff00000000000000000000000000000000"; 
   attribute INIT_07 of RAMB4_S8_S8_2 : label is   
       "00000000000000000000000000000000ffffffffffffffffffffffffffffffff"; 
       --------------end------------------
       
       -------start a heart--------------
   attribute INIT_08 of RAMB4_S8_S8_2 : label is
      "E03FFE07C01FFC03800FF8010000000000000000000000000000000000000000";
   attribute INIT_09 of RAMB4_S8_S8_2 : label is
      "C0FFFF07C0FFFF0FE0FFFF0FE0FFFF0FE0FFFF0FE0FFFF0FE0FFFF0FE0FFFF0F";
   attribute INIT_0a of RAMB4_S8_S8_2 : label is
      "00E00F0000F01F0000F83F0000FC7F0000FEFF0000FEFF0100FFFF0380FFFF07";
   attribute INIT_0b of RAMB4_S8_S8_2 : label is
      "0000000000000000000000000000000000000000000001000080030000C00700";
          ---------end------------------
          
          --------start a pig (move left)-------------
 --- attribute INIT_0c of RAMB4_S8_S8_2 : label is
 --- attribute INIT_0d of RAMB4_S8_S8_2 : label is
 --- attribute INIT_0e of RAMB4_S8_S8_2 : label is
 --- attribute INIT_0f of RAMB4_S8_S8_2 : label is
         --------end-----------------
  
  
  
  
  
  

begin  -- Behavioral

  -----------------------------------------------------------------------------
  --
  -- Instances of the block RAMs
  -- Each is 512 bytes, so 4K total
  --
  -----------------------------------------------------------------------------

  -- First 2.5K is used for characters (5 block RAMs)
  -- Remaining 1.5K holds the font     (3 block RAMs)
  --
 
  -- Port A is used for communication with the OPB
  -- Port B is for video
 
  RAMB4_S8_S8_0 : RAMB4_S8_S8
    port map (
      DOA   => DOUT0,                      ---out std_logic_vector (7 downto 0); 
      ADDRA => ABus(8 downto 0),           ---in std_logic_vector (8 downto 0);  
      CLKA  => OPB_Clk,                    ---in std_logic;                      
      DIA   => DBus(7 downto 0),           ---in std_logic_vector (7 downto 0);  
      ENA   => '1',                        ---in std_logic;                      
      RSTA  => RST(0),                     ---in std_logic;                      
      WEA   => WE(0),                      ---in std_logic;                      
      DOB   => DOUTB0,                     ---out std_logic_vector (7 downto 0); 
      ADDRB => CharAddr(8 downto 0),       ---in std_logic_vector (8 downto 0);  
      CLKB  => Pixel_Clock,                ---in std_logic;                      
      DIB   => X"00",                      ---in std_logic_vector (7 downto 0);  
      ENB   => '1',                        ---in std_logic;                      
      RSTB  => CharRamSelect_N,            ---in std_logic;                      
      WEB   => '0');                       ---in std_logic);                     
 
                   

    RAMB4_S8_S8_1 : RAMB4_S8_S8
    port map (
      DOA   => DOUT1,                      ---out std_logic_vector (7 downto 0);
      ADDRA => ABus(8 downto 0),           ---in std_logic_vector (8 downto 0); 
      CLKA  => OPB_Clk,                    ---in std_logic;                     
      DIA   => DBus(7 downto 0),           ---in std_logic_vector (7 downto 0); 
      ENA   => '1',                        ---in std_logic;                     
      RSTA  => RST(1),                     ---in std_logic;                     
      WEA   => WE(1),                      ---in std_logic;                     
      DOB   => DOUTB1,                     ---out std_logic_vector (7 downto 0);
      ADDRB => FontAddr(8 downto 0),       ---in std_logic_vector (8 downto 0); 
      CLKB  => Pixel_Clock,                ---in std_logic;                     
      DIB   => X"00",                      ---in std_logic_vector (7 downto 0); 
      ENB   => '1',                        ---in std_logic;                     
      RSTB  => FontRamSelect_N(0),         ---in std_logic;                     
      WEB   => '0');                       ---in std_logic);                    

    RAMB4_S8_S8_2 : RAMB4_S8_S8
    port map (
      DOA   => DOUT2,                      ---out std_logic_vector (7 downto 0);
      ADDRA => ABus(8 downto 0),           ---in std_logic_vector (8 downto 0); 
      CLKA  => OPB_Clk,                    ---in std_logic;                     
      DIA   => DBus(7 downto 0),           ---in std_logic_vector (7 downto 0); 
      ENA   => '1',                        ---in std_logic;                     
      RSTA  => RST(2),                     ---in std_logic;                     
      WEA   => WE(2),                      ---in std_logic;                     
      DOB   => DOUTB2,                     ---out std_logic_vector (7 downto 0);
      ADDRB => FontAddr(8 downto 0),       ---in std_logic_vector (8 downto 0); 
      CLKB  => Pixel_Clock,                ---in std_logic;                     
      DIB   => X"00",                      ---in std_logic_vector (7 downto 0); 
      ENB   => '1',                        ---in std_logic;                     
      RSTB  => FontRamSelect_N(1),         ---in std_logic;                     
      WEB   => '0');                       ---in std_logic);       
                   

  -----------------------------------------------------------------------------
  --  
  -- OPB-RAM controller
  --
  -----------------------------------------------------------------------------
  
  -- Unused OPB control signals
  VGA_errAck <= '0';
  VGA_retry <= '0';
  VGA_toutSup <= '0';

  -- Latch the relevant OPB signals from the OPB, since they arrive late
  LatchOPB: process (OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Abus <= ( others => '0' );
      DBus <= ( others => '0' );
      RNW <= '1';
      select_delayed <= '0';
    elsif OPB_Clk'event and OPB_Clk = '1' then
      ABus <= OPB_ABus;
      DBus <= OPB_DBus;
      RNW <= OPB_RNW;
      select_delayed <= OPB_Select;
    end if;
  end process LatchOPB;

  -- Address bits 31 downto 11 is our chip select
  -- 10 downto 9 is the RAM page select
  -- 8 downto 0 is the RAM byte select
  
  ChipSelect <=
    '1' when select_delayed = '1' and
       (ABus(31 downto 11) = BASEADDR(31 downto 11)) and
       MemCycle1 = '0' and MemCycle2 = '0' else
    '0';

  RamPageAddress <= ABus(10 downto 9); 
  
 
     
  
  RamSelect <=
    "001" when RamPageAddress = "00" else
    "010" when RamPageAddress = "01" else
    "100" when RamPageAddress = "10" else
    "000";

  MemCycleFSM : process(OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      MemCycle1 <= '0';
      MemCycle2 <= '0';
    elsif OPB_Clk'event and OPB_Clk = '1' then
      MemCycle2 <= MemCycle1;
      MemCycle1 <= ChipSelect;
    end if;
  end process MemCycleFSM;
  
  VGA_xferAck <= MemCycle2;

  WE <=
  RamSelect when ChipSelect = '1' and RNW = '0' and OPB_Rst = '0' else
    "000";
 
  RST <=
   not RamSelect when ChipSelect = '1' and RNW = '1' and OPB_Rst = '0' else
    "111"; 

  ReadData <= DOUT0 or DOUT1 or DOUT2 when MemCycle1 = '1'
                                           else "00000000";

  -- DBus(31 downto 24) is the byte for addresses ending in 0

  GenDOut: process (OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      DBus_out <= ( others => '0');
    elsif OPB_Clk'event and OPB_Clk = '1' then
      DBus_out <= ReadData & ReadData & ReadData & ReadData;  ---32bits
    end if;
  end process GenDOut;

  VGA_DBus <= DBus_out;

  ----------------------------------------------------------------------------
  --
  -- Video controller
  --
  -----------------------------------------------------------------------------

 -- Pixel clock divider
  
  Pixel_clk_divider : process (Pixel_Clock_2x, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Pixel_Clock <= '0';
    elsif Pixel_Clock_2x'event and Pixel_Clock_2x = '1' then
      Pixel_Clock <= not Pixel_Clock;
    end if;
  end process Pixel_clk_divider;
  
  -- Horizontal and vertical counters

  HCounter : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      Hcount <= (others => '0');
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL  else '0';
 
  VCounter: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Vcount <= (others => '0');
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VIDOUT_HSYNC <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        VIDOUT_HSYNC <= '1';
      elsif Hcount = HSYNC -1 then
        VIDOUT_HSYNC <= '0';
      end if;      
    end if;
  end process HSyncGen;

  HBlankGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      HBLANK_N <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if Hcount = HSYNC + HBACK_PORCH -1 then
        HBLANK_N <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE - 1 then
        HBLANK_N <= '0';
      end if;      
    end if;
  end process HBlankGen;
  
  VSyncGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VIDOUT_VSYNC <= '1';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine ='1' then
        if EndOfField = '1' then
          VIDOUT_VSYNC <= '1';
        elsif VCount = VSYNC - 1 then
          VIDOUT_VSYNC <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VBLANK_N <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine ='1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          VBLANK_N <= '1';
        elsif VCount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          VBLANK_N <= '0';
        end if;
      end if;      
    end if;
  end process VBlankGen;
  
  
  -- RAM read triggers and shift register control

  LoadChar   <= '1' when Hcount(2 downto 0) = X"5" else '0';
  FontLoad   <= '1' when Hcount(2 downto 0) = X"7" else '0'; 
  LoadNShift <= '1' when Hcount(2 downto 0) = X"7" else '0';

  -- Correction of 4 needed to calculate the character address before the
  -- character is displayed
  CharColumn <= Hcount - HSYNC - HBACK_PORCH + 4 ;    --- hcount -140 ??
  Column <= CharColumn(9 downto 5);   ---/32
  CharRow <= Vcount - VSYNC - VBACK_PORCH ;           --- vcount - 35 ??
  Row <= CharRow(8 downto 5);     ----  /32

  -- Column + Row * 20
 CharAddr <= Column +
              ("0000" & Row(3 downto 0) & "0000") +
              ("000000" & Row(3 downto 0) & "00");

  CharRamPage <= CharAddr(9);
  CharRamSelect_N <=
    '0' when CharRamPage = '0' else     ------select charram0 (RAMB4_S8_S8_0)
    '1';     ------select none

  -- fontaddr(9) selects FontRam 
  -- fontaddr(8 downto 7) select picture 
  -- fontaddr(6 downto 0) data of bit for picture(128 x 8 bits)
  -- Most significant bit of character ignored such as DOUTBx(7 downto 4) ----
  FontAddr(9 downto 7) <=
    DOUTB0(2 downto 0);
    --(DOUTB0(6 downto 0) or DOUTB1(6 downto 0) or DOUTB2(6 downto 0)); ---those data included first address of picture selected by 2bits (1 downto 0)
                                                                        --- (1 downto 0) indicate picture selected and (3 downto 2) indicate selecting FontRam
 FontAddr(6 downto 2) <= CharRow(4 downto 0);
 FontAddr(1 downto 0) <= CharColumn(4 downto 3);          
 
  FontRamPage <= FontAddr(9);
  FontRamSelect_N <=
    "10" when FontRamPage = '0' else   ----select fontram0 (RAMB4_S8_S8_1)
    "01" when FontRamPage = '1' else   ----select fontram1 (RAMB4_S8_S8_2)
    "11";   ----none of fontrams has been selected

  FontData <= DOUTB1 or DOUTB2;
  
  -- Shift register

  ShiftRegister: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      ShiftData <= X"AB";
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if LoadNShift = '1' then
        ShiftData <= FontData;
      else
        ShiftData <= ShiftData(6 downto 0) & ShiftData(7);
      end if;
    end if;
  end process ShiftRegister;

  VideoData <= ShiftData(7);
     
                                
  -- Video signals going to the "video DAC"
  

  
  DelayFontAddr: process (Pixel_Clock, OPB_Rst)
     begin
      if OPB_Rst = '1' then
        FontAddrDelayed1 <= (others => '0');
        FontAddrDelayed <= (others => '0');
        FontAddrDelayed2 <= (others => '0');
      elsif Pixel_Clock'event and Pixel_Clock = '1' then
        FontAddrDelayed <= FontAddrDelayed2;
        FontAddrDelayed2 <= FontAddrDelayed1;
        FontAddrDelayed1 <= FontAddr;
      end if;
    end process DelayFontAddr;
 


   VideoOut: process (Pixel_Clock, OPB_Rst)
   begin
     if OPB_Rst = '1' then
       VIDOUT_RED   <= '0';
       VIDOUT_BLUE  <= '0';
       VIDOUT_GREEN <= '0';
     elsif Pixel_Clock'event and Pixel_Clock = '1' then
                if HBLANK_N = '1' and VBLANK_N = '1'  then  
  
                    if FontAddrDelayed(9 downto 7) = "000" then      -----backgroud
                      if VideoData = '1' then
                       
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;    
                     
                   elsif FontAddrDelayed(9 downto 7) = "001" then       -----wall
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '1';
                         VIDOUT_BLUE  <= '1';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '1';
                       end if;
                   
                   elsif FontAddrDelayed(9 downto 7) = "010" then        -----box
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '1';
                         VIDOUT_BLUE  <= '1';
                       else
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '1';
                         VIDOUT_BLUE  <= '0';
                       end if;
                    
                   elsif FontAddrDelayed(9 downto 7) = "011" then        ------flag(target)
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '1';
                         VIDOUT_BLUE  <= '0';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;
                     
                   elsif FontAddrDelayed(9 downto 7) = "100" then        ------pig 
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '1';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;
                     
                   elsif FontAddrDelayed(9 downto 7) = "101" then        ------pig (move right)
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '1';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;
                 
                   elsif FontAddrDelayed(9 downto 7) = "110" then         -----heart (move left)
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;
                     
                   elsif FontAddrDelayed(9 downto 7) = "111" then         -----pig (move right)
                       if VideoData = '1' then 
                         VIDOUT_RED   <= '1';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '1';
                       else
                         VIDOUT_RED   <= '0';
                         VIDOUT_GREEN <= '0';
                         VIDOUT_BLUE  <= '0';
                       end if;
                   end if;    
                else
                  VIDOUT_RED   <= '0'; 
                  VIDOUT_GREEN <= '0'; 
                  VIDOUT_BLUE  <= '0'; 
                  
                         
                end if;
                
          end if;
 end process VideoOut;
  
  
 --  VideoOut: process (Pixel_Clock, OPB_Rst)
 -- begin
 --   if OPB_Rst = '1' then
 --     VIDOUT_RED   <= '0';
 --     VIDOUT_BLUE  <= '0';
 --     VIDOUT_GREEN <= '0';
 --   elsif Pixel_Clock'event and Pixel_Clock = '1' then
 --     if VideoData = '1' and HBLANK_N = '1' and VBLANK_N = '1' then
 --       VIDOUT_RED   <= '1';
 --       VIDOUT_GREEN <= '0';
 --       VIDOUT_BLUE  <= '0';
 --    
 --    
 --    
 --    
 --       
 --     else
 --       
 --       VIDOUT_RED   <= '0';
 --       VIDOUT_GREEN <= '0';
 --       VIDOUT_BLUE  <= '0';
 --     end if;
 --   end if;
 -- end process VideoOut;


  VIDOUT_CLK <= Pixel_Clock;

end Behavioral;