-------------------------------------------------------------------------------
--
-- Text-mode VGA controller for the Digilent Spartan 3 starter board
-- with color output ability and reduced ram size
-- Uses an OPB interface, e.g., for use with the Microblaze soft core
--
-- Advisor: Stephen A. Edwards
--
-- Team:    Battle Snake
-- Student: Way-Cheng Sun
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity opb_s3board_vga is
  
  generic (
    C_OPB_AWIDTH : integer                   := 32;
    C_OPB_DWIDTH : integer                   := 32;
    C_BASEADDR   : std_logic_vector(31 downto 0) := X"FEFF0000";
    C_HIGHADDR   : std_logic_vector(31 downto 0) := X"FEFF1FFF");

  port (
    OPB_Clk        : in std_logic;
    OPB_Rst        : in std_logic;

    -- OPB signals
    OPB_ABus       : in std_logic_vector (31 downto 0);
    OPB_BE         : in std_logic_vector (3 downto 0);
    OPB_DBus       : in std_logic_vector (31 downto 0);
    OPB_RNW        : in std_logic;
    OPB_select     : in std_logic;
    OPB_seqAddr    : in std_logic;

    VGA_DBus       : out std_logic_vector (31 downto 0);
    VGA_errAck     : out std_logic;
    VGA_retry      : out std_logic;
    VGA_toutSup    : out std_logic;
    VGA_xferAck    : out std_logic;

    -- Video signals
    -- connection must be specified in the constraint file
    Pixel_Clock_2x : in std_logic;
    
    VIDOUT_CLK     : out std_logic;
    VIDOUT_RED     : out std_logic;
    VIDOUT_GREEN   : out std_logic;
    VIDOUT_BLUE    : out std_logic;
    VIDOUT_HSYNC : out std_logic;
    VIDOUT_VSYNC : out std_logic);

end opb_s3board_vga;

architecture Behavioral of opb_s3board_vga is
  
  constant BASEADDR : std_logic_vector(31 downto 0) := X"FEFF0000";

  -- Video parameters
  
  constant HTOTAL : integer := 800;
  constant HSYNC : integer := 96;
  constant HBACK_PORCH : integer := 48;
  constant HACTIVE : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL : integer := 525;
  constant VSYNC : integer := 2;
  constant VBACK_PORCH : integer := 33;
  constant VACTIVE : integer := 480;
  constant VFRONT_PORCH : integer := 10;

	component RAMB16_S9_S9
	  port (
           DOA      : out std_logic_vector( 7 downto 0); -- Port A 8-bit Data Output
           DOPA     : out std_logic_vector( 0 downto 0); -- Port A  1-bit Parity Output
           ADDRA    : in  std_logic_vector(10 downto 0); -- Port A 11-bit Address Input      
           DIA      : in  std_logic_vector( 7 downto 0); -- Port A  8-bit Data Input      
           DIPA     : in  std_logic_vector( 0 downto 0); -- Port A  1-bit parity Input
           CLKA     : in  std_logic;                     -- Port A Clock
           ENA      : in  std_logic;                     -- Port A RAM Enable Input      
           SSRA     : in  std_logic;                     -- Port A Synchronous Set/Reset Input
           WEA      : in  std_logic;                     -- Port A Write Enable Input

           DOB      : out std_logic_vector( 7 downto 0); -- Port B  8-bit Data Output
           DOPB     : out std_logic_vector( 0 downto 0); -- Port B  1-bit Parity Output
           ADDRB    : in  std_logic_vector(10 downto 0); -- Port B 11-bit Address Input      
           DIB      : in  std_logic_vector( 7 downto 0); -- Port B  8-bit Data Input
           DIPB     : in  std_logic_vector( 0 downto 0); -- Port B  1-bit parity Input      
           CLKB     : in  std_logic;                     -- Port B Clock                                   
           ENB      : in  std_logic;                     -- Port B RAM Enable Input      
           SSRB     : in  std_logic;                     -- Port B Synchronous Set/Reset Input      
           WEB      : in  std_logic                      -- Port B Write Enable Input
	  );
	end component;

	component RAMB16_S36_S36
	  port (
	    DOA      : out std_logic_vector(31 downto 0); -- Port A 32-bit Data Output
	    DOPA     : out std_logic_vector( 3 downto 0); -- Port A 4-bit Parity Output
            ADDRA    : in  std_logic_vector( 8 downto 0); -- Port A 9-bit Address Input      
            DIA      : in  std_logic_vector(31 downto 0); -- Port A 32-bit Data Input      
            DIPA     : in  std_logic_vector( 3 downto 0); -- Port A 4-bit parity Input
            CLKA     : in  std_logic;                     -- Port A Clock
            ENA      : in  std_logic;                     -- Port A RAM Enable Input      
            SSRA     : in  std_logic;                     -- Port A Synchronous Set/Reset Input
            WEA      : in  std_logic;                     -- Port A Write Enable Input

	    DOB      : out std_logic_vector(31 downto 0); -- Port B 32-bit Data Output
	    DOPB     : out std_logic_vector( 3 downto 0); -- Port B  4-bit Parity Output
	    ADDRB    : in  std_logic_vector( 8 downto 0); -- Port B  9-bit Address Input      
	    DIB      : in  std_logic_vector(31 downto 0); -- Port B 32-bit Data Input
	    DIPB     : in  std_logic_vector( 3 downto 0); -- Port B  4-bit parity Input      
	    CLKB     : in  std_logic;                     -- Port B Clock				    
            ENB      : in  std_logic;                     -- Port B RAM Enable Input      
	    SSRB     : in  std_logic;                     -- Port B Synchronous Set/Reset Input      
	    WEB      : in  std_logic                      -- Port B Write Enable Input
	  );
	end component;

  -- Signals latched from the OPB
  signal ABus : std_logic_vector (31 downto 0);
  signal DBus : std_logic_vector (31 downto 0);
  signal RNW  : std_logic;
  signal select_delayed : std_logic;

  -- Latched output signals for the OPB
  signal DBus_out : std_logic_vector (31 downto 0);

--------------------------------------------------------------------------------------------------

  -- Signals for the OPB-mapped RAM
  signal ChipSelect : std_logic;            -- Address decode
  signal MemCycle1, MemCycle2 : std_logic;  -- State bits
  signal RamPageAddress : std_logic_vector(1 downto 0); 
  signal RamSelect : std_logic_vector (2 downto 0); -- latched signal for RST, WE 
  
  --BRAM control signals
  signal RST, WE : std_logic_vector (2 downto 0);
  
  signal DOUT0, DOUT1, DOUT2 : std_logic_vector(7 downto 0);     -- Char Ram Font port output
  signal DOUTB0, DOUTB1, DOUTB2 : std_logic_vector(7 downto 0);  -- Char Ram OPB port output
  signal FOUT, SOUT : std_logic_vector(31 downto 0);             -- font ram and sprite ram output
  
  signal ReadData : std_logic_vector(7 downto 0); -- latched signal to DBus_out

----------------------------------------------------------------------------------------------------
  -- Signals for the video controller
  signal Pixel_Clock : std_logic;       -- 25 MHz clock divided from 50 MHz
  signal FontData : std_logic_vector(23 downto 0);   -- Input to shift register
  signal LoadNShift : std_logic;        -- Shift register control
  signal ShiftData_R, ShiftData_G, ShiftData_B : std_logic_vector(7 downto 0);  -- Shift register data

  signal Hcount : std_logic_vector(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : std_logic_vector(9 downto 0);  -- Vertical position (0-524)
  signal HBLANK_N, VBLANK_N : std_logic;   -- Blanking signals
 
  signal FontAddr        : std_logic_vector(8 downto 0);
  signal CharRamPage     : std_logic_vector(1 downto 0);
  signal CharRamSelect_N : std_logic_vector(2 downto 0);
  signal CharAddr        : std_logic_vector(12 downto 0);
   
  signal Horizontal      : std_logic_vector(9 downto 0);  -- 4-653
  signal Vertical        : std_logic_vector(9 downto 0);  -- 0-479
  signal Column          : std_logic_vector(6 downto 0);  -- 0-79
  signal Row             : std_logic_vector(6 downto 0);  -- 0-59
  
  signal EndOfLine, EndOfField : std_logic;
  
-----------------------------------------------------------------------------------------------------
  -- Signals for the sprite generator
  
 -- signal SpriteData : std_logic_vector(23 downto 0); -- Input to sprite shift register
 -- signal SpriteNShift : std_logic;                   -- Sprite shift register control
 -- signal SpriteData_R, SpriteData_G, SpriteData_B : std_logic_vector(7 downto 0);  -- Sprite shifter data
  
  signal SpriteAddr : std_logic_vector(8 downto 0); -- BRAM sprite port address
 -- signal VideoSelect : std_logic;                   -- Sprite/tile output select
 
  
  -------------------------------------------------------------------------------
  --
  -- Initialize Font RAM 
  --
  -------------------------------------------------------------------------------

  attribute INIT_00 : string;
  attribute INIT_01 : string;
  attribute INIT_02 : string;
  attribute INIT_03 : string;
  attribute INIT_04 : string;
  attribute INIT_05 : string;
  attribute INIT_06 : string;
  attribute INIT_07 : string;
  attribute INIT_08 : string;
  attribute INIT_09 : string;
  attribute INIT_0a : string;
  attribute INIT_0b : string;
  attribute INIT_0c : string;
  attribute INIT_0d : string;
  attribute INIT_0e : string;
  attribute INIT_0f : string;
  attribute INIT_10 : string;
  attribute INIT_11 : string;
  attribute INIT_12 : string;
  attribute INIT_13 : string;
  attribute INIT_14 : string;
  attribute INIT_15 : string;
  attribute INIT_16 : string;
  attribute INIT_17 : string;
  attribute INIT_18 : string;
  attribute INIT_19 : string;
  attribute INIT_1a : string;
  attribute INIT_1b : string;
  attribute INIT_1c : string;
  attribute INIT_1d : string;
  attribute INIT_1e : string;
  attribute INIT_1f : string;
  attribute INIT_20 : string;
  attribute INIT_21 : string;
  attribute INIT_22 : string;
  attribute INIT_23 : string;
  attribute INIT_24 : string;
  attribute INIT_25 : string;
  attribute INIT_26 : string;
  attribute INIT_27 : string;
  attribute INIT_28 : string;
  attribute INIT_29 : string;
  attribute INIT_2a : string;
  attribute INIT_2b : string;
  attribute INIT_2c : string;
  attribute INIT_2d : string;
  attribute INIT_2e : string;
  attribute INIT_2f : string;
  attribute INIT_30 : string;
  attribute INIT_31 : string;
  attribute INIT_32 : string;
  attribute INIT_33 : string;
  attribute INIT_34 : string;
  attribute INIT_35 : string;
  attribute INIT_36 : string;
  attribute INIT_37 : string;
  attribute INIT_38 : string;
  attribute INIT_39 : string;
  attribute INIT_3a : string;
  attribute INIT_3b : string;
  attribute INIT_3c : string;
  attribute INIT_3d : string;
  attribute INIT_3e : string;
  attribute INIT_3f : string;
  
  attribute INIT_00 of RAMB16_S36_S36_1 : label is
   "0000007e003c3cff006666ff006666ff006666ff006666ff003c3cff0000007e";
  attribute INIT_01 of RAMB16_S36_S36_1 : label is
   "0000007e003c3c7e0018187e0018183c0018183c0018187c0038387c0000007c";
  attribute INIT_02 of RAMB16_S36_S36_1 : label is
   "000000ff007e7eff006060ff006060fc003c3c7e000606ff007c7cff0000007e";
  attribute INIT_03 of RAMB16_S36_S36_1 : label is
   "000000fe007c7cfe000606ff0006067f003c3c7e000606ff007c7cff000000fe";
  attribute INIT_04 of RAMB16_S36_S36_1 : label is
   "0000000e000404ff007e7eff006464ff006464fe006464fe006464fe000000fe";
  attribute INIT_05 of RAMB16_S36_S36_1 : label is
   "000000fe007c7cff000e0e7f000e0e7f007c7cff006060fe007c7cfe000000fe";
  attribute INIT_06 of RAMB16_S36_S36_1 : label is
   "0000007e003c3cff006666ff006666ff007c7cff006060fe003c3cfe0000007e";
  attribute INIT_07 of RAMB16_S36_S36_1 : label is
   "000000780030307c0018183e000c0c1f0006060f000606ff007e7eff000000ff";
  attribute INIT_08 of RAMB16_S36_S36_1 : label is
   "0000007e003c3cff006666ff006666ff003c3cff006666ff003c3cff0000007e";
  attribute INIT_09 of RAMB16_S36_S36_1 : label is
   "0000007e003c3c7f0006067f003e3eff006666ff006666ff003c3cff0000007e";
  attribute INIT_0a of RAMB16_S36_S36_1 : label is
   "000000ff006666ff006666ff007e7eff006666ff006666ff003c3cff0000007e";
  attribute INIT_0b of RAMB16_S36_S36_1 : label is
   "000000fe007c7cff006666ff006666ff007c7cff006666ff007c7cfe000000fc";
  attribute INIT_0c of RAMB16_S36_S36_1 : label is
   "0000003e003c3c7f006666ff006060ff006060ff006666ff003c3cff0000007e";
  attribute INIT_0d of RAMB16_S36_S36_1 : label is
   "000000fe007c7cff006666ff006666ff006666ff006c6cff007878fe000000fc";
  attribute INIT_0e of RAMB16_S36_S36_1 : label is
   "000000ff007e7eff006060ff006060ff007c7cfe006060ff007e7eff000000ff";
  attribute INIT_0f of RAMB16_S36_S36_1 : label is
   "000000f0006060f0006060f0006060fe007c7cfe006060ff007e7eff000000ff";
  attribute INIT_10 of RAMB16_S36_S36_1 : label is
   "0000003e003c3c7f006666ff006e6eff006060ff006666ff003c3cff0000007e";
  attribute INIT_11 of RAMB16_S36_S36_1 : label is
   "000000ff006666ff006666ff006666ff007e7eff006666ff006666ff000000ff";
  attribute INIT_12 of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0018183c0018183c0018183c0018183c0018183c0000003c";
  attribute INIT_13 of RAMB16_S36_S36_1 : label is
   "0000003c003c3c7e006666ff006666ff000606ff0006060f0006060f0000000f";
  attribute INIT_14 of RAMB16_S36_S36_1 : label is
   "000000f7006666ff006c6cfe007878fc007878fc006c6cfe006666ff000000f7";
  attribute INIT_15 of RAMB16_S36_S36_1 : label is
   "000000ff007e7eff006060ff006060f0006060f0006060f0006060f0000000f0";
  attribute INIT_16 of RAMB16_S36_S36_1 : label is
   "000000ff005656ff005656ff005656ff005656ff005656ff007c7cfe000000fc";
  attribute INIT_17 of RAMB16_S36_S36_1 : label is
   "000000ff006666ff006666ff006666ff006666ff006666ff007c7cfe000000fc";
  attribute INIT_18 of RAMB16_S36_S36_1 : label is
   "0000003c003c3c7e006666ff006666ff006666ff006666ff003c3c7e0000003c";
  attribute INIT_19 of RAMB16_S36_S36_1 : label is
   "000000f0006060f0006060fc007c7cfe006666ff006666ff007c7cfe000000fc";
  attribute INIT_1a of RAMB16_S36_S36_1 : label is
   "0000003b003a3a7f006c6cfe006666ff006666ff006666ff003c3c7e0000003c";
  attribute INIT_1b of RAMB16_S36_S36_1 : label is
   "000000ff006666ff006464ff007c7cfe006666ff006666ff007c7cfe000000fc";
  attribute INIT_1c of RAMB16_S36_S36_1 : label is
   "000000fe007c7cff007e7eff000606ff007c7cfe006060ff003e3e7f0000003f";
  attribute INIT_1d of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0018183c0018183c0018183c001818ff007e7eff000000ff";
  attribute INIT_1e of RAMB16_S36_S36_1 : label is
   "0000007e003c3cff006666ff006666ff006666ff006666ff006666ff000000ff";
  attribute INIT_1f of RAMB16_S36_S36_1 : label is
   "000000180018183c003c3cfe002424ff006666ff006666ff006666ff000000ff";
  attribute INIT_20 of RAMB16_S36_S36_1 : label is
   "000000fc007c7cfe005656ff005656ff005656ff005656ff005656ff000000ff";
  attribute INIT_21 of RAMB16_S36_S36_1 : label is
   "000000e3004242e7006464fe0030307c0018183c002c2c7e004646ef000000c7";
  attribute INIT_22 of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0018183c001818ff007e7eff006666ff006666ff000000ff";
  attribute INIT_23 of RAMB16_S36_S36_1 : label is
   "000000ff007e7eff006060ff0030307c0018183e000c0cff007e7eff000000ff";
  attribute INIT_24 of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0000003c0018183c0018183c0018183c0018183c0000003c";
  attribute INIT_25 of RAMB16_S36_S36_1 : label is
   "000000380010103c0008083c0018183c0000003c000000000000000000000000";
  attribute INIT_26 of RAMB16_S36_S36_1 : label is
   "0000000000000000000000fe007c7cfe000000fe000000000000000000000000";
  attribute INIT_27 of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0018183c0000003c00000000000000000000000000000000";
  attribute INIT_28 of RAMB16_S36_S36_1 : label is
   "000000e0004040e0006060f0003030780018183c000c0c1e0006060f00000007";
  attribute INIT_29 of RAMB16_S36_S36_1 : label is
   "0000003c0018183c0018183c0000003c0018183c0018183c0000003c00000000";
  attribute INIT_2a of RAMB16_S36_S36_1 : label is
   "000000380010103c0018183c0000003c0018183c0018183c0000003c00000000";
  attribute INIT_2b of RAMB16_S36_S36_1 : label is
   "0000007800303078000000780030307c001c1c7e000606ff007c7cff0000007e";
  attribute INIT_2c of RAMB16_S36_S36_1 : label is
   "00000000007e7e7e007e7e7e007e7e7e007e7e7e007e7e7e007e7e7e00000000";
  attribute INIT_2d of RAMB16_S36_S36_1 : label is
   "0000000000000000003c003c003c003c003c003c003c003c0000000000000000";
  attribute INIT_2e of RAMB16_S36_S36_1 : label is
   "000000000000000000003c0000003c0000003c0000003c000000000000000000";
  attribute INIT_2f of RAMB16_S36_S36_1 : label is
   "00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff00ff";
  attribute INIT_30 of RAMB16_S36_S36_1 : label is
   "0000ff000000ff000000ff000000ff000000ff000000ff000000ff000000ff00";
  attribute INIT_31 of RAMB16_S36_S36_1 : label is
   "00000000007e007e007e007e007e007e007e007e007e007e007e007e00000000";
  attribute INIT_32 of RAMB16_S36_S36_1 : label is
   "0000000000007e0000007e0000007e0000007e0000007e0000007e0000000000";
  attribute INIT_33 of RAMB16_S36_S36_1 : label is
   "0000000000000000000000000000000000000000000000000000000000000000";
  attribute INIT_34 of RAMB16_S36_S36_1 : label is
   "00000000007e7e00007e7e00007e7e00007e7e00007e7e00007e7e0000000000";
  attribute INIT_35 of RAMB16_S36_S36_1 : label is
   "000000000000007e0000007e0000007e0000007e0000007e0000007e00000000";
  attribute INIT_36 of RAMB16_S36_S36_1 : label is
   "00ffffff0081818100bdbdbd00a5a5a500b5b5b50085858500fdfdfd00000000";
  attribute INIT_37 of RAMB16_S36_S36_1 : label is
   "00000000007e7e7e00424242005a5a5a004a4a4a007a7a7a0002020200000000";
  attribute INIT_38 of RAMB16_S36_S36_1 : label is
   "00ff00000081000000bd000000a5000000b500000085000000fd000000000000";
  attribute INIT_39 of RAMB16_S36_S36_1 : label is
   "0000ffff000081810000bdbd0000a5a50000b5b5000085850000fdfd00000000";
  attribute INIT_3a of RAMB16_S36_S36_1 : label is
   "0000ff00000081000000bd000000a5000000b500000085000000fd0000000000";
  attribute INIT_3b of RAMB16_S36_S36_1 : label is
   "0000000000000000007f0000007f0000007f0000007f0000007f000000000000";
  attribute INIT_3c of RAMB16_S36_S36_1 : label is
   "0000000000000000007f7f00007f7f00007f7f00007f7f00007f7f0000000000";
  attribute INIT_3d of RAMB16_S36_S36_1 : label is
   "0000000000000000001f1f00003f3f00003f3f00003f3f00001f1f0000000000";
  attribute INIT_3e of RAMB16_S36_S36_1 : label is
   "0000000000000000007c7c00007e7e00007e7e00007e7e00007c7c0000000000";
  attribute INIT_3f of RAMB16_S36_S36_1 : label is
   "0000000000000000007c0000007e0000007e0000007e0000007c000000000000";
   
-------------------------------------------------------------------------------   

begin  -- Behavioral

    RAMB16_S9_S9_1 : RAMB16_S9_S9
    port map (
    
    -- port A: R/W from OPB
      DOA    => DOUT0,             -- Port A  8-bit Data Output
      --DOPA   => '0',             -- Port A  1-bit Parity Output
      ADDRA  => ABus(10 downto 0), -- Port A 11-bit Address Input      		
      DIA    => DBus(7 downto 0),  -- Port A  8-bit Data Input      
      DIPA   => (others => '0'),   -- Port A  1-bit parity Input
      CLKA   => OPB_Clk,           -- Port A Clock
      ENA    => '1',               -- Port A RAM Enable Input      
      SSRA   => RST(0),            -- Port A Synchronous Set/Reset Input
      WEA    => WE(0),             -- Port A Write Enable Input
      
    -- port B: R from controller   
      DOB    => DOUTB0,                -- Port B  8-bit Data Output
      --DOPB   => '0',                 -- Port B  1-bit Parity Output
      ADDRB  => CharAddr(10 downto 0), -- Port B 11-bit Address Input      
      DIB    => X"00",                 -- Port B  8-bit Data Input      
      DIPB   => (others => '0'),       -- Port B  1-bit parity Input
      CLKB   => Pixel_Clock,           -- Port B Clock
      ENB    => '1',                   -- Port B RAM Enable Input      
      SSRB   => CharRamSelect_N(0),    -- Port B Synchronous Set/Reset Input      
      WEB    => '0'                    -- Port B Write Enable Input
    );


    RAMB16_S9_S9_2 : RAMB16_S9_S9
    port map (
    
    -- port A: R/W from OPB
      DOA    => DOUT1,             -- Port A  8-bit Data Output
      --DOPA   => '0',             -- Port A  1-bit Parity Output
      ADDRA  => ABus(10 downto 0), -- Port A 11-bit Address Input      		
      DIA    => DBus(7 downto 0),  -- Port A  8-bit Data Input      
      DIPA   => (others => '0'),   -- Port A  1-bit parity Input
      CLKA   => OPB_Clk,           -- Port A Clock
      ENA    => '1',               -- Port A RAM Enable Input      
      SSRA   => RST(1),            -- Port A Synchronous Set/Reset Input
      WEA    => WE(1),             -- Port A Write Enable Input
      
    -- port B: R from controller  
      DOB    => DOUTB1,                -- Port B  8-bit Data Output
      --DOPB   => '0',                 -- Port B  1-bit Parity Output
      ADDRB  => CharAddr(10 downto 0), -- Port B 11-bit Address Input      
      DIB    => X"00",                 -- Port B  8-bit Data Input      
      DIPB   => (others => '0'),       -- Port B  1-bit parity Input
      CLKB   => Pixel_Clock,           -- Port B Clock
      ENB    => '1',                   -- Port B RAM Enable Input      
      SSRB   => CharRamSelect_N(1),    -- Port B Synchronous Set/Reset Input      
      WEB    => '0'                    -- Port B Write Enable Input
    );
    
    RAMB16_S9_S9_3 : RAMB16_S9_S9
    port map (
    
    -- port A: R/W from OPB
      DOA    => DOUT2,              -- Port A  8-bit Data Output
      --DOPA   => '0',              -- Port A  1-bit Parity Output
      ADDRA  => ABus(10 downto 0),  -- Port A 11-bit Address Input      		
      DIA    => DBus(7 downto 0),   -- Port A  8-bit Data Input      
      DIPA   => (others => '0'),    -- Port A  1-bit parity Input
      CLKA   => OPB_Clk,            -- Port A Clock
      ENA    => '1',                -- Port A RAM Enable Input      
      SSRA   => RST(2),             -- Port A Synchronous Set/Reset Input
      WEA    => WE(2),              -- Port A Write Enable Input
      
    -- port B: R from controller  
      DOB    => DOUTB2,                -- Port B  8-bit Data Output
      --DOPB   => '0',                 -- Port B  1-bit Parity Output
      ADDRB  => CharAddr(10 downto 0), -- Port B 11-bit Address Input      
      DIB    => X"00",                 -- Port B  8-bit Data Input      
      DIPB   => (others => '0'),       -- Port B  1-bit parity Input
      CLKB   => Pixel_Clock,           -- Port B Clock
      ENB    => '1',                   -- Port B RAM Enable Input      
      SSRB   => CharRamSelect_N(2),    -- Port B Synchronous Set/Reset Input      
      WEB    => '0'                    -- Port B Write Enable Input
    );
    
    RAMB16_S36_S36_1 : RAMB16_S36_S36
    port map (
    
    -- port A: R for tile
      DOA    => FOUT,                 -- Port A 32-bit Data Output
      --DOPA   => X"00",              -- Port A  4-bit Parity Output
      ADDRA  => FontAddr(8 downto 0), -- Port A  9-bit Address Input      		
      DIA    => (others => '0'),      -- Port A 32-bit Data Input      
      DIPA   => (others => '0'),      -- Port A  4-bit parity Input
      CLKA   => Pixel_Clock,          -- Port A Clock
      ENA    => '1',                  -- Port A RAM Enable Input      
      SSRA   => '0',                  -- Port A Synchronous Set/Reset Input
      WEA    => '0',                  -- Port A Write Enable Input
      
    -- port B: R for sprite(optional)  
      DOB    => SOUT,                    -- Port B 32-bit Data Output
      --DOPB   => (others => '0'),       -- Port B  4-bit Parity Output
      ADDRB  => SpriteAddr(8 downto 0),  -- Port B  9-bit Address Input      
      DIB    => (others => '0'),         -- Port B 32-bit Data Input      
      DIPB   => (others => '0'),         -- Port B  4-bit parity Input
      CLKB   => Pixel_Clock,             -- Port B Clock
      ENB    => '1',                     -- Port B RAM Enable Input      
      SSRB   => '0',                     -- Port B Synchronous Set/Reset Input      
      WEB    => '0'                      -- Port B Write Enable Input
    );
    
  -----------------------------------------------------------------------------
  --  
  -- OPB-RAM controller
  --
  -----------------------------------------------------------------------------
  
  -- Unused OPB control signals
  VGA_errAck  <= '0';
  VGA_retry   <= '0';
  VGA_toutSup <= '0';

  -- Latch the relevant OPB signals from the OPB, since they arrive late
  LatchOPB: process (OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Abus <= ( others => '0' );
      DBus <= ( others => '0' );
      RNW  <= '1';
      select_delayed <= '0';
    elsif OPB_Clk'event and OPB_Clk = '1' then
      ABus <= OPB_ABus;
      DBus <= OPB_DBus;
      RNW  <= OPB_RNW;
      select_delayed <= OPB_Select;
    end if;
  end process LatchOPB;
  
  -- Address bits 31 downto 13 is our chip select
  -- 12 downto 11 is the RAM page select
  -- 10 downto  0 is the RAM byte select
  
  ChipSelect <=
    '1' when select_delayed = '1' and  
	    (ABus(31 downto 13) = BASEADDR(31 downto 13)) and
       MemCycle1 = '0' and MemCycle2 = '0' else
    '0';

  RamPageAddress <= ABus(12 downto 11);
  
  RamSelect <=
    "001" when RamPageAddress = "00" else
    "010" when RamPageAddress = "01" else
    "100" when RamPageAddress = "10" else
    "001" when RamPageAddress = "11" else
    "000";

  MemCycleFSM : process(OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      MemCycle1 <= '0';
      MemCycle2 <= '0';
    elsif OPB_Clk'event and OPB_Clk = '1' then
      MemCycle2 <= MemCycle1;
      MemCycle1 <= ChipSelect;
    end if;
  end process MemCycleFSM;
  
  VGA_xferAck <= MemCycle2;

  WE <=
  RamSelect when ChipSelect = '1' and RNW = '0' and OPB_Rst = '0' else
    "000";
 
  RST <=
   not RamSelect when ChipSelect = '1' and RNW = '1' and OPB_Rst = '0' else
    "111"; 

  ReadData <= DOUT0 or DOUT1 or DOUT2 when MemCycle1 = '1'
              else X"00";

  -- DBus(31 downto 24) is the byte for addresses ending in 0
  -- latch memory output to OPB
  GenDOut: process (OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      DBus_out <= ( others => '0');
    elsif OPB_Clk'event and OPB_Clk = '1' then
      DBus_out <= ReadData & ReadData & ReadData & ReadData;
    end if;
  end process GenDOut;

  VGA_DBus <= DBus_out;

  ----------------------------------------------------------------------------
  --
  -- Video controller
  --
  -----------------------------------------------------------------------------

  -- Pixel clock divider
  
  Pixel_clk_divider : process (Pixel_Clock_2x, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Pixel_Clock <= '0';
    elsif Pixel_Clock_2x'event and Pixel_Clock_2x = '1' then
      Pixel_Clock <= not Pixel_Clock;
    end if;
  end process Pixel_clk_divider;
  
  -- Horizontal and vertical counters

  HCounter : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      Hcount <= (others => '0');
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
 
  VCounter: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      Vcount <= (others => '0');
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VIDOUT_HSYNC <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine = '1' then
        VIDOUT_HSYNC <= '1';
      elsif Hcount = HSYNC - 1 then
        VIDOUT_HSYNC <= '0';
      end if;      
    end if;
  end process HSyncGen;

  HBlankGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      HBLANK_N <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if Hcount = HSYNC + HBACK_PORCH - 1 then
        HBLANK_N <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE - 1 then
        HBLANK_N <= '0';
      end if;      
    end if;
  end process HBlankGen;
  
  VSyncGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VIDOUT_VSYNC <= '1';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine ='1' then
        if EndOfField = '1' then
          VIDOUT_VSYNC <= '1';
        elsif VCount = VSYNC - 1 then
          VIDOUT_VSYNC <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (Pixel_Clock, OPB_Rst)
  begin    
    if OPB_Rst = '1' then
      VBLANK_N <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if EndOfLine ='1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          VBLANK_N <= '1';
        elsif VCount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          VBLANK_N <= '0';
        end if;
      end if;      
    end if;
  end process VBlankGen;
  
  

  ------------------------------
  --
  --  Read font RAM 
  --
  ------------------------------

  -- Correction of 4 needed to calculate the character address before the
  -- character is displayed
  Horizontal <= Hcount - HSYNC - HBACK_PORCH + 4; -- H pixel count
  Vertical <= Vcount - VSYNC - VBACK_PORCH;       -- V pixel count
  Column <= Horizontal(9 downto 3);  -- /8        -- column count
  Row <= Vertical(9 downto 3);       -- /8        -- row count

  -- Column + Row * 80
  CharAddr <= Column + ( Row(6 downto 0) & "000000")
                     + ("00" & Row(6 downto 0) & "0000");

  CharRamPage <= CharAddr(12 downto 11);
  CharRamSelect_N <=
    "110" when CharRamPage = "00" else
    "101" when CharRamPage = "01" else
    "011" when CharRamPage = "10" else
    "110" when CharRamPage = "11" else
    "111";
    
  -- generate font address for accessing
  -- Most significant bit of character ignored
  FontAddr(8 downto 3) <= (DOUTB0(5 downto 0) or DOUTB1(5 downto 0) or DOUTB2(5 downto 0));
  FontAddr(2 downto 0) <= Vertical(2 downto 0); -- see which font row is there

  FontData <= FOUT(23 downto 0);
  
  
  
  ---------------------------------
  --
  -- Tile Output to VGA
  --
  ---------------------------------

  -- Shift registers for RGB
  
  LoadNShift <= '1' when Hcount(2 downto 0) = X"7" else '0';

  ShiftRegister_R: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      ShiftData_R <= X"00";
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if LoadNShift = '1' then
        ShiftData_R <= FontData(23 downto 16);
      else
        ShiftData_R <= ShiftData_R(6 downto 0) & ShiftData_R(7);
      end if;
    end if;
  end process ShiftRegister_R;

  ShiftRegister_G: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      ShiftData_G <= X"00";
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if LoadNShift = '1' then
        ShiftData_G <= FontData(15 downto 8);
      else
        ShiftData_G <= ShiftData_G(6 downto 0) & ShiftData_G(7);
      end if;
    end if;
  end process ShiftRegister_G;

  ShiftRegister_B: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      ShiftData_B <= X"00";
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if LoadNShift = '1' then
        ShiftData_B <= FontData(7 downto 0);
      else
        ShiftData_B <= ShiftData_B(6 downto 0) & ShiftData_B(7);
      end if;
    end if;
  end process ShiftRegister_B;
  
  ----------------------------
  --
  --
  --
  ----------------------------

  ----------------------------
  --
  -- Video signals going out
  --
  ----------------------------
  
  VideoOut: process (Pixel_Clock, OPB_Rst)
  begin
    if OPB_Rst = '1' then
      VIDOUT_RED   <= '0';
      VIDOUT_BLUE  <= '0';
      VIDOUT_GREEN <= '0';
    elsif Pixel_Clock'event and Pixel_Clock = '1' then
      if HBLANK_N = '1' and VBLANK_N = '1' then
      
        VIDOUT_RED   <= ShiftData_R(7);
        VIDOUT_GREEN <= ShiftData_G(7);
        VIDOUT_BLUE  <= ShiftData_B(7);
        
      else
        VIDOUT_RED   <= '0';
        VIDOUT_GREEN <= '0';
        VIDOUT_BLUE  <= '0';
      end if;
    end if;
  end process VideoOut;

  VIDOUT_CLK <= Pixel_Clock;

end Behavioral;
