/***************************************************************************
 *  Copyright (C) 2004 Michael E. Locasto
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the:
 *       Free Software Foundation, Inc.
 *       59 Temple Place, Suite 330 
 *       Boston, MA  02111-1307  USA
 *
 * $Id: Keywords.java,v 1.2 2004/10/22 06:05:59 locasto Exp $
 **************************************************************************/
package org.postgresql;

import java.util.HashMap;
import java.util.Collection;
import java.util.ArrayList;

/**
 * The <code>Keywords</code> class provides a lookup facility for the
 * Postgresql SQL commands and keywords.
 *
 * @author mlocasto@acm.org
 */
public final class Keywords
{
   /**
    * An O(1) lookup store of keywords.
    */
   private static HashMap ms_keywords = new HashMap();
   
   /**
    * A searchable list of keywords in alpha order.
    */
   private static ArrayList ms_keys = null;

   static
   {
      ms_keywords.put("abort", "abort");
      ms_keywords.put("absolute", "absolute");
      ms_keywords.put("access", "access");
      ms_keywords.put("action", "action");
      ms_keywords.put("add", "add");
      ms_keywords.put("after", "after");
      ms_keywords.put("aggregate", "aggregate");
      ms_keywords.put("all", "all");
      ms_keywords.put("alter", "alter");
      ms_keywords.put("analyse", "analyse");		/* British spelling */
      ms_keywords.put("analyze", "analyze");
      ms_keywords.put("and", "and");
      ms_keywords.put("any", "any");
      ms_keywords.put("array", "array");
      ms_keywords.put("as", "as");
      ms_keywords.put("asc", "asc");
      ms_keywords.put("assertion", "assertion");
      ms_keywords.put("assignment", "assignment");
      ms_keywords.put("at", "at");
      ms_keywords.put("authorization", "authorization");
      ms_keywords.put("backward", "backward");
      ms_keywords.put("before", "before");
      ms_keywords.put("begin", "begin");
      ms_keywords.put("between", "between");
      ms_keywords.put("bigint", "bigint");
      ms_keywords.put("binary", "binary");
      ms_keywords.put("bit", "bit");
      ms_keywords.put("boolean", "boolean");
      ms_keywords.put("both", "both");
      ms_keywords.put("by", "by");
      ms_keywords.put("cache", "cache");
      ms_keywords.put("called", "called");
      ms_keywords.put("cascade", "cascade");
      ms_keywords.put("case", "case");
      ms_keywords.put("cast", "cast");
      ms_keywords.put("chain", "chain");
      ms_keywords.put("char", "char");
      ms_keywords.put("character", "character");
      ms_keywords.put("characteristics", "characteristics");
      ms_keywords.put("check", "check");
      ms_keywords.put("checkpoint", "checkpoint");
      ms_keywords.put("class", "class");
      ms_keywords.put("close", "close");
      ms_keywords.put("cluster", "cluster");
      ms_keywords.put("coalesce", "coalesce");
      ms_keywords.put("collate", "collate");
      ms_keywords.put("column", "column");
      ms_keywords.put("comment", "comment");
      ms_keywords.put("commit", "commit");
      ms_keywords.put("committed", "committed");
      ms_keywords.put("constraint", "constraint");
      ms_keywords.put("constraints", "constraints");
      ms_keywords.put("conversion", "conversion");
      ms_keywords.put("convert", "convert");
      ms_keywords.put("copy", "copy");
      ms_keywords.put("create", "create");
      ms_keywords.put("createdb", "createdb");
      ms_keywords.put("createuser", "createuser");
      ms_keywords.put("cross", "cross");
      ms_keywords.put("current_date", "current_date");
      ms_keywords.put("current_time", "current_time");
      ms_keywords.put("current_timestamp", "current_timestamp");
      ms_keywords.put("current_user", "current_user");
      ms_keywords.put("cursor", "cursor");
      ms_keywords.put("cycle", "cycle");
      ms_keywords.put("database", "database");
      ms_keywords.put("day", "day");
      ms_keywords.put("deallocate", "deallocate");
      ms_keywords.put("dec", "dec");
      ms_keywords.put("decimal", "decimal");
      ms_keywords.put("declare", "declare");
      ms_keywords.put("default", "default");
      ms_keywords.put("defaults", "defaults");
      ms_keywords.put("deferrable", "deferrable");
      ms_keywords.put("deferred", "deferred");
      ms_keywords.put("definer", "definer");
      ms_keywords.put("delete", "delete");
      ms_keywords.put("delimiter", "delimiter");
      ms_keywords.put("delimiters", "delimiters");
      ms_keywords.put("desc", "desc");
      ms_keywords.put("distinct", "distinct");
      ms_keywords.put("do", "do");
      ms_keywords.put("domain", "domain");
      ms_keywords.put("double", "double");
      ms_keywords.put("drop", "drop");
      ms_keywords.put("each", "each");
      ms_keywords.put("else", "else");
      ms_keywords.put("encoding", "encoding");
      ms_keywords.put("encrypted", "encrypted");
      ms_keywords.put("end", "end");
      ms_keywords.put("escape", "escape");
      ms_keywords.put("except", "except");
      ms_keywords.put("excluding", "excluding");
      ms_keywords.put("exclusive", "exclusive");
      ms_keywords.put("execute", "execute");
      ms_keywords.put("exists", "exists");
      ms_keywords.put("explain", "explain");
      ms_keywords.put("external", "external");
      ms_keywords.put("extract", "extract");
      ms_keywords.put("false", "false");
      ms_keywords.put("fetch", "fetch");
      ms_keywords.put("first", "first");
      ms_keywords.put("float", "float");
      ms_keywords.put("for", "for");
      ms_keywords.put("force", "force");
      ms_keywords.put("foreign", "foreign");
      ms_keywords.put("forward", "forward");
      ms_keywords.put("freeze", "freeze");
      ms_keywords.put("from", "from");
      ms_keywords.put("full", "full");
      ms_keywords.put("function", "function");
      ms_keywords.put("global", "global");
      ms_keywords.put("grant", "grant");
      ms_keywords.put("group", "group");
      ms_keywords.put("handler", "handler");
      ms_keywords.put("having", "having");
      ms_keywords.put("hold", "hold");
      ms_keywords.put("hour", "hour");
      ms_keywords.put("ilike", "ilike");
      ms_keywords.put("immediate", "immediate");
      ms_keywords.put("immutable", "immutable");
      ms_keywords.put("implicit", "implicit");
      ms_keywords.put("in", "in");
      ms_keywords.put("including", "including");
      ms_keywords.put("increment", "increment");
      ms_keywords.put("index", "index");
      ms_keywords.put("inherits", "inherits");
      ms_keywords.put("initially", "initially");
      ms_keywords.put("inner", "inner");
      ms_keywords.put("inout", "inout");
      ms_keywords.put("input", "input");
      ms_keywords.put("insensitive", "insensitive");
      ms_keywords.put("insert", "insert");
      ms_keywords.put("instead", "instead");
      ms_keywords.put("int", "int");
      ms_keywords.put("integer", "integer");
      ms_keywords.put("intersect", "intersect");
      ms_keywords.put("interval", "interval");
      ms_keywords.put("into", "into");
      ms_keywords.put("invoker", "invoker");
      ms_keywords.put("is", "is");
      ms_keywords.put("isnull", "isnull");
      ms_keywords.put("isolation", "isolation");
      ms_keywords.put("join", "join");
      ms_keywords.put("key", "key");
      ms_keywords.put("lancompiler", "lancompiler");
      ms_keywords.put("language", "language");
      ms_keywords.put("last", "last");
      ms_keywords.put("leading", "leading");
      ms_keywords.put("left", "left");
      ms_keywords.put("level", "level");
      ms_keywords.put("like", "like");
      ms_keywords.put("limit", "limit");
      ms_keywords.put("listen", "listen");
      ms_keywords.put("load", "load");
      ms_keywords.put("local", "local");
      ms_keywords.put("localtime", "localtime");
      ms_keywords.put("localtimestamp", "localtimestamp");
      ms_keywords.put("location", "location");
      ms_keywords.put("lock", "lock");
      ms_keywords.put("match", "match");
      ms_keywords.put("maxvalue", "maxvalue");
      ms_keywords.put("minute", "minute");
      ms_keywords.put("minvalue", "minvalue");
      ms_keywords.put("mode", "mode");
      ms_keywords.put("month", "month");
      ms_keywords.put("move", "move");
      ms_keywords.put("names", "names");
      ms_keywords.put("national", "national");
      ms_keywords.put("natural", "natural");
      ms_keywords.put("nchar", "nchar");
      ms_keywords.put("new", "new");
      ms_keywords.put("next", "next");
      ms_keywords.put("no", "no");
      ms_keywords.put("nocreatedb", "nocreatedb");
      ms_keywords.put("nocreateuser", "nocreateuser");
      ms_keywords.put("none", "none");
      ms_keywords.put("not", "not");
      ms_keywords.put("nothing", "nothing");
      ms_keywords.put("notify", "notify");
      ms_keywords.put("notnull", "notnull");
      ms_keywords.put("null", "null");
      ms_keywords.put("nullif", "nullif");
      ms_keywords.put("numeric", "numeric");
      ms_keywords.put("of", "of");
      ms_keywords.put("off", "off");
      ms_keywords.put("offset", "offset");
      ms_keywords.put("oids", "oids");
      ms_keywords.put("old", "old");
      ms_keywords.put("on", "on");
      ms_keywords.put("only", "only");
      ms_keywords.put("operator", "operator");
      ms_keywords.put("option", "option");
      ms_keywords.put("or", "or");
      ms_keywords.put("order", "order");
      ms_keywords.put("out", "out");
      ms_keywords.put("outer", "outer");
      ms_keywords.put("overlaps", "overlaps");
      ms_keywords.put("overlay", "overlay");
      ms_keywords.put("owner", "owner");
      ms_keywords.put("partial", "partial");
      ms_keywords.put("password", "password");
      ms_keywords.put("path", "path");
      ms_keywords.put("pendant", "pendant");
      ms_keywords.put("placing", "placing");
      ms_keywords.put("position", "position");
      ms_keywords.put("precision", "precision");
      ms_keywords.put("prepare", "prepare");
      ms_keywords.put("preserve", "preserve");
      ms_keywords.put("primary", "primary");
      ms_keywords.put("prior", "prior");
      ms_keywords.put("privileges", "privileges");
      ms_keywords.put("procedural", "procedural");
      ms_keywords.put("procedure", "procedure");
      ms_keywords.put("read", "read");
      ms_keywords.put("real", "real");
      ms_keywords.put("recheck", "recheck");
      ms_keywords.put("references", "references");
      ms_keywords.put("reindex", "reindex");
      ms_keywords.put("relative", "relative");
      ms_keywords.put("rename", "rename");
      ms_keywords.put("replace", "replace");
      ms_keywords.put("reset", "reset");
      ms_keywords.put("restart", "restart");
      ms_keywords.put("restrict", "restrict");
      ms_keywords.put("returns", "returns");
      ms_keywords.put("revoke", "revoke");
      ms_keywords.put("right", "right");
      ms_keywords.put("rollback", "rollback");
      ms_keywords.put("row", "row");
      ms_keywords.put("rows", "rows");
      ms_keywords.put("rule", "rule");
      ms_keywords.put("schema", "schema");
      ms_keywords.put("scroll", "scroll");
      ms_keywords.put("second", "second");
      ms_keywords.put("security", "security");
      ms_keywords.put("select", "select");
      ms_keywords.put("sequence", "sequence");
      ms_keywords.put("serializable", "serializable");
      ms_keywords.put("session", "session");
      ms_keywords.put("session_user", "session_user");
      ms_keywords.put("set", "set");
      ms_keywords.put("setof", "setof");
      ms_keywords.put("share", "share");
      ms_keywords.put("show", "show");
      ms_keywords.put("similar", "similar");
      ms_keywords.put("simple", "simple");
      ms_keywords.put("smallint", "smallint");
      ms_keywords.put("some", "some");
      ms_keywords.put("stable", "stable");
      ms_keywords.put("start", "start");
      ms_keywords.put("statement", "statement");
      ms_keywords.put("statistics", "statistics");
      ms_keywords.put("stdin", "stdin");
      ms_keywords.put("stdout", "stdout");
      ms_keywords.put("storage", "storage");
      ms_keywords.put("strict", "strict");
      ms_keywords.put("substring", "substring");
      ms_keywords.put("sysid", "sysid");
      ms_keywords.put("table", "table");
      ms_keywords.put("temp", "temp");
      ms_keywords.put("template", "template");
      ms_keywords.put("temporary", "temporary");
      ms_keywords.put("then", "then");
      ms_keywords.put("time", "time");
      ms_keywords.put("timestamp", "timestamp");
      ms_keywords.put("to", "to");
      ms_keywords.put("toast", "toast");
      ms_keywords.put("trailing", "trailing");
      ms_keywords.put("transaction", "transacation");
      ms_keywords.put("treat", "treat");
      ms_keywords.put("trigger", "trigger");
      ms_keywords.put("trim", "trim");
      ms_keywords.put("true", "true");
      ms_keywords.put("truncate", "truncate");
      ms_keywords.put("trusted", "trusted");
      ms_keywords.put("type", "type");
      ms_keywords.put("unencrypted", "unencrypted");
      ms_keywords.put("union", "union");
      ms_keywords.put("unique", "unique");
      ms_keywords.put("unknown", "unknown");
      ms_keywords.put("unlisten", "unlisten");
      ms_keywords.put("until", "until");
      ms_keywords.put("update", "update");
      ms_keywords.put("usage", "usage");
      ms_keywords.put("user", "user");
      ms_keywords.put("using", "using");
      ms_keywords.put("vacuum", "vacuum");
      ms_keywords.put("valid", "valid");
      ms_keywords.put("validator", "validator");
      ms_keywords.put("values", "values");
      ms_keywords.put("varchar", "varchar");
      ms_keywords.put("varying", "varying");
      ms_keywords.put("verbose", "verbose");
      ms_keywords.put("version", "version");
      ms_keywords.put("view", "view");
      ms_keywords.put("volatile", "volatile");
      ms_keywords.put("when", "when");
      ms_keywords.put("where", "where");
      ms_keywords.put("with", "with");
      ms_keywords.put("without", "without");
      ms_keywords.put("work", "work");
      ms_keywords.put("write", "write");
      ms_keywords.put("year", "year");
      ms_keywords.put("zone", "zone");

      ms_keys = new ArrayList(ms_keywords.size());

      ms_keys.add("abort");
      ms_keys.add("absolute");
      ms_keys.add("access");
      ms_keys.add("action");
      ms_keys.add("add");
      ms_keys.add("after");
      ms_keys.add("aggregate");
      ms_keys.add("all");
      ms_keys.add("alter");
      ms_keys.add("analyse");
      ms_keys.add("analyze");
      ms_keys.add("and");
      ms_keys.add("any");
      ms_keys.add("array");
      ms_keys.add("as");
      ms_keys.add("asc");
      ms_keys.add("assertion");
      ms_keys.add("assignment");
      ms_keys.add("at");
      ms_keys.add("authorization");
      ms_keys.add("backward");
      ms_keys.add("before");
      ms_keys.add("begin");
      ms_keys.add("between");
      ms_keys.add("bigint");
      ms_keys.add("binary");
      ms_keys.add("bit");
      ms_keys.add("boolean");
      ms_keys.add("both");
      ms_keys.add("by");
      ms_keys.add("cache");
      ms_keys.add("called");
      ms_keys.add("cascade");
      ms_keys.add("case");
      ms_keys.add("cast");
      ms_keys.add("chain");
      ms_keys.add("char");
      ms_keys.add("character");
      ms_keys.add("characteristics");
      ms_keys.add("check");
      ms_keys.add("checkpoint");
      ms_keys.add("class");
      ms_keys.add("close");
      ms_keys.add("cluster");
      ms_keys.add("coalesce");
      ms_keys.add("collate");
      ms_keys.add("column");
      ms_keys.add("comment");
      ms_keys.add("commit");
      ms_keys.add("committed");
      ms_keys.add("constraint");
      ms_keys.add("constraints");
      ms_keys.add("conversion");
      ms_keys.add("convert");
      ms_keys.add("copy");
      ms_keys.add("create");
      ms_keys.add("createdb");
      ms_keys.add("createuser");
      ms_keys.add("cross");
      ms_keys.add("current_date");
      ms_keys.add("current_time");
      ms_keys.add("current_timestamp");
      ms_keys.add("current_user");
      ms_keys.add("cursor");
      ms_keys.add("cycle");
      ms_keys.add("database");
      ms_keys.add("day");
      ms_keys.add("deallocate");
      ms_keys.add("dec");
      ms_keys.add("decimal");
      ms_keys.add("declare");
      ms_keys.add("default");
      ms_keys.add("defaults");
      ms_keys.add("deferrable");
      ms_keys.add("deferred");
      ms_keys.add("definer");
      ms_keys.add("delete");
      ms_keys.add("delimiter");
      ms_keys.add("delimiters");
      ms_keys.add("desc");
      ms_keys.add("distinct");
      ms_keys.add("do");
      ms_keys.add("domain");
      ms_keys.add("double");
      ms_keys.add("drop");
      ms_keys.add("each");
      ms_keys.add("else");
      ms_keys.add("encoding");
      ms_keys.add("encrypted");
      ms_keys.add("end");
      ms_keys.add("escape");
      ms_keys.add("except");
      ms_keys.add("excluding");
      ms_keys.add("exclusive");
      ms_keys.add("execute");
      ms_keys.add("exists");
      ms_keys.add("explain");
      ms_keys.add("external");
      ms_keys.add("extract");
      ms_keys.add("false");
      ms_keys.add("fetch");
      ms_keys.add("first");
      ms_keys.add("float");
      ms_keys.add("for");
      ms_keys.add("force");
      ms_keys.add("foreign");
      ms_keys.add("forward");
      ms_keys.add("freeze");
      ms_keys.add("from");
      ms_keys.add("full");
      ms_keys.add("function");
      ms_keys.add("global");
      ms_keys.add("grant");
      ms_keys.add("group");
      ms_keys.add("handler");
      ms_keys.add("having");
      ms_keys.add("hold");
      ms_keys.add("hour");
      ms_keys.add("ilike");
      ms_keys.add("immediate");
      ms_keys.add("immutable");
      ms_keys.add("implicit");
      ms_keys.add("in");
      ms_keys.add("including");
      ms_keys.add("increment");
      ms_keys.add("index");
      ms_keys.add("inherits");
      ms_keys.add("initially");
      ms_keys.add("inner");
      ms_keys.add("inout");
      ms_keys.add("input");
      ms_keys.add("insensitive");
      ms_keys.add("insert");
      ms_keys.add("instead");
      ms_keys.add("int");
      ms_keys.add("integer");
      ms_keys.add("intersect");
      ms_keys.add("interval");
      ms_keys.add("into");
      ms_keys.add("invoker");
      ms_keys.add("is");
      ms_keys.add("isnull");
      ms_keys.add("isolation");
      ms_keys.add("join");
      ms_keys.add("key");
      ms_keys.add("lancompiler");
      ms_keys.add("language");
      ms_keys.add("last");
      ms_keys.add("leading");
      ms_keys.add("left");
      ms_keys.add("level");
      ms_keys.add("like");
      ms_keys.add("limit");
      ms_keys.add("listen");
      ms_keys.add("load");
      ms_keys.add("local");
      ms_keys.add("localtime");
      ms_keys.add("localtimestamp");
      ms_keys.add("location");
      ms_keys.add("lock");
      ms_keys.add("match");
      ms_keys.add("maxvalue");
      ms_keys.add("minute");
      ms_keys.add("minvalue");
      ms_keys.add("mode");
      ms_keys.add("month");
      ms_keys.add("move");
      ms_keys.add("names");
      ms_keys.add("national");
      ms_keys.add("natural");
      ms_keys.add("nchar");
      ms_keys.add("new");
      ms_keys.add("next");
      ms_keys.add("no");
      ms_keys.add("nocreatedb");
      ms_keys.add("nocreateuser");
      ms_keys.add("none");
      ms_keys.add("not");
      ms_keys.add("nothing");
      ms_keys.add("notify");
      ms_keys.add("notnull");
      ms_keys.add("null");
      ms_keys.add("nullif");
      ms_keys.add("numeric");
      ms_keys.add("of");
      ms_keys.add("off");
      ms_keys.add("offset");
      ms_keys.add("oids");
      ms_keys.add("old");
      ms_keys.add("on");
      ms_keys.add("only");
      ms_keys.add("operator");
      ms_keys.add("option");
      ms_keys.add("or");
      ms_keys.add("order");
      ms_keys.add("out");
      ms_keys.add("outer");
      ms_keys.add("overlaps");
      ms_keys.add("overlay");
      ms_keys.add("owner");
      ms_keys.add("partial");
      ms_keys.add("password");
      ms_keys.add("path");
      ms_keys.add("pendant");
      ms_keys.add("placing");
      ms_keys.add("position");
      ms_keys.add("precision");
      ms_keys.add("prepare");
      ms_keys.add("preserve");
      ms_keys.add("primary");
      ms_keys.add("prior");
      ms_keys.add("privileges");
      ms_keys.add("procedural");
      ms_keys.add("procedure");
      ms_keys.add("read");
      ms_keys.add("real");
      ms_keys.add("recheck");
      ms_keys.add("references");
      ms_keys.add("reindex");
      ms_keys.add("relative");
      ms_keys.add("rename");
      ms_keys.add("replace");
      ms_keys.add("reset");
      ms_keys.add("restart");
      ms_keys.add("restrict");
      ms_keys.add("returns");
      ms_keys.add("revoke");
      ms_keys.add("right");
      ms_keys.add("rollback");
      ms_keys.add("row");
      ms_keys.add("rows");
      ms_keys.add("rule");
      ms_keys.add("schema");
      ms_keys.add("scroll");
      ms_keys.add("second");
      ms_keys.add("security");
      ms_keys.add("select");
      ms_keys.add("sequence");
      ms_keys.add("serializable");
      ms_keys.add("session");
      ms_keys.add("session_user");
      ms_keys.add("set");
      ms_keys.add("setof");
      ms_keys.add("share");
      ms_keys.add("show");
      ms_keys.add("similar");
      ms_keys.add("simple");
      ms_keys.add("smallint");
      ms_keys.add("some");
      ms_keys.add("stable");
      ms_keys.add("start");
      ms_keys.add("statement");
      ms_keys.add("statistics");
      ms_keys.add("stdin");
      ms_keys.add("stdout");
      ms_keys.add("storage");
      ms_keys.add("strict");
      ms_keys.add("substring");
      ms_keys.add("sysid");
      ms_keys.add("table");
      ms_keys.add("temp");
      ms_keys.add("template");
      ms_keys.add("temporary");
      ms_keys.add("then");
      ms_keys.add("time");
      ms_keys.add("timestamp");
      ms_keys.add("to");
      ms_keys.add("toast");
      ms_keys.add("trailing");
      ms_keys.add("transaction");
      ms_keys.add("treat");
      ms_keys.add("trigger");
      ms_keys.add("trim");
      ms_keys.add("true");
      ms_keys.add("truncate");
      ms_keys.add("trusted");
      ms_keys.add("type");
      ms_keys.add("unencrypted");
      ms_keys.add("union");
      ms_keys.add("unique");
      ms_keys.add("unknown");
      ms_keys.add("unlisten");
      ms_keys.add("until");
      ms_keys.add("update");
      ms_keys.add("usage");
      ms_keys.add("user");
      ms_keys.add("using");
      ms_keys.add("vacuum");
      ms_keys.add("valid");
      ms_keys.add("validator");
      ms_keys.add("values");
      ms_keys.add("varchar");
      ms_keys.add("varying");
      ms_keys.add("verbose");
      ms_keys.add("version");
      ms_keys.add("view");
      ms_keys.add("volatile");
      ms_keys.add("when");
      ms_keys.add("where");
      ms_keys.add("with");
      ms_keys.add("without");
      ms_keys.add("work");
      ms_keys.add("write");
      ms_keys.add("year");
      ms_keys.add("zone");
   }

   /**
    * Perform a binary search on the array to find a keyword. Tries to
    * find an approximate match.
    */
   public static String find(String word)
   {
      int len = 0;
      int i = 0;
      int low = 0;
      int high = 0;
      int middle = 0;
      int difference = 0;
      String middleString = "";
      String lastMatch = ""; //useful for when we blow past it
      word = word.toLowerCase();

      len = word.length();
      low = 0;
      high = ms_keys.size()-1;

      while (low <= high) 
      {
         middle = 0;
         difference = 0;
         middle = low + (high - low) / 2;
         lastMatch = middleString;
         middleString = (String)ms_keys.get(middle);
         difference = middleString.compareTo(word);
         
         System.err.println("\tconsidering: ["
                            +middleString+"] w/ diff="+difference);
         if (difference == 0)
         {
            System.err.println("exact match");
            return middleString;
         }else if (difference < 0){
            low = middle + 1;
         }else{
            high = middle - 1;
         }
      }

      //should return highest index candidate
      //return null; //not found
      System.err.println("best match is: ");
      if(word.startsWith(lastMatch))
         return lastMatch;
      else
         return middleString;
   }

   /** 
    * Return true if the token starts with a legal keyword. Do longest
    * prefix matching. Be greedy. Don't stop at 'to' when given 'toast'.
    */
   public static boolean startsWithKeyword(String token)
   {
      token = token.toLowerCase();
      //optimistic
      if(Keywords.isKeyword(token))
         return true;
      boolean detectedKeyword = false;
      boolean temp = false;
      StringBuffer tokenParts = new StringBuffer(token.length());
      // to store any keyword we recognize along the way
      String latestKeywordMatch = ""; 
      for(int i=0;i<token.length();i++)
      {
         tokenParts.append(token.charAt(i));
         temp = Keywords.isKeyword(tokenParts.toString());
         if(temp)
         {
            detectedKeyword = true;
            latestKeywordMatch = tokenParts.toString();
         }
      }
      if(detectedKeyword)
      {
         //due to the caller considering all tokens (not just 'keyword'
         // tokens as indicated by the grammar) this has the possibility
         // for false positives like 'userid' ... it starts with user and
         // doesn't end with the randomization key (unless, of course, 
         // the key is 'id', in which case the 'id' would have been stripped
         // and the SQL statement would have been re-written incorrectly.
         //System.err.println("found that token starts with ["+latestKeywordMatch+"]");
         return true;
      }else{
         return false;
      }
   }

   /**
    * Return an indication of whether the parameter is a valid
    * PostgreSQL keyword. The implementation lowercases the input.
    * The implementation returns <code>false</code> if the input parameter
    * is <code>null</code> or the empty string.
    * @param  word  a string that may be a valid SQL keyword
    * @return a boolean value indicating if the parameter is a keyword
    * 
    */
   public static boolean isKeyword(String word)
   {
      if(null==word || "".equals(word))
         return false;
      word = word.toLowerCase();
      return ms_keywords.containsKey(word);
   }
}
