/***************************************************************************
 *  Copyright (C) 2004 Michael E. Locasto
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the:
 *       Free Software Foundation, Inc.
 *       59 Temple Place, Suite 330 
 *       Boston, MA  02111-1307  USA
 *
 * $Id: JSQLRandomize.java,v 1.3 2004/10/22 06:05:59 locasto Exp $
 **************************************************************************/
package jdbcrand;

import java.io.File;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.security.SecureRandom;
import java.security.NoSuchAlgorithmException;
import java.util.StringTokenizer;
import java.sql.SQLException;

/**
 * The <code>JSQLRandomize</code> class is a utility to convert standard 
 * SQL strings to randomized SQL strings. For example, the following SQL
 * string:
 * <pre>
 * SELECT userid, flags 
 * FROM sometable, flagtable 
 * WHERE (userid='john' AND flagtable.type=1);
 * </pre>
 * could be randomized with the key &quot;<code>1234567890</code>&quot; to
 * become the following randomized SQL query:
 * <pre>
 * SELECT1234567890 userid, flags 
 * FROM1234567890 sometable, flagtable 
 * WHERE1234567890 (userid=1234567890'john' AND1234567890 
 *                  flagtable.type=12345678901);
 * </pre>
 * Notice that every keyword, as well as the equivalence test operator
 * &quot;<code>=</code>&quot; has the key appended to it.
 * <p>
 * An attacker attempting to inject SQL into this string in place of the
 * parameter for the <code>userid</code> field would need to guess the 
 * key because the JDBC driver would expect to decode every keyword. Any
 * keyword (in reality, any token that the lexer creates and the parser
 * expects to be a keyword) that does not have the key appended to it will
 * be flagged as an invalid keyword and an error returned to the client. If
 * the query parses correctly (e.g., it doesn't have any foreign or 
 * injected SQL), then the query is passed down through the driver and then
 * to the database where it is interpreted normally. In no case does randomized
 * SQL ever make it to the database parser. The driver acts as a proxy for us.
 * <p>
 * More specifically, the connection and statement objects provide a way to
 * deal with any concurrency issues. The driver is a shared resource and
 * making it do the de-randomization would be a bottleneck. However, using the
 * <code>java.sql.Connection</code> and <code>java.sql.Statement</code> objects
 * allows the solution to be scaleable, as each <code>Connection</code> or
 * <code>Statement</code> may service one particular client thread (or be
 * shared among a small number of client threads). This setup distributes the
 * work of de-randomization to all the individual objects that are actually
 * performing the SQL calls.
 * <p>
 * An example of injection is the following SQL statement:
 * <pre>
 * SELECT * FROM foo WHERE (name='' OR 1=1); --' AND userid=2;
 * </pre>
 * The original statement looks like this:
 * <pre>
 * SELECT * FROM foo WHERE (name='"+$input+"' AND userid=2);
 * </pre>
 * The attacker has been able to replace the value of the <code>input</code>
 * data field with:
 * <pre>
 * ' OR 1=1); --
 * </pre>
 * If SQL randomization were employed, then the injected code string above
 * would not match the language of the other SQL keywords. If the randomization
 * key in use was <code>1234567890</code>, then the injected code would have to
 * be:
 * <pre>
 * ' OR1234567890 1=12345678901); --
 * </pre>
 * Such an input is difficult and time-consuming for an attacker to guess, 
 * especially if the example value is not used, or a string with more entropy
 * is used as the key (e.g., <code>D@(*F*JADSafdskjIEOIJ#*FDKJ$FJMV#U</code>).
 * <p>
 * As a result, injection attacks become harder to automate and are more
 * noticable to victims (as the attacker must increase his rate of injection
 * attempts, a large number of which will be failed).
 *
 * @author mlocasto@acm.org
 */
public class JSQLRandomize
{

   /**
    * The default randomization key length, currently 64 bytes or
    * 512 bits. The value of this field is in bytes.
    */
   public static final int DEFAULT_KEY_LENGTH = 64;

   /** A flag indicating whether or not the tool is randomizing or
    *  de-randomizing. By default, randomizes.
    */
   private boolean m_randomizeMode = true;

   /**
    * The key to encode with.
    */
   private String m_key = "";
   
   /**
    * The filename string of SQL statements to encode.
    */
   private String m_filename = "";

   /**
    * Encode the given SQL statement by randomizing the keywords with the
    * given key.
    * The current implementation is very stupid. Keywords are obtained by
    * tokenizing the SQL string by spaces and testing if any of the
    * returned tokens are keywords. This approach is wrong, but a useful
    * first stab. You have 2 problems: false positives from user variables
    * and tables (like hard coded strings) and false negatives from SQL
    * keywords that are butted up against symbols (like CHAR(128)). This
    * will be fixed by adding a full-blown SQL parser. Once a keyword is
    * matched, it can append the randomization key.
    * 
    * @param   sql   the SQL string to encode by appending the key to it
    * @param   key   the (de)randomization key
    * @return  a string containing the randomized SQL statement
    */
   public String encode(String sql, String key)
   {
      StringBuffer sb = new StringBuffer(2048);
      StringTokenizer tokens = new StringTokenizer(sql, " \n\t\r\f", false);
      String token = "";
      while(tokens.hasMoreTokens())
      {
         token = tokens.nextToken();
         if(Keywords.isKeyword(token))
         {
            //append key
            sb.append(" ");
            sb.append(token);
            sb.append(key);
         }else{
            sb.append(" ");
            sb.append(token);
         }
      }
      return sb.toString().trim();
   }

   /**
    * Decode the given SQL statement by derandomizing the keywords with the
    * given key. This tool really doesn't do de-randomization. De-randomization
    * takes place in the driver. This method is merely a reference
    * implementation.
    * <p>NOTE: not implemented. Defaults to null (identity) decode.
    * 
    * @param   sql   the SQL string to decode by appending the key to it
    * @param   key   the (de)randomization key
    * @return  a string containing the derandomized SQL statement
    */
   public String decode(String sql, String key)
      throws SQLException
   {
      StringBuffer sb = new StringBuffer(2048);
      StringTokenizer tokens = new StringTokenizer(sql, " \n\t\r\f", false);
      String token = "";
      while(tokens.hasMoreTokens())
      {
         token = tokens.nextToken();
         if(token.endsWith(key))
         {
            int k = token.lastIndexOf(key);
            //assert (k!=-1);
            //this will break if toast is submitted and 'ast' is the key
            String possibleKeyword = token.substring(0,k);
            if(Keywords.isKeyword(possibleKeyword))
            {
               sb.append(" ");
               sb.append(possibleKeyword); //correct de-randomization
               //possibleKeyword represents a 'stripped' SQL command, we
               // effectively de-randomized it in the previous step.
            }else{
               sb.append(" ");
               sb.append(token); //oops, this was something else, sorry
            }
         }else{
            if(Keywords.startsWithKeyword(token))
            {
               //covers both == to keyword and startswith
               //attacker tried to inject SQL with no key or with bad key
               //throw new SQLException
               String errMessage = "["+token
                  +"] contains an improperly randomized keyword";
               throw new SQLException(errMessage);
                   
            }else{
               //doesn't start with keyword and doesn't end in key
               //let it through
               sb.append(" ");
               sb.append(token);
            }
            //if token is a keyword, then this was a keyword without the
            // randomization key. SQL injection attack (false positive with
            // toast..to case
            
            //if token is not a keyword, then it may be an SQL injection
            //attack, we need to see if it starts with a keyword. If it
            // doesn't start with a keyword, then it is probably just a 
            // symbol or regular user data.

            //token does not end with the key, so see if it starts with
            // a keyword. If it doesn't, then this is something else, like
            // a right paren that didn't end with the key and is not a keyword
            // if, however, this token does start with a keyword, then we've
            // caught someone trying to guess the key (or having a null key).
            // XXX - Again, this is subject to false positives, like if a user
            // has a table name of selectpeople and the key is not 'people'.
            // In the future, we will only consider tokens that are 
            // supposed to be keywords, so we won't run into this problem.
            // For right now, we don't care.
            /*
            String bestMatch = Keywords.find(token);
            if(token.startsWith(bestMatch))
            {
               throw new SQLException("["+token+"] is a keyword");
            }else{
               sb.append(" ");
               sb.append(token);
            }
            */
         }
      }
      return sb.toString().trim();

      //first cut: check if sql tokens.endsWith(key), then strip and see
      // if remainder is a keyword. suffers from false positives, if a user
      // table happens to end in the key and begin with a command.

      //use parser to recognize keywords with a possible randomization
      // key appended. Strip off the randomization key (if any) and see
      // if it matches the 'key' input param. If not, throw an exception
      // or return the empty string (or an error message).
   }

   /**
    * Generate a random high-entropy JDBC-SQL randomization key of the
    * specified length. This method returns a string of integer values
    * separated by colons, one for each byte of the key as indicated
    * by the input parameter. This output should be mapped to printable
    * characters (like A..Za..z0..9)
    *
    * @param  length  the length in <code>byte</code> elements of the key.
    */
   public String generateKey(int length)
   {
      if(length<=0||length<JSQLRandomize.DEFAULT_KEY_LENGTH)
      {
         System.err.println("Warning: key length ("
                            +length
                            +") is shorter than default ("
                            +JSQLRandomize.DEFAULT_KEY_LENGTH
                            +"). Length will be set to default.");
         length = JSQLRandomize.DEFAULT_KEY_LENGTH;
      }
      byte [] keydata = new byte[length];
  
      SecureRandom random = null;
      try
      {
         random = SecureRandom.getInstance("SHA1PRNG");
         random.nextBytes(keydata);
      }catch(NoSuchAlgorithmException nsex){
         System.err.println("failed to generate key, no random alg found.");
         return "";
      }
      StringBuffer key = new StringBuffer(3*length);
      for(int i=0;i<keydata.length;i++)
      {
         key.append( 127+((int)keydata[i]) );
         key.append(":");
      }
      key.deleteCharAt(key.length()-1);
      return key.toString();
   }

   /**
    * Set the key material.
    * @param  key  the keymaterial. No checks are done on the key for
    *              length or goodness.
    */
   public void setKey(String key)
   {
      this.m_key = key;
   }

   /**
    * Set the name of the file containing SQL statements to encode.
    * @param  filename  the name of the file
    */
   public void setFile(String filename)
   {
      this.m_filename = filename;
   }

   public void setRandomizeMode(boolean mode)
   {
      m_randomizeMode = mode;
   }

   /**
    * For each line in the file, encode it with the key set by the
    * {@link #setKey(String key)} method.
    */
   public void encodeFile()
      throws IllegalArgumentException, IOException
   {
      if(null==m_key||"".equals(m_key))
         throw new IllegalArgumentException("key is null");
      if(null==m_filename||"".equals(m_filename))
         throw new IllegalArgumentException("filename is null");
      if(m_key.length()<JSQLRandomize.DEFAULT_KEY_LENGTH)
         System.err.println("Warning: key material length ("
                            +m_key.length()
                            +") is shorter than default ("
                            +JSQLRandomize.DEFAULT_KEY_LENGTH
                            +")");

      File file = new File(m_filename);
      if(null==file||!file.exists())
         throw new IllegalArgumentException("file does not exist");
      if(!file.canRead())
         throw new IllegalArgumentException("no permissions to read file");
      if(file.isDirectory())
         throw new IllegalArgumentException("file is directory. can't read.");

      String line = "";
      BufferedReader reader =
         new BufferedReader(new FileReader(file));
      
      while( null!=(line=reader.readLine()))
      {
         line = line.trim();
         if(m_randomizeMode)
         {
            line = encode(line,m_key);
            System.out.println(line);
         }else{
            try{
               line = decode(line,m_key);
               System.out.println(line);
            }catch(SQLException sqlex){
               System.err.println("statement rejected: "+sqlex.getMessage());
            }
         }
      }
      reader.close();
   }

   /**
    * Open up a buffered stream from stdin and read SQL statements line
    * by line.
    */
   public void listen()
      throws IllegalArgumentException, IOException
   {
		InputStreamReader isr = new InputStreamReader(System.in);
		BufferedReader keyboard = new BufferedReader(isr);
      String input = "";

      if(null==m_key||"".equals(m_key))
         throw new IllegalArgumentException("key is null");
      if(m_key.length()<JSQLRandomize.DEFAULT_KEY_LENGTH)
         System.err.println("Warning: key material length ("
                            +m_key.length()
                            +") is shorter than default ("
                            +JSQLRandomize.DEFAULT_KEY_LENGTH
                            +")");

      while( null!=(input=keyboard.readLine()))
      {
         input = input.trim();
         if(m_randomizeMode)
         {
            input = encode(input,m_key);
            System.out.println(input);
         }else{
            try{
               input = decode(input,m_key);
               System.out.println(input);
            }catch(SQLException sqlex){
               System.err.println("statement rejected: "+sqlex.getMessage());
            }
         }
      }
      keyboard.close();
   }

   //--------------------------------------------------------- private

   /**
    * Print out a usage message.
    */
   private static void doUsage()
   {
      System.out.println("jsqlr --genkey [keylength]");
      System.out.println("jsqlr [-d|-r] --key [key] [file]");
      System.out.println("jsqlr [-d|-r] --key [key]");
   }

   /**
    * Print a message to the <code>System.err</code> stream.
    * @param  message  the message to output
    */
   private static void serr(String message)
   {
      System.err.println(message);
   }

   //--------------------------------------------------------- MAIN

   /**
    * Given a line, parse it according to the SQL99 grammar and append
    * the given key to each keyword.
    * <p>
    * Invoke this program by doing the following:
    * <pre>
    * java JSQLRandomize [-r|-d] --key [key] [file]
    * java JSQLRandomize [-r|-d] --key [key]
    * java JSQLRandomize --genkey [key length]
    * </pre>
    * In the first invocation form, the <code>file</code> argument should
    * specify a file containing SQL statements, one per line.
    * In the second invocation form, the program expects input from stdin.
    * In any invocation form, the program outputs to <code>stdout</code>.
    *
    */
   public static void main(String [] args)
   {
      JSQLRandomize jsqlr = null;

      if(args.length==4 && "--key".equals(args[1]))
      {
         jsqlr = new JSQLRandomize();
         jsqlr.setKey(args[2]);
         jsqlr.setFile(args[3]);
         try{
            if(args[0].equals("-r"))
            {
               jsqlr.setRandomizeMode(true);
            }else if(args[0].equals("-d")){
               jsqlr.setRandomizeMode(false);
            }else{
               serr("Error: bad args for -r/-d option");
               System.exit(-8);
            }            
            jsqlr.encodeFile();
         }catch(IllegalArgumentException illargex){
            serr("Error: "+illargex.getMessage()+" Check args.");
            System.exit(-2);
         }catch(IOException ioex){
            serr("I/O Error: "+ioex.getMessage());
            System.exit(-3);
         }
         System.exit(0);
      }else if(args.length==2 && "--genkey".equals(args[0])){
         int keylength = JSQLRandomize.DEFAULT_KEY_LENGTH;
         String key = "";
         try{
            keylength = Integer.parseInt(args[1]);
         }catch(NumberFormatException nfex){
            keylength = JSQLRandomize.DEFAULT_KEY_LENGTH;
            serr("key length not a number. proceeding with default.");
         }
         jsqlr = new JSQLRandomize();
         key = jsqlr.generateKey(keylength);
         System.out.println(key);
         System.exit(0);
      }else if(args.length==3 && "--key".equals(args[1])){
         jsqlr = new JSQLRandomize();
         jsqlr.setKey(args[2]);
         try{
            if(args[0].equals("-r"))
            {
               jsqlr.setRandomizeMode(true);
            }else if(args[0].equals("-d")){
               jsqlr.setRandomizeMode(false);
            }else{
               serr("Error: bad args for -r/-d option");
               System.exit(-9);
            }            
            jsqlr.listen();
        }catch(IllegalArgumentException illargex){
            serr("Error: "+illargex.getMessage()+" Check args.");
            System.exit(-4);
         }catch(IOException ioex){
            serr("I/O Error: "+ioex.getMessage());
            System.exit(-5);
         }
         System.exit(0);
      }else{
         doUsage();
         System.exit(-1);
      }

   }
}
