/***************************************************************************
 *  Copyright (C) 2004 Michael E. Locasto
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the:
 *       Free Software Foundation, Inc.
 *       59 Temple Place, Suite 330 
 *       Boston, MA  02111-1307  USA
 *
 * $Id: RJDBCTestMain.java,v 1.1.1.1 2004/08/30 14:50:02 locasto Exp $
 **************************************************************************/
package locasto.test;

import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.FileNotFoundException;
import locasto.prompt.Prompt;
import locasto.prompt.PromptDataListener;

/**
 * A test client that uses the modified pgsql driver. This
 * modified driver can perform SQL instruction de-randomization.
 * The driver, statement, and connection classes effectively act
 * as a proxy for us.
 *
 * @author mlocasto@acm.org
 */
public class RJDBCTestMain
   implements PromptDataListener
{

   /**
    * A handle to our prompt.
    */
   private Prompt m_prompt = null;

   /**
    * The pipe to communicate via the driver to the DB.
    */
   private Connection m_connection = null;

   /**
    * The object that maintains state about our queries and commands.
    */
   private Statement m_statement = null;


   //------------------------------------------------------------- methods

   /**
    * Initialize the test by providing it with a database URL to 
    * connect to, as well as username and password. Finally, also provide
    * the <code>SQLRandomizationKey</code> parameter as part of the URL.
    * Doing so should make the driver attempt to decode every SQL string
    * by stripping the key off any keywords (including special symbols).
    *
    * @param   filename   the name of the file containing the database URL
    */
   public void init(String filename)
      throws ClassNotFoundException, InstantiationException, 
             SQLException, FileNotFoundException, IOException
   {
      Class.forName("org.postgresql.Driver");
      String url = "";
      File file = new File(filename);
      if(null==file||!file.exists()||file.isDirectory())
         throw new IllegalArgumentException("bad config file");
      BufferedReader reader = new BufferedReader(new FileReader(file));
      url = reader.readLine();
      if(null==url)
         url = "";
      System.out.println("Connecting to: "+url+" ...");
      reader.close();

      m_connection = DriverManager.getConnection(url,"michael","");
      m_statement = m_connection.createStatement();

      //register JVM shutdown hook
      Runtime jvm = Runtime.getRuntime();
      /*
      jvm.addShutdownHook(
                               new Thread()
                               {
                                  public void run()
                                  {
                                     System.out.println("running sd hook");
                                     RJDBCTestMain.this.shutDown();
                                  }
                               }
      );
      */

      m_prompt = new Prompt(false,"exit","jsql> ");
      m_prompt.addPromptDataListener(((PromptDataListener)this));
   }

   /**
    * Gracefully close the prompt and disconnect from the database.
    */
   public void shutDown()
   {
      System.out.println("Shutting down...");
      if(null!=m_prompt)
         m_prompt.shutdown();
      try{
         if(null!=m_statement)
            m_statement.close();
         if(null!=m_connection&&!m_connection.isClosed())
            m_connection.close();
      }catch(SQLException sqlex){
         System.err.println("error shutting down: "+sqlex.getMessage());
      }
      System.exit(0);
   }

   //---------------------------------------- interface PromptDataListener

   public void processPromptData(String data)
   {
      if("exit".equalsIgnoreCase(data))
         return;
      data = data.trim();
      ResultSet resultSet = null;
      try
      {
         System.out.println(m_statement.execute(data));
         //         resultSet = m_statement.executeQuery(data);
         //display(resultSet);
      }catch(SQLException sqlex){
         System.err.println("error: "+sqlex.getMessage());
      }
   }

   public void stopProcessingPromptData()
   {
      shutDown();
   }

   //------------------------------------------------------ private

   /**
    * Display minimal info about the result set. Really not important.
    */
   private void display(ResultSet rs)
      throws SQLException
   {
      if(null==rs)
         return;
      System.out.println(rs.next());
   }

   /**
    * Spit out a usage message.
    */
   private static void doUsage()
   {
      System.err.println("java RJDBCTestMain [connection info file]");
   }

   /**
    * The main point of this test is to supply a randomized SQL statement
    * to this object and have it execute the statement against a database.
    * The argument to this object is a file containing the 
    * de-randomization key and connection information. The key is 
    * embedded in the db URL.
    * <p>USAGE info:
    * <pre>
    * java RJDBCTestMain [connection.info.txt]
    * </pre>
    * <p>
    * You can also redirect the output of a file to std in and the program
    * will use the prompt to read std in for SQL statements. This is very
    * useful for testing purposes, like so:
    * <pre>
    * [user@host]$ cat sql-statements.txt > java RJDBCTestMain cxn-info.txt
    * </pre>
    * 
    */
   public static void main(String [] args)
   {
      RJDBCTestMain application = null;

      if(args.length==1)
      {
         try{

            application = new RJDBCTestMain();
            application.init(args[0]);
         }catch(Exception ex){
            System.err.println("("
                               +ex.getClass().getName()
                               +") occurred. Message = "
                               +ex.getMessage());
            application.shutDown();
            System.exit(-2);
         }

      }else{
         doUsage();
         System.exit(-1);
      }

   }
}
