/***************************************************************************
 *  Copyright (C) 2004 Michael E. Locasto
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the:
 *       Free Software Foundation, Inc.
 *       59 Temple Place, Suite 330 
 *       Boston, MA  02111-1307  USA
 *
 * $Id: Prompt.java,v 1.2 2004/08/30 14:54:32 locasto Exp $
 **************************************************************************/
package locasto.prompt;

import java.io.*;
import java.util.Vector;

/** 
 * A class for gathering input from the command line.
 * Runs as a separate thread so as not to block your
 *   main application from its work.
 * Can be run standalone as an echo prompt, which is TONS of fun.
 * <p>SAMPLE USE:<br>
 * <pre>
 * &gt; javac locasto/prompt/Prompt*.java
 * &gt; java -cp . locasto.prompt.Prompt
 * 
 * jprompt$ _
 * jprompt$ exit
 * &gt;
 * </pre>
 * <p>EXAMPLE USAGE IN SRC:<br>
 * <pre>
 *   import locasto.prompt.*;
     ...
     Prompt prompt = new Prompt();
     prompt.addPromptDataListener(((PromptDataListener)this));
   </pre>
 * Stuff to work on:
 * <ul>
 *  <li>
 *  provide the prompt class with an XML list of command specific
 *   to your application so that you can tab-autocomplete them</li>
 *  <li>
 *  add the ability to distinguish if your provided prompt string has
 *   a space at the end, if not, make sure there is</li>
 * </ul>
 *
 * @author mlocasto@acm.org
 */
public class Prompt 
   implements Runnable
{
        
    public static final String DEFAULT_PROMPT_STRING = "jprompt$ ";
    public static final String DEFAULT_EXIT_SEQUENCE = "exit";
    
    private Vector dataListeners = new Vector();
        
    private String input="";
    private Thread listener = null;
    private boolean stop = false;
    private boolean echoOn = false;
    private InputStreamReader isr;
    private BufferedReader keyboard;
    private String promptString = DEFAULT_PROMPT_STRING;
    private String exitSequence = DEFAULT_EXIT_SEQUENCE;
    
    public Prompt()
    {
	this(false, DEFAULT_EXIT_SEQUENCE, DEFAULT_PROMPT_STRING);
    }

    public Prompt(boolean hasEcho, String exitSequence, String promptString)
    {
	this.promptString = promptString;
	this.exitSequence = exitSequence;
	this.echoOn = hasEcho;
	isr = new InputStreamReader(System.in);
	keyboard = new BufferedReader(isr);
	init();
    }

    public boolean isExitSequence(String data)
    {
	//assert exitSequence!=null;
	return(exitSequence.equals(data));
    }

    /** We only want one prompt and then an exit. */
    public void useOnce()
    {
	stop = true;
    }

    public void addPromptDataListener(PromptDataListener pdl)
    {
	dataListeners.add(pdl);
    }
        
    private synchronized void stopListeners()
    {
	int size = dataListeners.size();
	PromptDataListener temp = null;
	for( int i=0;i<size;i++ )
	{
	    temp = (PromptDataListener)dataListeners.elementAt(i);
	    temp.stopProcessingPromptData();
	}
    }

    private synchronized void notifyListeners(String input){
	int size = dataListeners.size();
	PromptDataListener temp = null;
	for( int i=0;i<size;i++ )
	{
	    temp = (PromptDataListener)dataListeners.elementAt(i);
	    temp.processPromptData(input);
	}
    }
        
    private void init()
    {
	listener = new Thread( ((Runnable)this) );
	listener.start();
    }
        
    public void run(){
        
	System.err.println( "" );
	
	do{
	    listener.yield();
	    try {
		System.out.print( promptString );
		input = keyboard.readLine();
		/* if readLine() == null, die */
		if(null==input)
		{
		    stop=true;
		    stopListeners();
		}else{                                  
		    /* echo for debug purposes */
		    if(echoOn)
			System.out.println(input);
		    if(input.equals(exitSequence))
		    {
			stop=true;
			stopListeners();
		    }else{
			notifyListeners(input);
		    }
		}		
	    }catch(Exception e){
		System.err.println("("+e.getClass().getName()
				   +") getting prompt input: "
				   +e.getMessage());
		//e.printStackTrace();
		//System.exit( -2 );
	    }

	}while(!stop);
	//shutdown();
    }

    public void shutdown(){
	stop = true;
	try{
	    keyboard.close();
	}catch( Exception e ){
	    e.printStackTrace();
	    System.err.println(e.getMessage());
	}
    }
        
    
    public static void main( String [] args )
    {
	new Prompt( true, "exit", "jprompt$ " );
    }
}
