/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this software; If not, write to the Free Software Foundation,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
                
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netdb.h>

#include <stdio.h>
#include <regex.h>
#include <keynote.h>
#include <ar.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>

#include "dava.h"

/* These is only needed to pull in the SSL include files */
#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#define MAX_CREDENTIAL 1024000
#define CREDENTIALS	"CREDENTIALS"
#define MAX_REQUESTER	10240
#define REQUESTER	"REQUESTER"
#define MAX_REQUEST	102400
#define REQUEST		"REQUEST"

#define FT_DIR_PATH	"./ft-files"
#define HOSTNAME	"localhost"

int ft_port = 8080;

void keynote_debug(int sessionid);
void process_request(FILE *fd, char *pos);
int process_get_request(char * policy_assertions, FILE *fd, char *handle);
void process_put_request(char * policy_assertions, FILE *fd, long file_len, char *fname);


void usage(int d) {
	fprintf(stderr, "usage(%d) server -P policy -p port -u public-key -r private-key\n", d);
	exit(1);
}

#define NUM_RETURN_VALUES 2

char *returnvalues[NUM_RETURN_VALUES];

int get_arfile(FILE *fd, char *name, int max_buf, char **buf)
{
	struct ar_hdr arf_header;
	int i;
	int c;
	char *vp;
	int flen;
	int iolen;

	if (fread(&arf_header, sizeof(struct ar_hdr), 1, fd) != 1) {
		fprintf(stderr, " can't read %s header\n", name);
		perror("fread");
		exit(1);
	}

	arf_header.ar_name[strlen(name)]='\0';
	if (strncmp(arf_header.ar_name, name, strlen(name)) != 0) {
		for (i=0; i< sizeof(arf_header.ar_name); i++)
			fprintf(stderr, "0x%x ", arf_header.ar_name[i] & 0xff);
		fprintf(stderr, " BAD NAME, expected %s\n", name);
		exit(1);
	}

	arf_header.ar_size[sizeof(arf_header.ar_size)] = '\0';
	flen = strtol(arf_header.ar_size, NULL, 8);
// fprintf(stderr, "get_arfile: reading %d bytes (%s in octal)\n", flen, arf_header.ar_size);
	/*
	 * archive are always an even number of bytes long, files
	 * which are an odd number of bytes long are padded with a newline (`\n')
	 * character, although the size in the header does not reflect this.
	 * So, if the length is odd, we read an extra byte from the file.
	 */

	iolen = flen + (flen % 2); 
	if (iolen > max_buf)
		return(-1);
	if ((vp = malloc(iolen+1)) == NULL) {
		fprintf(stderr, "malloc(%d) failed\n", iolen);
		exit(1);
	}

	if ((c = fread(vp, iolen, 1, fd)) != 1) {
		fprintf(stderr, " can't read %s data\n", name);
		perror("fread");
		exit(1);
	}
	if (vp[flen-1] == '\n')		/* zap trailing LF */
		vp[flen-1] = '\0';
	vp[flen] = '\0';	/* NULL terminate the string */
// fprintf(stderr, "%s = <<<%s>>>\n", name, vp);
	*buf = vp;
	return(flen);
}

char *mkhandle(long *h)
{
	int i;
        struct stat fileinfo;
        static char fname[MAXPATHLEN];

        for (i=1; i<22; i++) {
                *h = random();
                snprintf(fname, MAXPATHLEN, "%s/ft%08x", FT_DIR_PATH, *h);
                if (lstat(fname, &fileinfo) < 0)  
                        return(fname);
        }
        fprintf(stderr, "Couldn't allocate a handle after %d attempts\n", i);
        exit(1);
}

/* 
 */

char *my_public_key;
char *my_private_key;

main(int argc, char **argv)
{

	int fd;
	FILE *sfd;
	int c;
	int policy_len;
	char *policy_file = "policy";
	char *policy_assertions;
	struct sockaddr_in server;
	int pid, s0, s;
	struct stat fileinfo;

	while (argc > 1) {
	    if (argv[1][0] == '-') {
		switch (argv[1][1]) {
		case 'p':
			argv++ ; argc--;
			ft_port = atoi(argv[1]);
			break;
		case 'P':
			argv++ ; argc--;
			policy_file = argv[1];
			break;
		case 'u':
			argv++ ; argc--;
			my_public_key = argv[1];
			break;
		case 'r':
			argv++ ; argc--;
			my_private_key = argv[1];
			break;
		default:
			usage(1);
		}
	    } else
		usage(2);
	    argv++ ; argc--;
	}

	policy_assertions = copy_file(policy_file, &policy_len);

	
	/*
	 * Wait for client connection!
	 */
	if ((s0 = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("opening stream socket");
		exit(1);
	}

	server.sin_family = AF_INET;
	server.sin_addr.s_addr = INADDR_ANY;
	server.sin_port = htons(ft_port);
	if (bind(s0,  (struct sockaddr *) &server, sizeof(server)) < 0 ) {
		perror("binding stream socket");
		exit(1);
	}
	listen(s0, 10);
	while (1) {
		if ((s = accept(s0, 0, 0)) < 0) {
			// perror("Accepting a connection");
			exit(1);
		}
		if (fork() == 0) {
			close(s0);
			if ((sfd = fdopen(s, "r+")) == NULL) {
				perror("fdopen");
				exit(1);
			}
			process_request(sfd, policy_assertions);
		} else {
			close(s);
			wait3(0,WNOHANG,0);
		}
	}
}

#define IOBUFSIZE 10240
static char iobuf[IOBUFSIZE];
#define MAXNUM 12

void process_request(FILE *fd, char *pass)
{
	long file_length;
	char name_buf[MAXPATHLEN];

	/*
	 * Read request to determine if this is a GET or a PUT
	 */
        read_token(fd, iobuf, IOBUFSIZE, ':');
        if (strncmp(iobuf, "110 PUT", strlen("110 PUT")) == 0) {
		char *file_name = name_buf;

		/* read file length */
		read_token(fd, iobuf, IOBUFSIZE, ':');
		file_length = atol(iobuf);
		/* read file name */
		read_token(fd, iobuf, IOBUFSIZE, ':');
		strlcpy(file_name, iobuf, MAXPATHLEN);
		process_put_request(pass, fd, file_length, file_name);
                exit(0);
        }
	else if (strncmp(iobuf, "111 GET", strlen("111 GET")) == 0) {
		char *file_handle = name_buf;
		/* read file handle */
		read_token(fd, iobuf, IOBUFSIZE, ':');
		strlcpy(file_handle, iobuf, MAXPATHLEN);
		process_get_request(pass, fd, file_handle);
                exit(0);
        } else
                exit(1);
}

static char *action_authorizer; /* read from ar file submitted by client */

int load_keynote(char * policy_assertions, FILE *fd)
{
	int c;
	char buf[1024];
	char *credential_assertions;
	char **decomposed;
	int sessionid, num, i, j;

    /*
     * We are creating a new KeyNote session here. A session may be temporary
     * (we create it, add policies, credentials, authorizers, action set, do
     *  the query, and then we destroy it), or it may be persistent (if, for
     * example, policies remain the same for a while).
     *
     * In this example, we'll just assume the session is temporary, but there
     * will be comments as to what to do if this were a persistent session.
     */
    sessionid = kn_init();
    if (sessionid == -1)
    {
	fprintf(stderr, "Failed to create a new session.\n");
	exit(1);
    }

    /*
     * Assume we have read a file, or somehow securely acquired our policy
     * assertions, and we have stored them in policy_assertions.
     */

    /* Let's find how many policies we just "read". */
    decomposed = kn_read_asserts(policy_assertions, strlen(policy_assertions),
				 &num);
    if (decomposed == NULL)
    {
	fprintf(stderr, "Failed to allocate memory for policy assertions.\n");
	exit(1);
    }

    /*
     * If there were no assertions in the first argument to kn_read_asserts,
     * we'll get a valid pointer back, which we need to free. Note that this
     * is an error; we always MUST have at least one policy assertion.
     */
    if (num == 0)
    {
	free(decomposed);
	fprintf(stderr, "No policy assertions provided.\n");
	exit(1);
    }

    /*
     * We no longer need a copy of policy_assertions, so we could 
     * free it here.
     */
	free(policy_assertions);

    /*
     * decomposed now contains num pointers to strings, each containing a
     * single assertion. We now add them all to the session. Note that
     * we must provide the ASSERT_FLAG_LOCAL flag to indicate that these
     * are policy assertions and thus do not have a signature field.
     */
    for (i = 0; i < num; i++)
    {
// fprintf(stderr, "Assertion %d = <<<%s>>>\n", i, decomposed[i]);
	j = kn_add_assertion(sessionid, decomposed[i],
			     strlen(decomposed[i]), ASSERT_FLAG_LOCAL);
	if (j == -1)
	{
	    switch (keynote_errno)
	    {
		case ERROR_MEMORY:
		    fprintf(stderr, "Out of memory, trying to add policy "
			    "assertion %d.\n", j);
		    break;

		case ERROR_SYNTAX:
		    fprintf(stderr, "Syntax error parsing policy "
			    "assertion %d.\n", j);
		    break;

		case ERROR_NOTFOUND:
		    fprintf(stderr, "Session %d not found while adding "
			    "policy assertion %d.\n", sessionid, j);
		default:
		    fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			    "while adding policy assertion %d.\n",
			    keynote_errno, j);
		    break;
	    }

	    /* We don't need the assertion any more. */
	    free(decomposed[i]);
	}
    }

	/* Now free decomposed itself. */
	free(decomposed);

	/*
	 * read credentials, authorizer key and request
	 *  These come packaged in an ar(5) file read from stdin
	 *	the credentials are in the ar-file named CREDENTIALS
	 *	the authorizer key is in the ar-file named REQUESTER
	 *	the request is in the ar-file named REQUEST
	 */

	if (fread(buf, SARMAG, 1, fd) != 1) {
		perror("read stdin SARMAG");
		exit(1);
	}
	if (strncmp(buf, ARMAG, SARMAG) != 0) {
		for (i=0; i<SARMAG; i++)
			fprintf(stderr, "0x%x ", (buf[i]) & 0xff);
		fprintf(stderr, " BAD MAGIC\n");
		exit(1);
	}


	/*
	 * Read Credential Assertions from file
	 */
	if ((c = get_arfile(fd, CREDENTIALS, MAX_CREDENTIAL,
		&credential_assertions)) < 0) {
		fprintf(stderr, "failed to read credentials\n");
		exit(1);
	}

    /*
     * So, we have some credentials in credential_assertions, and a key
     * in action_authorizer.
     */

    /* Let's find how many credentials we just "received". */
    decomposed = kn_read_asserts(credential_assertions,
				 strlen(credential_assertions), &num);
    if (decomposed == NULL)
    {
	fprintf(stderr, "Failed to allocate memory for credential "
		"assertions.\n");
	exit(1);
    }

    /*
     * If there were no assertions in the first argument to kn_read_asserts,
     * we'll get a valid pointer back, which we need to free. Note that
     * it is legal to have zero credentials.
     */
    if (num == 0)
    {
	free(decomposed);
	fprintf(stderr, "No credential assertions provided.\n");
    }

    /*
     * We no longer need a copy of credential_assertions, so we could 
     * free it here.
     */
	free(credential_assertions);

    /*
     * decomposed now contains num pointers to strings, each containing a
     * single assertion. We now add them all to the session. Note that here
     * we must NOT provide the ASSERT_FLAG_LOCAL flag, since these are
     * all credential assertions and need to be cryptographically verified.
     */
    for (i = 0; i < num; i++)
    {
	/*
	 * The value returned by kn_add_assertion() is an ID for that
	 * assertion (unless it's a -1, which indicates an error). We could
	 * use this ID to remove the assertion from the session in the future,
	 * if we needed to. We would need to store the IDs somewhere of
	 * course.
	 *
	 * If this were a persistent session, it may make sense to delete
	 * the credentials we just added after we are done with the query,
	 * simply to conserve memory. On the other hand, we could just leave
	 * them in the session; this has no security implications.
	 *
	 * Also note that we could do the same with policy assertions.
	 * However, if we want to delete policy assertions, it usually then
	 * makes sense to just destroy the whole session via kn_close(),
	 * which frees all allocated resources.
	 */
// fprintf(stderr, "Assertion %d = <<<%s>>>\n", i, decomposed[i]);
	j = kn_add_assertion(sessionid, decomposed[i],
			     strlen(decomposed[i]), 0);
	if (j == -1)
	{
	    switch (keynote_errno)
	    {
		case ERROR_MEMORY:
		    fprintf(stderr, "Out of memory, trying to add credential "
			    "assertion %d.\n", j);
		    break;

		case ERROR_SYNTAX:
		    fprintf(stderr, "Syntax error parsing credential "
			    "assertion %d.\n", j);
		    break;

		case ERROR_NOTFOUND:
		    fprintf(stderr, "Session %d not found while adding "
			    "credential assertion %d.\n", sessionid, j);
		default:
		    fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			    "while adding credential assertion %d.\n",
			    keynote_errno, j);
		    break;
	    }

	    /* We don't need the assertion any more. */
	    free(decomposed[i]);
	}
    }

    /* No longer needed. */
    free(decomposed);

	/*
	 * Read the action authorizer from file
	 */
	if ((c = get_arfile(fd, REQUESTER, MAX_REQUESTER,
		&action_authorizer)) < 0) {
		fprintf(stderr, "failed to read action_authorizer\n");
		exit(1);
	}
	pack_string(action_authorizer);
	if (*action_authorizer == '\"') {
		char *vp;

		action_authorizer++;
		vp = strchr(action_authorizer, '\"');
		*vp = '\0';
	}
// fprintf(stderr, "action_authorizer = <<<%s>>>\n", action_authorizer);

    /*
     * Now add the action authorizer. If we have more than one, just
     * repeat. Note that the value returned here is just a success or
     * failure indicator. If we want to later on delete an authorizer from
     * the session (which we MUST do if this is a persistent session),
     * we must keep a copy of the key.
     */
    if (kn_add_authorizer(sessionid, action_authorizer) == -1)
    {
	switch (keynote_errno)
	{
	    case ERROR_MEMORY:
		fprintf(stderr, "Out of memory while adding action "
			"authorizer.\n");
		break;

	    case ERROR_SYNTAX:
		fprintf(stderr, "Malformed action authorizer.\n");
		break;

	    case ERROR_NOTFOUND:
		fprintf(stderr, "Session %d not found while adding action "
			"authorizer.\n", sessionid);
		break;

	    default:
		fprintf(stderr, "Unspecified error while adding action "
			"authorizer.\n");
		break;
	}
    }

    /*
     * Now we need to construct the action set. In a real application, we
     * would be gathering the relevant information. Here, we just construct
     * a fixed action set.
     */
	return(sessionid);
}


/* DOWNLOAD_PRICE is the price you have to pay to retrieve the file, this should
 * be calculated in some way rather than being hardwired into the code
 */
#define DOWNLOAD_PRICE	2

int process_get_request(char * policy_assertions, FILE *fd, char *handle)
{
	char nonce[16];
	static char pmt_amount[MAXNUM];
	int j, c;

	int sessionid;

	
	snprintf(pmt_amount, MAXNUM, "%d", DOWNLOAD_PRICE);

	snprintf(nonce, sizeof(nonce), "%08x%08x", random(), random());
	/* send offer (320 <handle>:<price>:<nonce>) */
fprintf(stderr, "320 %s:%s:%s.\n", handle, pmt_amount, nonce);	
	fprintf(fd, "320 %s:%s:%s\n", handle, pmt_amount, nonce);	

	if ((sessionid = load_keynote(policy_assertions, fd)) < 0) {
		fprintf(stderr, "load keynote failed, abort\n");
		exit(1);
	}

	/*
	* Add the relevant action attributes. Flags is zero, since we are not
	* using any callback functions (ENVIRONMENT_FLAG_FUNC) or a regular
	* expression for action attribute names (ENVIRONMENT_FLAG_REGEX).
	*/
// fprintf(stderr, "app_domain := 'FileTellerPay'\n");
	if (kn_add_action(sessionid, "app_domain", "FileTellerPay", 0) == -1) {
		fprintf(stderr, "kn_add_action(app_domain, FileTellerPay) failed: %s\n", 
			pr_keynote_error(keynote_errno));
		return(-1);
	}

// fprintf(stderr, "currency := 'USD'\n");
	if (kn_add_action(sessionid, "currency", "USD", 0) == -1) {
		fprintf(stderr, "kn_add_action(currency, USD) failed: %s\n", 
			pr_keynote_error(keynote_errno));
		return(-1);
	}
	if (kn_add_action(sessionid, "amount", pmt_amount, 0) == -1) {
		fprintf(stderr, "kn_add_action(amount, %s) failed: %s\n", pmt_amount,
			pr_keynote_error(keynote_errno));
		return(-1);
	}

	if (kn_add_action(sessionid, "handle", handle, 0) == -1) {
		fprintf(stderr, "kn_add_action(handle, %s) failed: %s\n", handle,
			pr_keynote_error(keynote_errno));
		return(-1);
	}
	if (kn_add_action(sessionid, "access", "R", 0) == -1) {
		fprintf(stderr, "kn_add_action(access, R) failed: %s\n", pr_keynote_error(keynote_errno));
		return(-1);
	}
	/* Set the return values for this application -- just "false" and "true" */
	returnvalues[0] = "false";
	returnvalues[1] = "true";

	/* Just do the query. */
	j = kn_do_query(sessionid, returnvalues, NUM_RETURN_VALUES);
	if (j == -1) {
		fprintf(stderr, "kn_do_query failed: %s\n", pr_keynote_error(keynote_errno));
	}
	else
	{
		fprintf(stdout, "Return value is [%s]\n", returnvalues[j]);
	}

	if (j == 0) { /* FALSE */

		/* send error message and return */
		/* TODO check to see WHY the request is refused and report message */
		fprintf(fd, "510 Request refused: unknown error\n");
keynote_debug(sessionid);
		kn_close(sessionid);
		return(-1);
	} else {
		static char fname[MAXPATHLEN];
		FILE *fd_data;

		/* payment was received correctly so we have to accept file and store it */
fprintf(stderr, "360 Ready to send file\n");
		fprintf(fd, "360 Ready to send file\n");

		/* open file and send it */

		snprintf(fname, MAXPATHLEN, "%s/ft%s", FT_DIR_PATH, handle);
		if ((fd_data = fopen(fname, "r")) == NULL) {
			fprintf(stderr, "cannot open data file (%s)\n", fname);
			perror("open");
			return(-1);
		}
		while ((c = getc(fd_data)) != EOF)
			putc(c, fd);
		fclose(fd_data);
		return(0);
	}
}


void process_put_request(char * policy_assertions, FILE *fd, long file_len, char *fname)
{
	int c;
	int sessionid, num, i, j;
	long handle;
	FILE *fd_file;
	char *file_name;
	/* signing request */
	char *pub_key;
	char *priv_key;
	int priv_key_len;
	char *file_access_csr;
	int file_access_csr_len;
	char *signature;
	char nonce[16];
	char pmt_amount[MAXNUM];

// fprintf(stderr, "process_put_request( %D, %s)\n", file_len, fname);

#define UPLOAD_PRICE	2
	pub_key = copy_file(my_public_key, NULL);
	snprintf(nonce, sizeof(nonce), "%08x%08x", random(), random());
	snprintf(pmt_amount, sizeof(pmt_amount), "%d", UPLOAD_PRICE);
fprintf(stderr, "210 %s:%s:%s\n", pmt_amount, nonce, pack_string(pub_key));
	fprintf(fd, "210 %s:%s:%s\n", pmt_amount, nonce, pack_string(pub_key));

	if ((sessionid = load_keynote(policy_assertions, fd)) < 0) {
		fprintf(stderr, "load keynote failed, abort\n");
		exit(1);
	}
    /*
     * Add the relevant action attributes. Flags is zero, since we are not
     * using any callback functions (ENVIRONMENT_FLAG_FUNC) or a regular
     * expression for action attribute names (ENVIRONMENT_FLAG_REGEX).
     */
// fprintf(stderr, "app_domain := 'FileTellerPay'\n");
    if (kn_add_action(sessionid, "app_domain", "FileTellerPay", 0) == -1)
    {
	switch (keynote_errno)
	{
	    case ERROR_SYNTAX:
		fprintf(stderr, "Invalid name action attribute name "
			"[app_domain]\n");
		break;

	    case ERROR_MEMORY:
		fprintf(stderr, "Out of memory adding action attribute "
			"[app_domain = \"test application\"]\n");
		break;

	    case ERROR_NOTFOUND:
		fprintf(stderr, "Session %d not found while adding action "
			"attribute [app_domain = \"test application\"]\n",
			sessionid);
		break;

	    default:
		fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			"while adding action attribute [app_domain = "
			"\"test application\"]\n", keynote_errno);
		break;
	}
    }

// fprintf(stderr, "currency := 'USD'\n");
    if (kn_add_action(sessionid, "currency", "USD", 0) == -1)
    {
	switch (keynote_errno)
	{
	    case ERROR_SYNTAX:
		fprintf(stderr, "Invalid name action attribute name "
			"[some_num]\n");
		break;

	    case ERROR_MEMORY:
		fprintf(stderr, "Out of memory adding action attribute "
			"[some_num = \"1\"]\n");
		break;

	    case ERROR_NOTFOUND:
		fprintf(stderr, "Session %d not found while adding action "
			"attribute [some_num = \"1\"]\n", sessionid);
		break;

	    default:
		fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			"while adding action attribute [some_num = \"1\"]",
			keynote_errno);
		break;
	}
    }

// fprintf(stderr, "amount := %s\n", pmt_amount);
    if (kn_add_action(sessionid, "amount", pmt_amount, 0) == -1)
    {
	switch (keynote_errno)
	{
	    case ERROR_SYNTAX:
		fprintf(stderr, "Invalid name action attribute name "
			"[some_var]\n");
		break;

	    case ERROR_MEMORY:
		fprintf(stderr, "Out of memory adding action attribute "
			"[some_var = \"some other value\"]\n");
		break;

	    case ERROR_NOTFOUND:
		fprintf(stderr, "Session %d not found while adding action "
			"attribute [some_var = \"some other value\"]\n",
			sessionid);
		break;

	    default:
		fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			"while adding action attribute [some_var = "
			"\"some other value\"]\n", keynote_errno);
		break;
	}
    }

    /* Set the return values for this application -- just "false" and "true" */
    returnvalues[0] = "false";
    returnvalues[1] = "true";

    /* Just do the query. */
    j = kn_do_query(sessionid, returnvalues, NUM_RETURN_VALUES);
    if (j == -1)
    {
	switch (keynote_errno)
	{
	    case ERROR_MEMORY:
		fprintf(stderr, "Out of memory while performing authorization "
			"query.\n");
		break;

	    case ERROR_NOTFOUND:
		fprintf(stderr, "Session %d not found while performing "
			"authorization query.\n", sessionid);
		break;

	    default:
		fprintf(stderr, "Unspecified error %d (shouldn't happen) "
			"while performing authorization query.\n",
			keynote_errno);
		break;
	}
    }
    else
    {
	fprintf(stdout, "Return value is [%s]\n", returnvalues[j]);
    }

	if (j == 0) { /* FALSE */
		/* send error message and return */
		/* TODO check to see WHY the request is refused and report message */
		fprintf(fd, "510 Request refused: unknown error\n");
keynote_debug(sessionid);
		kn_close(sessionid);
		return;
	}

	/* payment was received correctly so we have to accept file and store it */
fprintf(stderr, "230 Ready to receive file\n");
	fprintf(fd, "230 Ready to receive file\n");
	
	/* create handle, open file */
	file_name = mkhandle(&handle);
	if ((fd_file = fopen(file_name, "w")) == NULL) {
// fprintf(stderr, "can't open file [%s] for writing\n", file_name);
		fprintf(fd, "511 Internal error\n");
		kn_close(sessionid);
		return;
	}
	for (i = file_len; i > 0; i--) {
		c = getc(fd);
		putc(c, fd_file);  
	}
	fclose(fd_file);
	/* did the remote hang up on us? */
	if (c == EOF) {
		fprintf(stderr, "Connection closed unexepectedly, while reading file %s\n", fname);
		exit(1);
	}
	/* need to read in server's public key */
	pub_key = copy_file(my_public_key, NULL);
	/* copied the file, now return handle to client */
	fprintf(fd, "240 HANDLE:%s:%d:%d:%d:%08x:%s\n", HOSTNAME, ft_port,
		DOWNLOAD_PRICE, file_len, handle, pack_string(pub_key));
fprintf(stderr, "240 HANDLE:%s:%d:%d:%d:%08x:%s\n", HOSTNAME, ft_port, DOWNLOAD_PRICE, file_len, handle, pack_string(pub_key));
	/*
	 * now create and sign file access csr (for the cert that the client needs to access
	 * the file in the future).
	 */
	/* ... and the private key */
	priv_key = copy_file(my_private_key, &priv_key_len);

/* FIX FileTellerPay below should be FileTellerAccess, but I'd rather have a single app_domain
 for now */
	snprintf(iobuf, IOBUFSIZE, "Keynote-Version: 2\n\
Local-Constants:\n\
    AUTH_KEY = %s\n\
    LIC_KEY = \"%s\"\n\
Authorizer: AUTH_KEY\n\
Licensees:  LIC_KEY\n\
Comment: File Access Credential for %s\n\
Conditions: app_domain == \"FileTellerPay\" && handle == \"%08x\"\n\
	&& ( (access == \"R\") || (access == \"RP\") || (access == \"A\") || (access == \"D\") ) \
-> \"true\";\n\
Signature: ", pub_key, action_authorizer, fname, handle);

	file_access_csr_len = strlen(iobuf)+1;
	if ((file_access_csr = malloc(file_access_csr_len)) == NULL) {
		fprintf(stderr, "file_access_csr: malloc(%d) failed\n", file_access_csr_len);
		exit(1);
	}
	memcpy(file_access_csr, iobuf, file_access_csr_len);

	if ((signature = kn_sign_assertion(file_access_csr, file_access_csr_len,
		priv_key, SIG_RSA_SHA1_PKCS1_BASE64, 0)) == NULL)
	{
		switch (keynote_errno)
		{
		    case ERROR_SYNTAX:
			fprintf(stderr, "Invalid algorithm, key, or assertion while signing "
				"<<<%s>>>\n", file_access_csr);
			break;

		    case ERROR_MEMORY:
			fprintf(stderr, "Out of memory while signing "
				"<<<%s>>>\n", file_access_csr);
			break;

		    case ERROR_NOTFOUND:
			fprintf(stderr, "Argument missing while signing  "
				"<<<%s>>>\n", file_access_csr);
			break;

		    default:
			fprintf(stderr, "Unspecified error %d (shouldn't happen) while signing "
				"<<<%s>>>\n", keynote_errno, file_access_csr);
			break;
		}
	}
fprintf(stderr, "250 Transaction Complete\n");
// fprintf(stderr, "%s \"%s\"\n250 Transaction Complete\n", file_access_csr, signature);
	fprintf(fd, "%s \"%s\"\n", file_access_csr, signature);

	/* Destroy the session, freeing all allocated memory. */
	kn_close(sessionid);
	return;
}

void keynote_debug(int sessionid)
{
    int i = 0;
    int j = 0;
    /*
     * Once the query is done, we can find what assertions failed in what way.
     * One way is just going through the list of assertions, as shown here
     * for assertions that failed due to memory exhaustion.
     */

    do
    {
	i = kn_get_failed(sessionid, KEYNOTE_ERROR_MEMORY, j++);
	if (i != -1)
	  fprintf(stderr, "Assertion %d failed due to memory exhaustion.\n",
		  i);
    } while (i != -1);

    /*
     * Another way is to go through the list of failed assertions by deleting
     * the "first" one.
     */
    do
    {
	i = kn_get_failed(sessionid, KEYNOTE_ERROR_SYNTAX, 0);
	if (i != -1)
	{
	    fprintf(stderr, "Assertion %d failed due to some syntax error.\n",
		    i);
	    kn_remove_assertion(sessionid, i);  /* Delete assertion */
	}
    } while (i != -1);

    /*
     * Signature failures, another way.
     */
    for (j = 0, i = kn_get_failed(sessionid, KEYNOTE_ERROR_SIGNATURE, 0);
	 i != -1; i = kn_get_failed(sessionid, KEYNOTE_ERROR_SIGNATURE, j++))
      fprintf(stderr, "Failed to verify signature on assertion %d.\n", i);

    /*
     * Here's how to find all errors.
     */
    for (i = kn_get_failed(sessionid, KEYNOTE_ERROR_ANY, 0); i != -1;
	 i = kn_get_failed(sessionid, KEYNOTE_ERROR_ANY, 0))
    {
	fprintf(stderr, "Unspecified error in processing assertion %d.\n", i);
	kn_remove_assertion(sessionid, i);
    }
}
