// ----------------------------------------------------------------------------
//
// Copyright (C) Columbia University, 1994. All Rights Reserved.
// Sameer A. Nene, Shree K. Nayar, Hiroshi Murase
//
// See file LICENSE for details of software license agreement.
//
// ----------------------------------------------------------------------------
//
// partialsearch.h
//
// Author:               Sameer Nene
// Date:                 05/26/94
// Version:              1.0
// Modification History:
//   06/25/94: PartialSearch is now derived directly from Persistent rather
//             than via a SearchScheme. This was necessary because
//             SearchScheme is no longer derived from a Persistent.
//   06/26/94: Change in Interface necessary due to change in interface of
//             DataSet. Internal data storage no longer as VectAry, but a
//             DataSet.
//
// Bugs:
//
// Classes:
//   PartialSearch
//
// Notes:
//   This header includes declarations for the class PartialSearch. This method
//   is a slight modification of the exhaustive search method and gives a
//   some improvement in performance over FullSearch, but uses a threshold. It
//   is crucial that the threshold be set in the range of the estimated
//   distance between a given point and a point closest to it. If the threshold
//   is too high then the performance degrades to a FullSearch. If too less, it
//   might "miss" the closest point. Partialearch is derived from SearchScheme
//   and hence obeys that protocol. setSearchParameters() is used to set the
//   threshold. The constructor PartialSearch::PartialSearch() is used to
//   construct or initialize the object for(with) the supplied data. Once it is
//   initialized, it can be immediately used to do a search, or can be stored
//   to disk for later use. A mapping between the table index of the supplied
//   data and the corresponding physical quantity has also to be supplied. This
//   ensures that the vector returned by search() has the correct reverse
//   mapping.
//   
// ----------------------------------------------------------------------------

#ifndef PARTIALSEARCH_INCLUDED
#define PARTIALSEARCH_INCLUDED

#ifndef REGISTRY_INCLUDED
#include "registry.h"
#endif

class ErrorScope;

#ifndef SEARCHSCHEME_INLCUDED
#include "searchscheme.h"
#endif

#ifndef VECTOR_INCLUDED
#include "vector.h"
#endif

#ifndef DATASET_INCLUDED
#include "dataset.h"
#endif

#ifndef PERSISTENTSEARCHSCHEME_INCLUDED
#include "persscheme.h"
#endif

#include <stdio.h>

class PartialSearch : public PersistentSearchScheme {

  // Private members
  
  static void *d_protocol;
  static char *d_name_p;
  static Registrar d_registrar;

  static Persistent* createFunc();
  // returns a pointer to a newly created PartialSearch

  double d_thresh;
  Vector d_rangemin, d_rangemax;
  DataSet d_data;

  // PartialSearch cannot be copied or assigned
  
  PartialSearch(const PartialSearch&);
  PartialSearch& operator=(const PartialSearch&);
  
  // Protected Members
  
protected:
  
  virtual int hasProtocol(void *protocol) const;
  // returns 1 if this object is of type(protocol) protocol OR is derived
  // from an object of type(protocol) protocol
  
  // Public members
  
public:
  
  // Construct and destroy

  PartialSearch();
  
  PartialSearch(const DataSet &ds, const Vector &rangemin,
		const Vector &rangemax);
  // Construct a linear search array from points in ds. rangemin & rangemax
  // give min. & max. values of output parameters for every dimension.
    
  ~PartialSearch() {}

  // Accessors

  static PartialSearch* safeCast(Persistent*);
  // Converts Persistent pointer to a PartialSearch pointer if the object
  // pointed to by Persistent is in fact a PartialSearch
  
  Vector search(const Vector &v) const;
  // searches for v and returns result vector
  
  ErrorScope::Error put(FILE*) const;
  // puts PartialSearch into file
  
  const char *name() const;
  // returns name of class

  const VectAry& getData() const;
  // returns a reference to the underlying data

  // Modifiers

  void setSearchParameters(const Vector &v);
  // sets threshold for the box size. v should have only one element
    
  ErrorScope::Error get(FILE*);
  // gets PartialSearch from disk
  
};

#endif

  
