#
# Bindings for the Editor canvas
#

#
# tool definition
#
# Usage: newtool Arrow @$bmd/arrow.xbm \
# "@$bmd/arrow.xbm $bmd/arrowmask.xbm black white"
#
# Tools is a list of all tools

set tools ""

proc newtool {name bitmap cursor} {
    global tools toolBitmap toolCursor
    lappend tools $name
    set toolBitmap($name) $bitmap
    set toolCursor($name) $cursor
}

newtool arrow @$bmd/arrow.xbm left_ptr
newtool state @$bmd/state.xbm tcross
newtool arc @$bmd/arc.xbm draft_small
newtool text @$bmd/text.xbm xterm

# name is the simple name for the binding type
#
# bindings($name) = X name for binding

set bindings(B1) <Button-1>
set bindings(B2) <Button-2>
set bindings(B1Motion) <B1-Motion>
set bindings(B2Motion) <B2-Motion>
set bindings(B1Release) <ButtonRelease-1>
set bindings(B2Release) <ButtonRelease-2>
set bindings(ShiftB1) <Shift-Button-1>
set bindings(ShiftB2) <Shift-Button-2>
set bindings(ShiftB1Motion) <Shift-B1-Motion>
set bindings(ShiftB2Motion) <Shift-B2-Motion>
set bindings(ShiftB1Release) <Shift-ButtonRelease-1>
set bindings(ShiftB2Release) <Shift-ButtonRelease-2>
set bindings(DoubleB1) <Double-Button-1>
set bindings(DoubleB2) <Double-Button-2>
set bindings(Key) <KeyPress>
set bindings(Left) <Left>
set bindings(Right) <Right>
set bindings(Return) <Return>
set bindings(Delete) <Delete>

#
# General binding assignment
#
# Usage:
#
# bindproc {%v %x %y} {view x y} {
#   ... your procedure ...
# }
# bindto arrow B1
# bindto text B2
# ...
#
#
# boundproc($tool$binding) = name of the bound procedure
# bindAArgs($bindprocname) = actual arguments to use to call procedure

set bindprocnum 1
set bindprocname ""

proc bindproc {aargs fargs block} {
    global bindprocnum bindprocname bindAArgs
    set bindprocname binding$bindprocnum ; incr bindprocnum
    set bindAArgs($bindprocname) $aargs
    proc $bindprocname $fargs $block
}

proc bindto {tool binding} {
    global bindprocname
    global bindings boundproc tools
    if { [lsearch $tools $tool] < 0 } {
	error "unknown tool `$tool'"
    }
    if { ![info exists bindings($binding)] } {
	error "unknown binding `$binding'"
    }
    set boundproc($tool$binding) $bindprocname
}

##################################################
#
# Global variables
#

# state being created

set createdState ""

##################################################
#
# Bindings
#

##############################
#
# Arrow tool bindings
#

# Binding for the arrow tool's Button-1

bindproc {%v %x %y} {view x y} {
    set object [$view currentObject]
#    puts "view: $view x: $x y: $y"
    if { $object != "" } {
	if { [ $object info method arrowB1 -body ] != "" } {
	    $object arrowB1 $x $y [$view currentData]
	}
    } else {
	$view unselectAll
    }
}
bindto arrow B1
bindto state B2
bindto arc B2
bindto text B2

# Binding for the arrow tool's Shift Button-1

bindproc {%v %x %y} {view x y} {
    set object [$view currentObject]
#    puts "view: $view x: $x y: $y"
    if { $object != "" } {
	if { [ $object info method shiftArrowB1 -body ] != "" } {
	    $object shiftArrowB1 $x $y [$view currentData]
	}
    } else {
	$view unselectAll
    }
}
bindto arrow ShiftB1
bindto state ShiftB2
bindto arc ShiftB2
bindto text ShiftB2

# Binding for the arrow tool's Button-1 motion

bindproc {%v %x %y} {view x y} { 
    set object [$view currentObject]
#    puts "view: $view x: $x y: $y"
    if { $object != "" } {
	if { [ $object info method arrowB1motion -body ] != "" } {
	    $object arrowB1motion $x $y [$view currentData]
	}
    } 
}
bindto arrow B1Motion
bindto arrow ShiftB1Motion
bindto state B2Motion
bindto state ShiftB2Motion
bindto arc B2Motion
bindto arc ShiftB2Motion
bindto text B2Motion
bindto text ShiftB2Motion

# Binding for the arrow tool's Button-1 release

bindproc {%v %x %y} {view x y} {
    set object [$view currentObject]
#    puts "view: $view x: $x y: $y"
    if { $object != "" } {
	if { [ $object info method arrowB1release -body ] != "" } {
	    $object arrowB1release $x $y [$view currentData]
	}
    }
}
bindto arrow B1Release
bindto arrow ShiftB1Release
bindto state B2Release
bindto state ShiftB2Release
bindto arc B2Release
bindto arc ShiftB2Release
bindto text B2Release
bindto text ShiftB2Release

# Binding for the arrow tool's Double Button-1

bindproc {%v %x %y} {view x y} {
    set object [$view currentObject]
    if { $object != "" } {
	if { [ $object info method arrowDoubleB1 -body ] != "" } {
	    $object arrowDoubleB1 $x $y [$view currentData]
	}
    } else {
	$view unselectAll
    }
}
bindto arrow DoubleB1
bindto state DoubleB2
bindto arc DoubleB2
bindto text DoubleB2

##############################
#
# State Tool Bindings
#

# Binding for the state tool's Button-1

bindproc {%v %x %y} {view x y} {

    set c [$view canvasOf]
    set x [$c canvasx $x]
    set y [$c canvasy $y]

    global createdState

    # Figure out what state, if any, the user clicked in
    #
    # If she clicked on something other than a state,
    # then find something underneath it that is a state

    set parent [lindex [$view findStates overlapping $x $y $x $y] 0]

    # Here, id contains either the State, or "" if clicked outside

    if { $parent != "" } {

	set x [$view viewx $x]
	set y [$view viewy $y]

	set newstate [State \#auto $parent \
			  -xw $x -yn $y -xe [expr $x + 20] -ys [expr $y + 20] \
			  -nameX [expr $x + 5] -nameY [expr $y + 10] ]
	set createdState $newstate

	# We need to find those views that should display this new state.
	#
	# See if the root of each view is among the ancestors of this
	# new state.  If it is, make sure that the depth of the view
	# isn't too shallow to prohibit its visibility

	set ancestors ""
	set state $newstate
	while { $state != "" } {
	    lappend ancestors $state
	    set state [$state parentOf]
	}

	foreach v [itcl_info object -class View] {

	    set distance [lsearch $ancestors [$v rootOf]]

	    # increase the depth of the current view if the new state would
	    # not be visible

	    if { $v == $view && $distance > [$v depthOf] } {
		$view setView [$view rootOf] [expr [$view depthOf] + 1] [$view zoomOf]
	    } else {
		if { $distance >= 0 && $distance <= [$v depthOf] } {
		    StateImage \#auto $newstate $v
		}
	    }
	}

    }
}
bindto state B1

# Binding for the state tool's Button-1 motion

bindproc {%v %x %y} {view x y} {
    global createdState
    if { $createdState != "" } {
	$createdState config -xe [$view viewx $x] -ys [$view viewy $y]
    }
}
bindto state B1Motion

# Binding for the state tool's Button-1 release

bindproc {%v %x %y} {view x y} {
    global createdState
    set createdState ""
}
bindto state B1Release

##############################
#
# Arc Tool Bindings
#

# Binding for the arc tool's Button-1

bindproc {%v %x %y} {view x y} {

    set c [$view canvasOf]
    set x [$c canvasx $x]
    set y [$c canvasy $y]

    global arcUnderConstruction startState

    set startState [lindex [$view findStates overlapping $x $y $x $y] 0]

    if { $startState != "" } {

	set arcUnderConstruction [$c create line $x $y $x $y -arrow last]
    } else {
	set arcUnderConstruction {}
    }

}
bindto arc B1

# Binding for the arc tool's Button-1 motion

bindproc {%v %x %y} {view x y} {

    set c [$view canvasOf]
    set x [$c canvasx $x]
    set y [$c canvasy $y]

    global arcUnderConstruction

    if { $arcUnderConstruction != "" } {
	set coords [$c coords $arcUnderConstruction]
	$c coords $arcUnderConstruction [lindex $coords 0] [lindex $coords 1] $x $y
    }
}
bindto arc B1Motion

# Binding for the arc tool's Button-1 release

bindproc {%v %x %y} {view x y} {

    set c [$view canvasOf]
    set x [$c canvasx $x]
    set y [$c canvasy $y]

    global arcUnderConstruction startState

    if { $arcUnderConstruction != "" } {
	$c delete $arcUnderConstruction
	set arcUnderConstruction {}

	set endState [lindex [$view findStates overlapping $x $y $x $y] 0]

	if { $endState != "" } {

	    # Create a transition and add its image to those views that care

	    set newtrans [Trans \#auto $startState $endState]

	    foreach v [itcl_info object -class View] {

		if { [$v stateIsVisible $startState] } {
		    TransImage \#auto $newtrans $v
		}
	    }

	}
    }
   
}
bindto arc B1Release

##############################
#
# Text Tool Bindings
#

bindproc {%v %x %y} {view x y} {
    set object [$view currentObject]
    $view clearTextFocus
    if { $object != "" } {
	if { [ $object info method textB1 -body ] != "" } {
	    $object textB1 $x $y [$view currentData]
	}
    } else {
	puts "Text Click in Free Space"
    }

}
bindto text B1

bindproc {%v %A} {view a} {
    set textObject [$view textObjectOf]
    if { $textObject != "" } {
	set c [$view canvasOf]
	set id [$view textIDOf]
	$c insert $id insert $a
    }
}
bindto text Key

bindproc {%v} {view} {
    set textObject [$view textObjectOf]
    if { $textObject != "" } {
	set c [$view canvasOf]
	set id [$view textIDOf]
	$c icursor $id [expr [$c index $id insert] - 1]
    }
}
bindto text Left

bindproc {%v} {view} {
    set textObject [$view textObjectOf]
    if { $textObject != "" } {
	set c [$view canvasOf]
	set id [$view textIDOf]
	$c icursor $id [expr [$c index $id insert] + 1]
    }
}
bindto text Right

bindproc {%v} {view} {
    set textObject [$view textObjectOf]
    if { $textObject != "" } {
	set c [$view canvasOf]
	set id [$view textIDOf]
	$c dchars $id [expr [$c index $id insert] - 1]
    }
}
bindto text Delete

bindproc {%v} {view} {
    $view clearTextFocus
}
bindto text Return

##############################
#
# Emacs-like bindings for Entry widgets
#

# Cursor movement

bind Entry <Left> {%W icursor [expr [%W index insert]-1]}
bind Entry <Right> {%W icursor  [expr [%W index insert]+1]}
bind Entry <Control-b> {%W icursor [expr [%W index insert]-1]}
bind Entry <Control-f> {%W icursor [expr [%W index insert]+1]}

# Go to beginning and end of line
bind Entry <Control-a> {%W icursor 0 ; %W view insert}
bind Entry <Control-e> {%W icursor [%W index end]}

# Delete character, remainder of line to the right of the cursor
bind Entry <Control-d> {%W delete insert}
bind Entry <Control-k> {%W delete insert end}

