/***********************************************************************/
class IcParser extends Parser;

options {
  buildAST = true;
  k = 2;
}

tokens {
  EXECS;
  LIST;
}

file
 : (icModule)+
 ;

icModule
 : "ic8:"!
   "module:"^ ID
   instanceTable

   (typeTable)?
   (constantTable)?
   (functionTable)?
   (procedureTable)?

   (signalTable)?
   (implicationTable)?
   (exclusionTable)?  
   (variableTable)?
   (taskTable)?
   (execTable)?
   (actionTable)?
   (pauseTable)?
   statementTable
   "endmodule:"
 ;

/******************************/

instanceTable: "instances:"^ num:Int "root:"! Int (instance)* "end:"! ;

instance: Int^ Colon! instanceRename parent:Int dir:String file:String ;

instanceRename: ID (Slash^ ID)? ;

/******************************/

typeTable: "types:"^ num:Int (type)* "end:"! ;

type: Int^ Colon! ID index index index index ;

/******************************/

constantTable: "constants:"^ num:Int (constant)* "end:"! ;

constant: Int^ Colon! ID index (initialValue)? ;

/******************************/

functionTable: "functions:"^ num:Int (function)* "end:"! ;

function: Int^ Colon! ID pindexn Colon index ;

/******************************/

procedureTable: "procedures:"^ num:Int (procedure)* "end:"! ;

procedure: Int^ Colon! ID pindexn pindexn ;

/******************************/

implicationTable: "implications:"^ num:Int (implication)* "end:"! ;

implication: Int^ Colon! index index;

/******************************/

exclusionTable: "exclusions:"^ num:Int (exclusion)* "end:"! ;

exclusion: Int^ Colon! pindexn ;

/******************************/

signalTable: "signals:"^ num:Int (signal)* "end:"! ;

signal: Int^ Colon! signalKind signalChannel (signalBool)? (signalPrevious)? ;

signalKind
 : "input:"^ ID index
 | "return:"^ ID index
 | "output:"^ ID index
 | "inputoutput:"^ ID index index
 | "sensor:"^ ID
 | "local:"^ ID
 | "trap:"^ ID
 ;

signalChannel
 : "pure:"^
 | "single:"^ index index index index index
 | "multiple:"^ index index index index index index
 ;

signalBool: "bool:"^ index index ;

signalPrevious: "previous:"^ index ;

/******************************/

variableTable: "variables:" num:Int (variable)* "end:"! ;

variable: Int^ Colon! type:index (initialValue)? ;

initialValue: "value:"^ expression ;

/******************************/

taskTable: "tasks:" num:Int (task)* "end:"! ;

task: Int^ Colon! ID pindexn pindexn ;

/******************************/

execTable: "execs:" num:Int (exec)* "end:"! ;

exec: Int^ Colon! index index pindexn pexpressionList ;

/******************************/

actionTable: "actions:" num:Int (action)* "end:"! ;

action
 : Int^ Colon!
 ( "goto:" index
 | "if:" expression
 | "call:" index pindexn pexpressionList
 | "output:" index
 | "reset:" index
 | "combine:" index expression
 | "start:" index
 | "kill:" index
 | "return:" index
 | "suspend:" index
 | "activate:" index
 | "dsz:" index
 | "input:" index
 )
 ;

/******************************/

pauseTable: "pauses:" num:Int (pause)* "end:"! ;

pause: Int^ Colon! ;

/******************************/

statementTable
  : "statements:"^ num:Int
    "startpoint:"! Int
    "goloops:"! Int
    "exitlevels:"! Int
    (statement)*
    "end:"!
  ;

statement
  : Int^ Colon!
  ( "Action:" bindex pindex
  | "Test:" bindex pindex2
  | "Reset:" bindex pindex
  | "Emit:" bindex pindex
  | "Updated:" bindex pindex
  | "Present:" Lbracket! sigExpression Rbracket! pindex2
  | "Goto:" pindex
  | "Goloop:" pindex
  | "Fork:" brindex pindexn
  | "Trapscope:" bindexn brindex pindex
  | "Sigscope:" bindexn brindex pindex
  | "Endscope:" bindex pindexn aindex
  | "Parallel:" bindex pindexn aindex
  | "Exit:" bindex brindex
  | "Pause:" (execs)? pindex aindex Int
  | "Watchdog:" brindex aindex
  | "Stay:" aindex
  | "Run:" instanceRename Lbracket! renamings Rbracket! pindex aindex
  | "Return:" Int
  | "Access:" bindexn pindex
  | "Branchsel:" bindex aindex
  )
  ;

execs: bindexn { #execs = #([EXECS], #execs); } ;

renamings
  : (constantRenamings)?
    (typeRenamings)?
    (functionRenamings)?
    (procedureRenamings)?
    (taskRenamings)?
    (signalRenamings)?
    lastSig:index 
  ;

typeRenamings : "types:"^ idRenaming (Comma! idRenaming)* Semicolon ;

idRenaming: ID Equals! index ;

constantRenamings : "constants:"^ constantRenaming (Comma! constantRenaming)* Semicolon ;

constantRenaming: ID Equals! (constantRef | Constant) ;

functionRenamings : "functions:"^ functionRenaming (Comma! functionRenaming)* Semicolon ;

functionRenaming: ID Equals! (index | String) ;

procedureRenamings : "procedures:"^ idRenaming (Comma! idRenaming)* Semicolon ;

taskRenamings : "tasks:"^ idRenaming (Comma! idRenaming)* Semicolon ;

signalRenamings : "signals:"^ idRenaming (Comma! idRenaming)* Semicolon ;

bindex: Lbracket! index Rbracket! ;
bindexn: Lbracket! commaList Rbracket! ;
pindex: Lparen! index Rparen! ;
pindex2: Lparen! index Comma^ index Rparen! ;
pindexn: Lparen! commaList Rparen! ;
brindex: Lbrace! index Rbrace! ;
brindexn: Lbrace! commaList Rbrace! ;
aindex: Lessthan! index Greaterthan!;
commaList: index (Comma! index)* { #commaList = #([LIST], #commaList); } ;

pexpressionList: Lparen! (expression (Comma^ expression (Comma! expression)* )?)? Rparen! ;

/******************************/

sigExpression
  : index
  | "not:"^ Lparen! sigExpression Rparen!
  | "and:"^ sigExpressionList
  | "or:"^ sigExpressionList
  ;

sigExpressionList
  :  Lparen!
       sigExpression (Comma! sigExpression)*
     Rparen!
  ;

/******************************/

expression: term | functionCall ;

term: atom | constantRef | sensorValue | variableRef ;

atom: Constant ;

constantRef: At^ index ;

sensorValue: Questionmark^ index ;

variableRef: index ;

functionCall: index pexpressionList ;

/******************************/

index: Int | PredefinedIndex | Dash ;

/***********************************************************************/
class IcLexer extends Lexer;
options {
  charVocabulary = '\3'..'\377'; // Handle all 8-bit characters
  k = 2;
}

Colon: ':' ;    

Slash: '/' ;    

Dash: '-';

Constant: '#'! (  '"'! ( ~('"' | '\n') | ('"'! '"'))* '"'!
               | ('-')?
                 ( ('0'..'9')+
		   ( '.' ('0'..'9')* (Exponent)? ('f'|'F')? )?
		 | '.' ('0'..'9')+ (Exponent)? ('f'|'F')?
                 )
               ) ;

protected
Exponent: ('e'|'E') ('+'|'-')? ('0'..'9')+ ;

At: '@';

Questionmark: '?';

Lparen: '(';
Rparen: ')';

Lbracket: '[';
Rbracket: ']';

Lbrace: '{';
Rbrace: '}';

Lessthan: '<' ;
Greaterthan: '>' ;

Comma: ',';

Semicolon: ';' ;

Equals: '=' ;

ID: ('a'..'z'|'A'..'Z'|'_')('a'..'z'|'A'..'Z'|'0'..'9'|'_')* (':')? ;

Int: ('0'..'9')+ ;

PredefinedIndex: '$' ('0'..'9')+ ;

String:  '"'! ( ~('"' | '\n') | ('"'! '"'))* '"'! ;

Comment: "--" (~'\n')* '\n' { newline(); $setType(Token.SKIP); } ;

Pragma: '%' ( ~('%' | '\n') )* '%' { $setType(Token.SKIP); } ;

Whitespace: (' ' | '\t' | '\f')+ { $setType(Token.SKIP); } ;

Newline: '\n' { newline(); $setType(Token.SKIP); } ;

/***********************************************************************/

class IcWalker extends TreeParser;

{

    String[] signals;

    public void carc(AST src, AST dest) {
	System.out.println("S" + src.getText() + " -> " +
			   "S" + dest.getText());
    }

    public void carcl(AST src, AST dest, String label) {
	System.out.println("S" + src.getText() + " -> " +
			   "S" + dest.getText() + " [label=\"" + label + "\"]");
    }

    public void rarc(AST src, AST dest) {
	System.out.println("S" + src.getText() + " -> " +
			   "S" + dest.getText() + " [ color=blue style=bold]" );
    }

    public void exitarc(AST src, AST dest) {
	System.out.println("S" + src.getText() + " -> " +
			   "S" + dest.getText() + " [ style=dotted dir=back ]" );
    }
}

file: (module)+ ;

module
 : #( "module:" module:ID
      {
	  System.out.println("digraph ic {\n" +
			     "  size=\"7.5,10\";\n" +
			     "  margin=0.5;\n" +
			     "  ratio=\"fill\";");
	  System.out.println("  label=\"" + module.getText() + "\";");
      }
      (~("statements:" | "signals:" ))*
      signalTable
      (~("statements:" | "signals:" ))*
      statementTable
      {
	  System.out.println("}");
      }
    )
 ;

signalTable
  : #( "signals:"
       num:Int { signals = new String[Integer.parseInt(num.getText(), 10)]; }
       (signal)*
     )
  ;


signal
  : #( num:Int
       { String name; }
       name=signalKind
       { signals[Integer.parseInt(num.getText(),10)] = name; }      
     )
  ;

signalKind returns [String name]
  { name = ""; }
  : #("input:" id1:ID { name = id1.getText(); } )
  | #("return:" id2:ID { name = id2.getText(); } )
  | #("output:" id3:ID { name = id3.getText(); } )
  | #("inputoutput:" id4:ID { name = id4.getText(); } )
  | #("sensor:" id5:ID { name = id5.getText(); } )
  | #("local:" id6:ID { name = id6.getText(); } )
  | #("trap:" id7:ID { name = id7.getText(); } )
  ;



statementTable
  : #( "statements:"
	 num:Int
	 startpoint:Int
	 {
	     System.out.println("Start [style=filled color=green]");
	     System.out.println("Start -> S" + startpoint.getText());
	     System.out.println("{ rank=min; Start; }");
	 }
	 goloops:Int
	 exitlevels: Int
	 (statement)*
     )
  ;

statement
  : #( num:Int
       { System.out.print( "S" + num.getText() +
			   " [ label=\"" + num.getText()); }
       ( "Action:" { System.out.println( "\" shape=box ]" ); }
         index
	 cont[num]
       | "Test:" { System.out.println( "\" shape=diamond]" ); }
         index
         #(Comma cont[num] cont[num])
       | "Reset:" { System.out.println( "\" shape=house orientation=90]" ); }
         index
	 cont[num]
       | "Emit:"
	 sig:index  
         { System.out.println( " " + signals[Integer.parseInt(sig.getText(),10)] +
			       "\" shape=house orientation=-90]" ); }  
         cont[num]
       | "Updated:" { System.out.println( "\" shape=house]" ); }
         index
         cont[num]
       | "Present:"
         { System.out.print(" "); }
         sigExpression
         { System.out.println( "\" shape=diamond]" ); }
         #(Comma cont[num] cont[num])
       | "Goto:" { System.out.println( "\" shape=circle]" ); }
         cont[num]
       | "Goloop:" { System.out.println( "\" shape=circle]" ); }
         cont[num]
       | "Fork:" { System.out.println( "\" shape=triangle]" ); }
          fes:index { System.out.println("{ rank=same; S" + num.getText() +
					  "; S" + fes.getText() + "; }"); }
	 #(LIST (cont[num])* )
       | "Trapscope:"
          sigindexn
          { System.out.println( "\" shape=triangle style=filled color=orange]" ); }
          tses:index { System.out.println("{ rank=same; S" + num.getText() +
					  "; S" + tses.getText() + "; }"); }
          cont[num]
       | "Sigscope:"
  	 sigindexn
         { System.out.println( "\" shape=triangle style=filled color=pink]" ); }
          sses:index { System.out.println("{ rank=same; S" + num.getText() +
					  "; S" + sses.getText() + "; }"); }
          cont[num]
       | "Endscope:" { System.out.println( "\" shape=triangle style=filled color=brown fontcolor=white]" ); }
          index
          parallelConts[num]
 	  recon[num]
       | "Parallel:" index
         {
	     System.out.println( "\" shape=triangle style=filled color=black fontcolor=white]" );
	 }
	  parallelConts[num]
	  recon[num]
       | "Exit:"
          level:index
         {
	     System.out.println( "@" + level.getText() + "\" shape=ellipse]" );
	 }
         exitpar:index { exitarc(exitpar, num); }
       | "Pause:" { System.out.println( "\" shape=box style=filled color=black fontcolor=white]" ); }
         ( #(EXECS (index)*) )?
         cont[num]
         recon[num]
	 Int

       | "Watchdog:" { System.out.println( "\" shape=diamond style=filled color=black fontcolor=white]" ); }
	 cont[num]
	 recon[num]         
       | "Stay:" { System.out.println( "\" shape=box style=filled color=black fontcolor=white]" ); }
         recon[num]
	 //       | "Run:" instanceRename renamings index index
       | "Return:" Int
         { System.out.println( "\" shape=circle style=filled color=red fontcolor=white]" );
	 System.out.println( "{ rank=min; S" + num.getText() + "; }" ); }
       | "Access:"
         sigindexn
         index
         { System.out.println( "\" shape=house orientation=180]" ); }
       | "Branchsel:" { System.out.println( "\" shape=diamond style=filled color=black fontcolor=white]" ); }
         index
	 recon[num]         
       )
     )
  ;

parallelConts [AST num]
  : 	 #(LIST labeledcont[num,"0"] { int exitlevel = 2; }
           ( { String lab = java.lang.String.valueOf(exitlevel); }
             labeledcont[num,lab]
             { exitlevel++; }
           )* )
  ;

cont [AST src]
    : dest:index { carc(src,dest); } ;

labeledcont [AST src, String label]
    : dest:index { carcl(src,dest,label); } ;

recon [AST dest]
    : src:index { rarc(src,dest); } ;

sigExpression
  : sig:index
    { System.out.print( signals[Integer.parseInt(sig.getText(),10)] ); }
  | #("not:"
    { System.out.print( "not (" ); }
      sigExpression )
    { System.out.print( ")" ); }
  | #("and:"
      { System.out.print( "(" ); }
	 sigExpression
         ( { System.out.print(" and "); }
           sigExpression )*
     { System.out.print( ")" ); }
     )
  | #("or:"
      { System.out.print( "(" ); }
	 sigExpression
         ( { System.out.print(" or "); }
           sigExpression )*
     { System.out.print( ")" ); }
     )
  ;

index: Int | PredefinedIndex | Dash ;

sigindexn
 : #(LIST (sig:Int
          { System.out.print(" " + signals[Integer.parseInt(sig.getText(),10)] ); }
          )*
    )
 ;
