#!/bin/nawk -f

# Converts a .aux file from POLIS into an equivalent interconnection of
# modules in Esterel

# Usage:
#
# aux2strl *.strl mca200_sim.aux mca200_sim.aux
#
# Reads the .strl files first to get interface information,
# then read the .aux file twice to resove forward references.

BEGIN {
  FS = "[ \\(\\),:/\\[\\]]+";     
}


##############################
#
# This section parses Esterel programs looking for IO port types
#

$1 == "module" && NF == 3 {
  modname = $2;
}

$1 == "input" && !modstarted {
  while ( getline ) {
    signal[ modname, $2 ] = 1;
    if ( $0 ~ /\(/ ) {
      iotype[ modname, $2 ] = $3;
#      print "iotype[" modname "," $2 "] = " $3;
    }
    if ( $0 ~ /;/ ) {
      break
    }
  }
}

$1 == "output" && !modstarted {
  while ( getline ) {
    signal[ modname, $2 ] = 1;
    if ( $0 ~ /\(/ ) {
      iotype[ modname, $2 ] = $3;
#      print "iotype[" modname "," $2 "] = " $3;
    }
    if ( $0 ~ /;/ ) {
      break
    }
  }
}

##############################
#
# This section looks for POLIS hierarchical net definitions
#

$1 == "define" {
  constant[ $2 ] = $3;
  print "#define " $2 " " $3 > "consts.h"
}

$1 == "net" {
  modstarted = 1;
  modname = $2;
  started = 0;
  runs = "";
  ninputs = 0;
  noutputs = 0;
}

$1 == "input" && modstarted {
  gatherline();
  gsub( /input */, "", line );
  ninputs = split( line, input, /[ ,;]*/ );
  for ( i = 1 ; i < ninputs ; i++ ) {
#    print "  input " input[i] ";";
    io[ input[i] ] = 1
  }
}

$1 == "output" && modstarted {
  gatherline();
  gsub( /output */, "", line );
  noutputs = split( line, output, /[ ,;]*/ );
  for ( i = 1 ; i < noutputs ; i++ ) {
#    print "  output " output[i] ";";
    io[ output[i] ] = 1
  }
}

$1 == "." && modstarted {
  # End-of-module

  print "module " modname " : ";
  if ( ninputs > 0 ) {
    print "  input";
      for ( i = 1 ; i < ninputs ; i++ ) {
	sig = input[i];
	printf("    %s", sig );
	if ( sig in type ) {
	  printf("(%s)", type[sig] );
	  iotype[ modname, sig ] = type[sig];
	}
	if ( i < ninputs - 1 ) {
	  print ",";
	} else {
	  print ";";
	}
      }
  }
  if ( noutputs > 0 ) {
    print "  output";
      for ( i = 1 ; i < noutputs ; i++ ) {
	sig = output[i];
	printf("    %s", sig );
	if ( sig in type ) {
	  printf("(%s)", type[sig] );
	  iotype[ modname, sig ] = type[sig];
	}
	if ( i < noutputs - 1 ) {
	  print ",";
	} else {
	  print ";";
	}
      }
  }
  for ( c in usedconstant ) {
#    print "  constant " c " = " usedconstant[c] " : integer;"
    print "  constant " c " : integer;"
  }
  print "  signal";
  for (sig in local) {
      printf ("    %s", sig );
      if ( sig in type ) {
         printf("(%s)", type[ sig ] );
      }
      print ",";
      delete local[sig]
    }
    print "    ignored"
    print "  in"
    print runs;
    print "  end signal"
    print "end module % " modname;
    print ""
    print "";
    for ( s in io ) {
      delete io[s]
    }
    for ( t in type ) {
      delete type[t]
    }
    for ( c in usedconstant ) {
      delete usedconstant[c]
    }
  modstarted = 0;
}

$1 == "module" && modstarted {
  if ( started ) { 
    runs = runs "  ||\n";    
  } else {
    started = 1;
  }
  gatherline();
  gsub( /%[^;]*/, "", line );
  gsub( /;.*$/, "", line );
  n = split( line, call, /[ ,\[\]\/]*/ );
  module = call[2];
  runs = runs "  run " module " \[\n";  
# The strange start here is for the few cases where the instance
# has a name (there's a different number of parameters then
  needsemi = 0;
  for ( i = 4 - n % 2 ; i < n ; i+= 2 ) {
    actualparam = call[i+1];
    formalparam = call[i];
    if (needsemi) {
      runs = runs ";\n"
    }
    if ( actualparam in constant ) {
      runs = runs "    constant " actualparam " / " formalparam
      usedconstant[ actualparam ] = constant[ actualparam ]
    } else {
      if ( (module, formalparam) in iotype ) {
        type[ actualparam ] = iotype[ module, formalparam ]
      }
      if ( !(actualparam in io) ) {
#      print "local: " actualparam
        local[actualparam] = 1;
      }
      runs = runs "    signal " actualparam " / " formalparam
    }
    needsemi = 1;
  }
  runs = runs "  ]\n"
}

function gatherline () {
  line = $0;
  while ( line !~ /;/ ) {
    getline;
    line = line $0;
  }
}

