// -*- C++ -*-

// SPARC code-generation routines for the Esterel compiler

// Stephen Edwards

#include "est.h"

/*

The SPARC has thirty-two integer registers:

Global registers:
%0  %g0		Always zero
%1  %g1
%2  %g2
%3  %g3
%4  %g4
%5  %g5
%6  %g6
%7  %g7

Outgoing parameter registers:
%8  %o0		Return value to calling C routine
		src1 for multiply/divide/mod, result for same
%9  %o1		src2 for multiply/divide/mod
%10 %o2
%11 %o3
%12 %o4
%13 %o5
%14 %o6		Stack pointer
%15 %o7

Local Registers:
%16 %l0		Used in expression evaluation	
%17 %l1
%18 %l2
%19 %l3
%20 %l4
%21 %l5
%22 %l6
%23 %l7

Incoming parameter registers:
%24 %i0
%25 %i1
%26 %i2		New PC returned by subprocesses
%27 %i3		New potential set calculation routine
%28 %i4		Returned status (0=terminated, 1=halted, -1=wait)
%29 %i5		Base address of arrays
%30 %i6		Frame pointer
%31 %i7		Return address

The runtime system uses the following externally-visible data:

int n_pc;	        // number of program counters
int n_var;		// number of variables
int n_sig;		// number of signals (includes valued and pure)
int n_sigv;		// number of valued signals
int n_cnt;		// number of counters
int n_exc;		// number of exceptions (includes valued and pure)
int n_excv;		// number of valued exceptions

void (*PC[ n_pc ])();	// Program counters
void (*PO[ n_pc ])();	// Potential-set calculators
char H[ n_pc ];		// Used to flag halted try instructions

int V[ n_var ];		// value of variables

char S[ n_sig ];	// 0 if signal unknown, 1 if present, -1 if absent
char P[ n_sig ];	// 0 if signal has the potential to be emitted

int SV[ n_sigv ];	// signal values

int C[ n_cnt ];		// counter values

char E[ n_exc ];	// 1 if exception raised, 0 otherwise

int EV[ n_excv ];	// exception values

typedef struct {
  char * name;
  char * sigpres;
  int * sigval;
} sig_t;

sig_t sigs[];

 */

#define ASM(x) void x::asmnode( ostream & s )

// Always zero
#define ZERO		"%g0"

// parameters and results for the multiply/divide/mod operators
#define PARAM1		"%o0"
#define PARAM2		"%o1"
#define RESULT		"%o0"

// registers for general temporary use
#define TMPREG		"%o0"
#define TMPREG2		"%o1"

// Program counter for the next call of the subprocess
#define NEWPCREG	"%i2"

// Program counter for the potential set calculator routine
#define NEWPOREG	"%i3"

// Returned status of the subprocess
#define STATUSREG	"%i4"

#define STATUSWAIT	"-1"
#define STATUSHALT	"1"

#define EXCEPRAISED	"1"

#define SIGPRESENT	"1"

// Base address for the arrays
#define BASEREG		"%i5"

#define ARRAY(name,offset) "[" BASEREG "+_" name "-_PC-0x1000+" << offset << "]"

#define PC(pc) ARRAY("PC",pc * 4)
#define PO(pc) ARRAY("PO",pc * 4)
#define H(pc) ARRAY("H",pc)
#define SIGNALPO(s) ARRAY("P",s)
#define SIGNALPRES(s) ARRAY("S",s)

#define PCD(pc) ARRAY("PCD",pc * 4)

// Local label for intra-instruction miscellany

#define LOCALLABEL(l) "LL" << l

// Label for an instruction---requires a process number and an index

#define INSTRUCTIONLABEL(p,i) "P" << (p) << "I" << (i)

// Label for returning processes

#define PRETURNLABEL(p) "PR" << (p)

// Label for returning potential-set calculators

#define PORETURNLABEL(p) "PO" << (p)

// Zero-, One-, two-, and three-argument instructions

#define I0(o) \
'\t' << o << '\n'

#define I1(o,a) \
'\t' << o << '\t' << a << '\n'

#define I2(o,a1,a2) \
'\t' << o << '\t' << a1 << ',' << a2 << '\n'

#define I3(o,a1,a2,a3) \
'\t' << o << '\t' << a1 << ',' << a2 << ',' << a3 << '\n'

#define LABEL(l) \
l << ":\n"

#define GLOBALLAB( name ) \
I1( ".align" , "4" ) << \
I1( ".global" , name ) << \
LABEL( name )

#define INIT_CHAR_ARRAY( name, size ) \
GLOBALLAB(name) << \
I1( ".skip" , ((size) ? (size) : 1) )

#define INIT_INT_ARRAY( name, size ) \
GLOBALLAB(name) << \
I1( ".skip" , ((size) ? (size)*4 : 4) )

#define INIT_INT( name, value ) \
GLOBALLAB(name) << \
I1( ".word" , (value) )

//////////////////////
//
// segment changing routine
//
// Usage:  s << segment( TEXT );
//

enum eseg { UNKNOWN, TEXT, DATA };

OMANIP1( segment, eseg ) {
  static eseg curseg = UNKNOWN;

  if ( i != curseg )
    switch (curseg = i) {

    case TEXT:
      s << I1( ".seg" , "\"text\"" );
      break;

    case DATA:
      s << I1( ".seg" , "\"data\"" );
      break;

    default:
      s << "! Unknown segment requested!\n";
      break;

    }

  return s;
}

//////////////////////////////
//
// Assembly output routines for class sd
//

OMANIP1( sdval, sd * ) {

  switch ( i->sdtypeof() ) {

    // Local register

  case SDreg:
    if ( i->argof() > 7 )
      yyerror("More than 8 local registers used!");
    s << "%l" << i->argof() ;
    break;

    // variable value

  case SDvar:
    s << ARRAY("V",i->argof() * 4);
    break;

    // signal value

  case SDsig:
    s << ARRAY("SV",i->arg2of() * 4);
    break;

    // counter value

  case SDcntr:
    s << ARRAY("C",i->argof() * 4);
    break;

    // exception value

  case SDexc:
    s << ARRAY("EV",i->arg2of() * 4);
    break;

    // integer constant

  case SDconst:
    s << i->argof();
    break;

  default:
    break;
  };

  return s;
}

OMANIP1( sdpres, sd * ) {

  switch ( i->sdtypeof() ) {
    
    // signal presence

  case SDsig:
    s << SIGNALPRES(i->argof());
    break;

    // exception presence

  case SDexc:
    s << ARRAY("E",i->argof());
    break;

  default:
    break;
  };

  return s;
}


//////////////////////////////
//
// Routine for keeping track of local labels
//
// Usage:
//
// int l = newlocal();
//
// s << "jmp" << LOCALLABEL(l) << NL;
//
// s << LOCALLABEL(l) << ":\n";

int newlocal()
{
  static locallab = 0;

  return locallab++;
}


//////////////////////////////
//
// The main assemble routine
//

void assemble( ostream & s, amprocess & p )
{
  int c;

  s << "! Compiled by est\n\n";

  s << "! Externally-visible arrays\n\n";

  int n_pc = p.n_pcs();		// get the number of program counters

  int n_var = TICKsd->n_variables();
  int n_sig = TICKsd->n_signals();
  int n_sigv = TICKsd->n_valuedsignals();
  int n_cnt = TICKsd->n_counters();
  int n_exc = TICKsd->n_exceptions();
  int n_excv = TICKsd->n_valuedexceptions();

  s << segment(DATA);

  s << INIT_INT_ARRAY( "_PC", n_pc );

  s << INIT_INT_ARRAY( "_PO", n_pc );

  s << INIT_CHAR_ARRAY( "_H", n_pc );

  s << INIT_INT_ARRAY( "_V", n_var );

  s << INIT_CHAR_ARRAY( "_S", n_sig );

  s << INIT_CHAR_ARRAY( "_P", n_sig );

  s << INIT_INT_ARRAY( "_SV", n_sigv );

  s << INIT_INT_ARRAY( "_C", n_cnt );

  s << INIT_CHAR_ARRAY( "_E", n_exc );

  s << INIT_INT_ARRAY( "_EV", n_excv );

  if ( asmdebug )
    s << INIT_INT_ARRAY( "_PCD", n_pc );

  s << "\n! Externally-visible variables\n\n";

  s << INIT_INT( "_n_pc", n_pc );
  s << INIT_INT( "_n_var", n_var );
  s << INIT_INT( "_n_sig", n_sig );
  s << INIT_INT( "_n_sigv", n_sigv );
  s << INIT_INT( "_n_cnt", n_cnt );
  s << INIT_INT( "_n_exc", n_exc );
  s << INIT_INT( "_n_excv", n_excv );

  s << segment(TEXT);

  // The inittick() routine---initialize those things which only we can
  // for the very first tick (program counters, etc..)

  s << I1( ".align" , "4" )
    << I1( ".global" , "_inittick" )
    << I1( ".proc" , "04" )
    << LABEL( "_inittick" );

  s << I2( "set" , INSTRUCTIONLABEL(0,0) , TMPREG )
    << I2( "set", "_PC+0x1000" , BASEREG )
    << I2( "st" , TMPREG , PC(0) );

  s << I0( "retl" )
    << I0( "nop" );
  
  // The int tick() routine---the main simulation loop

  s << I1( ".align" , "4" )
    << I1( ".global" , "_tick" )
    << I1( ".proc" , "04" )
    << LABEL( "_tick" )

    << I3( "save" , "%sp" , "-112" , "%sp" )

    << I2( "set" , "_PC+0x1000" , BASEREG )
    << I2( "mov" , ZERO , STATUSREG )

    // Call process zero

    << I2( "ld" , PC(0) , TMPREG )
    << I2( "call" , TMPREG , "0" )
    << I0( "nop" )

    // Store the new program counter

    << LABEL( PRETURNLABEL( 0 ) )
    << I2( "st" , NEWPCREG , PC( 0 ) )

    // Call the potential-set calculator

    << I2( "call" , NEWPOREG , "0" )
    << I0( "nop" )

    << LABEL( PORETURNLABEL( 0 ) )
    << I0( "ret" )

    // return the returned status

    << I3( "restore" , STATUSREG , ZERO , RESULT );

  s << NL "! Subprocess code begins" NL NL;

  for ( int i = 0 ; i < p.n_processes() ; i++ )
    p.process(i)->assemble(s);

  s << NL "! Subprocess code ends" NL NL;

}

//////////////////////////////
//
// assembler for an amprocess
//

void amprocess::assemble( ostream & s )
{

  printheader(s);
  s << "!" NL;

  for( int i = 0 ; i < size() ; i++ ) {
    s << LABEL( INSTRUCTIONLABEL(number,i) );
    s << *((*this)[i]);
    (*this)[i]->asmnode(s); s << '\n';
  }

}

//////////////////////////////
//
// asmnode() defined for each member of the amnode class
//

void
debugreturn(ostream & s, int process, int instruction, int programcounter )
{
  if ( asmdebug ) {
    s << I2( "set" , process << "*0x10000+" << instruction , TMPREG2 )
      << I2( "st" , TMPREG2 , PCD(programcounter) );
  }
}

ASM(amnode) {
}

ASM(AMexit) {
  if ( exceptionof() ) {

    // If there is a register, store its value as the value of the exception

    if ( registerof() )
      s << I2( "st" , sdval(registerof()) , sdval(exceptionof()) );

    // Since there is an exception, raise it

    s << I2( "mov" , EXCEPRAISED , TMPREG )
      << I2( "stb" , TMPREG , sdpres(exceptionof()) );
  }

  int l = newlocal();

  s << LABEL( LOCALLABEL(l) );

  // set the return PC to branch back here

  s << I2( "set" , LOCALLABEL(l) , NEWPCREG )

    // set the return PO to branch straight to the return

    << I2( "set" , PORETURNLABEL( processnumber() ) , NEWPOREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

  // branch to the calling try

  s << I1( "ba"  , PRETURNLABEL( processnumber() ) );

  // if this exit was with an exception, actually return halt
  // the appropriate try will catch the exit and terminate the process

  if ( exceptionof() )
    s << I3( "or" , STATUSREG, STATUSHALT, STATUSREG );
  else
    s << I0( "nop" );

}

ASM(AMhalt) {

  int l = newlocal();
  s << LABEL( LOCALLABEL(l) )

    // the the return PC to branch back here
    << I2( "set" , LOCALLABEL(l), NEWPCREG )

    // set the return PO to branch to the null potential-set calculator
    << I2( "set" , PORETURNLABEL( processnumber() )
	   , NEWPOREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

    // branch to the calling try
  s << I1( "ba" , PRETURNLABEL( processnumber() ) )

    // return a "halted" status
    << I3( "or" , STATUSREG , STATUSHALT , STATUSREG );

}

ASM(AMassign) {

  int l;

  sdtype src1type = src1of()->sdtypeof();
  sdtype desttype = destof()->sdtypeof();
  sdtype src2type = src2of() ? src2of()->sdtypeof() : SDnone;

  switch ( opof() ) {

    // When we are doing register-register moves or constant loads,
    // the "mov" opcode is the one of choice
    // otherwise, "ld" or "st" is appropriate, depending on the destination

  case amASSIGN:
    if ( src1type == SDconst ||
	 src1type == SDreg && desttype == SDreg )
      s << I2( "mov" , sdval(src1of()) , sdval(destof()) );
    else if ( src1type == SDreg )
      s << I2( "st" , sdval(src1of()) , sdval(destof()) );
    else if ( desttype == SDreg )
      s << I2( "ld" , sdval(src1of()) , sdval(destof()) );
    break;

    // unary operators

  case amNEG:
    s << I3( "sub" , ZERO , sdval(src1of()) , sdval(destof()) );
    break;

  case amNOT:
    s << I3( "xnor" , ZERO , sdval(src1of()) , sdval(destof()) );
    break;

  case amDEC:
    s << I3( "sub" , sdval(src1of()) , "1" , sdval(destof()) );
    break;

    // binary operators

  case amAND:
    s << I3( "and" , sdval(src1of()) , sdval(src2of()) , sdval(destof()) );
    break;

  case amOR:
    s << I3( "or" , sdval(src1of()) , sdval(src2of()) , sdval(destof()) );
    break;

    // The binary boolean operators use a trick--- ,a is specified for
    // the branch so when it is not taken, the following instruction is
    // annulled---not executed

  case amEQ:
    l = newlocal();
    s << TAB "cmp" TAB << sdval(src1of()) << ',' << sdval(src2of()) << '\n'
      << TAB "be,a" TAB << LOCALLABEL(l) << '\n'
      << TAB "mov" TAB "1," << sdval(destof()) << '\n'
      << TAB "mov" TAB "0," << sdval(destof()) << '\n'
      << LOCALLABEL(l) << ':';
    break;

  case amNE:
    l = newlocal();
    s << TAB "cmp" TAB << sdval(src1of()) << ',' << sdval(src2of()) << '\n'
      << TAB "bne,a" TAB << LOCALLABEL(l) << '\n'
      << TAB "mov" TAB "1," << sdval(destof()) << '\n'
      << TAB "mov" TAB "0," << sdval(destof()) << '\n'
      << LOCALLABEL(l) << ':';
    break;

  case amLT:
    l = newlocal();
    s << TAB "cmp" TAB << sdval(src1of()) << ',' << sdval(src2of()) << '\n'
      << TAB "bl,a" TAB << LOCALLABEL(l) << '\n'
      << TAB "mov" TAB "1," << sdval(destof()) << '\n'
      << TAB "mov" TAB "0," << sdval(destof()) << '\n'
      << LOCALLABEL(l) << ':';
    break;

  case amLE:
    l = newlocal();
    s << TAB "cmp" TAB << sdval(src1of()) << ',' << sdval(src2of()) << '\n'
      << TAB "ble,a" TAB << LOCALLABEL(l) << '\n'
      << TAB "mov" TAB "1," << sdval(destof()) << '\n'
      << TAB "mov" TAB "0," << sdval(destof()) << '\n'
      << LOCALLABEL(l) << ':';
    break;

  case amADD:
    s << TAB "add" TAB << sdval(src1of()) << ',' << sdval(src2of()) << ','
      << sdval(destof());
    break;

  case amSUB:
    s << TAB "sub" TAB << sdval(src1of()) << ',' << sdval(src2of()) << ','
      << sdval(destof());
    break;

  case amMUL:
    s << TAB "mov" TAB << sdval(src1of()) << "," PARAM1		NL
      << TAB "mov" TAB << sdval(src2of()) << "," PARAM2		NL
      << TAB "call" TAB ".mul,2"				NL
      << TAB "nop"						NL
      << TAB "mov" TAB RESULT "," << sdval(destof());
    break;

  case amDIV:
    s << TAB "mov" TAB << sdval(src1of()) << "," PARAM1 	NL
      << TAB "mov" TAB << sdval(src2of()) << "," PARAM2		NL
      << TAB "call" TAB ".div,2"				NL
      << TAB "nop"						NL
      << TAB "mov" TAB RESULT "," << sdval(destof());
    break;

  case amMOD:
    s << TAB "mov" TAB << sdval(src1of()) << "," PARAM1 	NL
      << TAB "mov" TAB << sdval(src2of()) << "," PARAM2		NL
      << TAB "call" TAB ".rem,2"				NL
      << TAB "nop"						NL
      << TAB "mov" TAB RESULT "," << sdval(destof());
    break;

  default:
    break;

  }
}

ASM(AMrequire) {
  
  // prepare a set for calculating the potentials at this node

  setc poten( TICKsd->n_signals() );

  potentialof(poten);

  s << "! Potential signals:" << poten << NL;

  int i;

  int missing = newlocal();

  // check all of the signals

  for ( i = n_signals() ; --i >= 0 ; ) {

    s << I2( "ldsb" , sdpres(signal(i)) , TMPREG )
      << I1( "tst" , TMPREG );

    // if this is not the last signal to check,
    // branch to the "missing" handler if the signal
    // otherwise, branch to next instruction if the signal is known

    if (i)
      s << I1( "be" , LOCALLABEL(missing) );
    else
      s << I1( "bne" , INSTRUCTIONLABEL(processnumber(),indexof()+1) );

  }

  s << I0( "nop" );

  int potential = newlocal();

  // We were missing some signals,
  // So set the return PC to the "require,"
  // the return PO to the potential routine below,
  // and the return status to

  s << LABEL( LOCALLABEL(missing) )
    << I2( "set" , INSTRUCTIONLABEL( processnumber(),indexof() ) , NEWPCREG )
    << I2( "set" , LOCALLABEL( potential ) , NEWPOREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

  s << I1( "ba" , PRETURNLABEL( processnumber() ) )
    << I3( "or" , STATUSREG , STATUSWAIT , STATUSREG );


  s << LABEL( LOCALLABEL(potential) );

  for ( i = TICKsd->n_signals() ; --i >= 0 ; )
    if ( poten.memberof(i) )
      s << I2( "stb" , ZERO , SIGNALPO(i) );

  s << I1( "ba" , PORETURNLABEL( processnumber() ) )
    << I0( "nop" );

}

ASM(AMemit) {

  // if there is an associated register, store it into the signal value

  if ( registerof() )
    s << TAB	"st" TAB	<< sdval(registerof()) << ","
      << sdval(signalof()) << NL;

  // set the signal presence

  s << I2( "mov" , SIGPRESENT , TMPREG )
    << I2( "stb" , TMPREG , sdpres(signalof()) );
}

ASM(AMbranch) {

  if ( testof() ) {

    // conditional branches

    switch( testof()->sdtypeof() ) {

    case SDreg:
      s << I1( "tst" , sdval(testof()) );
      break;

    case SDvar: case SDcntr:
      s << TAB 	"ld" TAB << sdval(testof()) << "," TMPREG NL
      	<< TAB  "tst" TAB TMPREG NL;
      break;

    case SDsig: case SDexc:
      s << TAB 	"ldsb" TAB << sdpres(testof()) << "," TMPREG NL
      	<< TAB  "tst" TAB TMPREG NL;
      break;
    }

    if ( testof()->sdtypeof() == SDsig ) {

      // Signals are positive when present, negative when absent

      if ( senseof() )
	s << TAB "bneg";
      else
	s << TAB "bpos";

    } else {

      // All others are either zero or non-zero

      if ( senseof() )
	s << TAB "be";
      else
	s << TAB "bne";

    }

  } else {

    // unconditional branch

    s << TAB	"ba";

  }

  s << TAB <<	INSTRUCTIONLABEL(processnumber(),targetof()) << NL
    << TAB	"nop" NL;

}

ASM(AMtry) {

  int i;

  int executeall = newlocal();
  int checksignals = newlocal();
  int missingsignals = newlocal();
  int notdonethisinst = newlocal();
  int notwaiting = newlocal();
  int calcpoten = newlocal();
  int absentsignal;

  //
  // On first entry, initialize all PCs to the appropriate values
  //

  s << "! Initializing all program counters\n";

  for ( i = n_subprocesses() ; --i >= 0 ; )
    s << I2( "set" , INSTRUCTIONLABEL( subprocess(i)->processnumber() , 0 ) ,
	     TMPREG )
      << I2( "st" , TMPREG , PC( subprocess(i)->pcof() ) );

  s << "! Lowering all exceptions\n";

  // Lower all relevant exceptions

  for ( i = n_handles() ; --i >= 0 ; )
    s << I2( "stb" , ZERO , sdpres( handle(i)->exceptionof() ) );
  
  // Execute all the subprocesses without checking any of the watched
  // signals

  s << I1( "ba" , LOCALLABEL( executeall ) )
    << I0( "nop" );

  //
  // On later entries, first check if the halt flag is set, indicating
  // that we are finished for this instant
  //

  s << LABEL( LOCALLABEL( checksignals ) );

  s << "! Checking the halted status of this try\n";

  s << I2( "ldsb" , H(pcof()) , TMPREG )
    << I1( "tst" , TMPREG )
    << I1( "be" , LOCALLABEL( notdonethisinst ) )
    << I0( "nop" );

  // We are halted for this instant, so return that information
  // The potential set calculator will return empty when the
  // halted flag is set, but may be non-empty when the halted flag is cleared

  s << "! Returning the halted status\n";

  s << I2( "set" , LOCALLABEL( checksignals ) , NEWPCREG )
    << I2( "set" , LOCALLABEL( calcpoten ) , NEWPOREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

  s << I1( "ba" , PRETURNLABEL( processnumber() ) )
    << I3( "or" , STATUSREG , STATUSHALT , STATUSREG );

  s << LABEL( LOCALLABEL( notdonethisinst ) );

  s << "! Checking watched signals\n";

  // Check all the watched signals
  // If it is unknown, we must return with the "waiting" status
  // If it is present, we must decrement the counter and/or branch

  for ( i = n_watches() ; --i >= 0 ; ) {

    absentsignal = newlocal();

    s << I2( "ldsb" , sdpres(watch(i)->signalof()) , TMPREG )
      << I1( "tst" , TMPREG )

      << I1( "be" , LOCALLABEL( missingsignals ) )
      << I0( "nop" )

      << I1( "bneg" , LOCALLABEL( absentsignal ) )
      << I0( "nop" );

    // If the watch has a counter, decrement it, saving the new value
    // if it is still positive, we are not done with this watch clause

    if ( watch(i)->counterof() ) {
      s << I2( "ld" , sdval( watch(i)->counterof() ) , TMPREG )
	<< I3( "subcc" , TMPREG , "1" , TMPREG )
	<< I1( "bg" , LOCALLABEL( absentsignal ) )
	<< I2( "st" , TMPREG , sdval( watch(i)->counterof() ) );
    }

    // If we are here, the occurrence has elapsed, so branch directly to
    // the handler

    s << I1( "ba" ,
	     INSTRUCTIONLABEL( processnumber() , watch(i)->targetof() ) )
      << I0( "nop" );

    s << LABEL( LOCALLABEL( absentsignal ));
  }

  //
  // Execute each of the subprocesses
  //

  s << LABEL( LOCALLABEL( executeall ) );

  s << "! Executing all subprocesses\n";

  // store the old status
  s << I2( "stb" , STATUSREG , H(pcof()) );

  // clear the new status
  s << I2( "mov" , ZERO , STATUSREG );

  // Call each subprocess
  
  for ( i = n_subprocesses() ; --i >= 0 ; ) {
    s << I2( "ld" , PC(subprocess(i)->pcof()) , TMPREG )
      << I2( "call" , TMPREG , "0" )
      << I0( "nop" )

      // This is where the subprocess will return
      // Save the returned PC and PO

      << LABEL( PRETURNLABEL( subprocess(i)->processnumber() ) )
      << I2( "st" , NEWPCREG , PC( subprocess(i)->pcof() ) )
      << I2( "st" , NEWPOREG , PO( subprocess(i)->pcof() ) );
  }

  // If the subprocesses returned "waiting for signals" then we will too

  s << I3( "subcc" , STATUSREG , ZERO , TMPREG )
    << I1( "bpos" , LOCALLABEL( notwaiting ) )
    << I0( "nop" );
  
  // Next time, execute all subprocesses again and
  // use our potential-set calculator (return with the "waiting" status)

  s << "! Return with the waiting status\n";

  s << I2( "set" , LOCALLABEL( executeall ) , NEWPCREG )
    << I2( "set" , LOCALLABEL( calcpoten ) , NEWPOREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

  // Clear the halted status for the potential set calculator since
  // we are not finished for the instant.

  s << I1( "ba" , PRETURNLABEL( processnumber() ) )
    << I2( "stb" , ZERO, H( pcof() ) );

  //
  // The subprocesses were not waiting for any signals,
  // so check the other conditions
  //
  // At this point, the status returned by the subprocesses is in TMPREG
  //

  s << LABEL( LOCALLABEL( notwaiting ) );

  // Restore the old status

  s << I2( "ldsb" , H(pcof()), STATUSREG );

  // Check all of the exceptions, branching as necessary

  s << "! Checking all exceptions\n";

  for ( i = n_handles() ; --i >= 0 ; ) {

    s << I2( "ldsb" , sdpres( handle(i)->exceptionof() ) , TMPREG2 )
      << I1( "tst" , TMPREG2 )
      << I1( "bne" , INSTRUCTIONLABEL( processnumber() ,
				       handle(i)->targetof() ) )
      << I0( "nop" );
  }

  // No relevant exception was raised, so we are either done for the instant,
  // or all processes terminated and we may exit by branching to the next
  // instruction

  s << I1( "tst" , TMPREG )		// Check the subprocess return status 
    << I1( "be" , INSTRUCTIONLABEL( processnumber(), indexof()+1 ) )
    << I0( "nop" );

  // The subprocesses did not all terminate (at least one halted),
  // So we, too are halted for the instant.  Set the halted flag so we
  // do not call the subprocesses again this instant
  // TMPREG is non-zero, so saving it in the halted flag ensures that
  // we will not be called again this instant

  s << "! Halting\n";

  s << I2( "set" , LOCALLABEL( checksignals ) , NEWPCREG )
    << I2( "set" , LOCALLABEL( calcpoten ), NEWPOREG )
    << I3( "or" , TMPREG , STATUSREG, STATUSREG );

  debugreturn( s, processnumber(), indexof(), pcof() );

  s << I1( "ba" , PRETURNLABEL( processnumber() ))
    << I2( "stb" , TMPREG , H( pcof() ) );

  // This is run when not all the watched signals are known yet
  // Return with the waiting status and resume at checksignals:

  if ( n_watches() ) {

    s << "! Returning with waiting status\n";

    s << LABEL( LOCALLABEL( missingsignals ) )
      << I2( "set" , LOCALLABEL( checksignals ) , NEWPCREG )
      << I2( "set" , LOCALLABEL( calcpoten ) , NEWPOREG );

    debugreturn( s, processnumber(), indexof(), pcof() );

    // we are not halted, so return with that status

    s << I2( "stb", ZERO, H( pcof() ) )
      << I1( "ba" , PRETURNLABEL( processnumber() ) )
      << I3( "or" , STATUSREG , STATUSWAIT , STATUSREG );
  }

  // This calls all the potential set calculation routines and
  // adds those signals in the potential sets of all the handlers and the
  // next instruction

  s << LABEL( LOCALLABEL( calcpoten ) );

  s << "! Calculating potential set\n";

  // Check for the halted status and branch away if this try
  // is halted

  s << I2( "ldsb" , H( pcof() ) , TMPREG )
    << I1( "tst" , TMPREG )
    << I1( "bne" , PORETURNLABEL( processnumber() ) )
    << I0( "nop" );

  for ( i = n_subprocesses() ; --i >= 0 ; ) {
    s << I2( "ld" , PO( subprocess(i)->pcof() ) , TMPREG )
      << I2( "call" , TMPREG , "0" )
      << I0( "nop" )

      // The potential-set calculator will return here

      << LABEL( PORETURNLABEL( subprocess(i)->processnumber() ) );
  }

  // Calculate the potential set of this try by ORing those
  // of all the watch targets, all the handle targets

  setc poten( TICKsd->n_signals() );

  for ( i = n_watches() ; --i >= 0 ; )
    (*processof())[watch(i)->targetof()]->potentialof(poten);

  for ( i = n_handles() ; --i >= 0 ; )
    (*processof())[handle(i)->targetof()]->potentialof(poten);

  nextinst()->potentialof(poten);

  s << "! Potential signals: " << poten << NL;

  // Mark all the potential signals

  for ( i = TICKsd->n_signals() ; --i >= 0 ; )
    if ( poten.memberof(i) )
      s << I2( "stb" , ZERO , SIGNALPO(i) );
  
  s << I1( "ba" , PORETURNLABEL( processnumber() ) )
    << I0( "nop" );

}

//////////////////////////////
//
// Dump the variable table in an externally-readable form
//

void dumpvartab( ostream & s, sd & thevars )
{
  int i;
  sd * sdp;

  s << segment( TEXT );

  s << GLOBALLAB("_sigs");
  
  for (i = 0 ; i < thevars.n_signals() ; i++ ) {
    sdp = thevars.signals[i];
    flavor f = sdp->declarationof()->flavorof();

    if ( f == INPUT || f == OUTPUT || f == INPUTOUTPUT || f == SENSOR )
      s << I1( ".global", "_sig_" << sdp->declarationof()->nameof() )
	<< LABEL( "_sig_" <<  sdp->declarationof()->nameof() );

    s << I1( ".word" , "VNAME" << i );

    if ( f == INPUT )
      s << I1( ".word" , "1" );
    else
      s << I1( ".word" , "0" );

    s << I1( ".word" , "_S+" <<  sdp->argof() );

    if ( sdp->arg2of() >= 0 )
      s << I1( ".word" , "_SV+" << sdp->arg2of() * 4 );
    else
      s << I1( ".word" , "0" );

  }

  for ( i = 0 ; i < thevars.n_signals() ; i++ ) {
    s << LABEL( "VNAME" << i )
      << I1( ".ascii" , '\"' << thevars.signals[i]->declarationof()->nameof()
	     << "\\0\"" );
  }

}
