/* -*- C -*- */

#include <stdio.h>

#define STATUSTERMINATED 0
#define STATUSHALT 1
#define STATUSWAIT -1

#define SIGPRESENT	1
#define SIGUNKNOWN	0
#define SIGABSENT	-1

#define SIGNOPO		1
#define SIGPO		0

extern int tick();

extern int inittick();

extern int n_pc;	/* number of program counters */
extern int n_var;	/* number of variables */
extern int n_sig;	/* number of signals (includes valued and pure) */
extern int n_sigv;	/* number of valued signals */
extern int n_cnt;	/* number of counters */
extern int n_exc;	/* number of exceptions (includes valued and pure) */
extern int n_excv;	/* number of valued exceptions */

extern void (*PC[ /* n_pc */ ])();	/* Program counters */
extern void (*PO[ /* n_pc */ ])();	/* Potential-set calculators */
extern char H[ /* n_pc */ ];	/* Used to flag halted try instructions */

extern int V[ /* n_var */ ];		/* value of variables */

extern char S[ /* n_sig */ ];/* 0 if signal unknown, 1 present, -1 if absent */
extern char P[ /* n_sig */ ];/* 0 if signal has the potential to be emitted */

extern int SV[ /* n_sigv */ ];	/* signal values */

extern int C[ /* n_cnt */ ];	/* counter values */

extern char E[ /* n_exc */ ];	/* 1 if exception raised, 0 otherwise */

extern int EV[ /* n_excv */ ];	/* exception values */

extern int PCD[ /* n_pc */ ];	/* symbolic program counters */

typedef struct {
  char * name;
  int globalinput;
  char * sigpres;
  int * sigval;
} sig_t;

extern sig_t sigs[];

#define FORALL(x) for ( i = 0 ; i < x ; i++ )
#define PRINTARRAY(x,y) FORALL(y) printf("%d ",x[i])

initialize()
{
  int i;

  FORALL(n_pc) H[i] = 0;
  FORALL(n_sig) S[i] = P[i] = 0;
  FORALL(n_sigv) SV[i] = 0;
  FORALL(n_cnt) C[i] = 0;
  FORALL(n_exc) E[i] = 0;
  FORALL(n_excv) EV[i] = 0;
  clearsignals();

  inittick();

  simtick(0);

}

printarrays()
{
  int i;

  printf("PC: "); PRINTARRAY(PC,n_pc); printf("\n");
  printf("PO: "); PRINTARRAY(PO,n_pc); printf("\n");
  printf("H: "); PRINTARRAY(H,n_pc); printf("\n");

  printf("V: "); PRINTARRAY(V,n_var); printf("\n");

  printf("S: "); PRINTARRAY(S,n_sig); printf("\n");
  printf("P: "); PRINTARRAY(P,n_sig); printf("\n");

  printf("SV: "); PRINTARRAY(SV,n_sigv); printf("\n");

  printf("C: "); PRINTARRAY(C,n_cnt); printf("\n");

  printf("E: "); PRINTARRAY(E,n_exc); printf("\n");

  printf("EV: "); PRINTARRAY(EV,n_excv); printf("\n");

}

printpcs()
{
  int i;

  for ( i = 0 ; i < n_pc ; i++ ) {
    if ( PCD[i] >= 0 )
      printf("PC%d: P%dI%d\n",i, PCD[i] >> 16, PCD[i] & 0xFFFF);
    else
      printf("PC%d: -\n",i);
  }
}

printsignals()
{
  int i;
  for ( i = 0 ; i < n_sig ; i++ ) {
    switch( *(sigs[i].sigpres) ) {
    case SIGPRESENT:
      printf("%s",sigs[i].name);
      if ( sigs[i].sigval )
	printf("(%d)",*(sigs[i].sigval));
      printf(" ");
      break;
    case SIGUNKNOWN:
      printf("?%s? ",sigs[i].name);
      break;
    case SIGABSENT:
      break;
    default:
      printf("!!%s ",sigs[i].name);
      break;
    }
  }

  printf("\n");
}

printpotentials()
{
  int i;

  printf("{ ");

  for ( i = 0 ; i < n_sig ; i++ ) {
    if ( P[i] == SIGPO )
      printf("%s ",sigs[i].name);
  }

  printf("} ");
}

clearsignals()
{
  int i;
  FORALL(n_sig) S[i] = sigs[i].globalinput ? SIGABSENT : SIGUNKNOWN;
}

int
simtick(debug)
int debug;
{
  int i;
  int tickreturn;
  int iterations = 0;

  FORALL(n_pc) { H[i] = 0; PCD[i] = -1; } /* Clear all halted flags */

  S[0] = SIGPRESENT;		/* Set TICK to be present */

  do {
    FORALL(n_sig) P[i] = SIGNOPO;	/* No signal has potential */

    /* printarrays(); */

    tickreturn = tick(); iterations++;

    /* If any signal is unknown and has no potential to be emitted, mark
       it as absent */

    FORALL(n_sig)
      if ( S[i] == SIGUNKNOWN && P[i] == SIGNOPO ) {
	S[i] = SIGABSENT;
      }

    if ( debug >= 4 )
      printarrays();
    if ( debug >= 3 )
      printpcs();
    if ( debug >= 2 ) {
      printpotentials(); printsignals();
    }

  } while ( tickreturn == STATUSWAIT );

  if ( debug >= 1 ) { printf("%d iterations\n",iterations); printsignals(); }
  if ( debug >= 2 ) printf("\n");

  return (tickreturn == STATUSTERMINATED) ? STATUSTERMINATED : iterations ;
}
