// -*- C++ -*-

// Parsing-related header file for the Esterel compiler

// Stephen Edwards

// Turn on to display debugging information from the string table routines
// #define STDEBUG

//////////////////////////////////////////////////////////////////////
//
// Forward class definitions
//

class stringtable;

class symtab;

class var;
class varinfo;

//////////////////////////////////////////////////////////////////////
//
// Structure and class definitions
//

//////////////////////////////
//
// The string table class
//
// A hash table with buckets which stores all the strings in the input file
//
// Ensures each string is stored exactly once, making later
// comparisons inexpensive
//
// The << operator is defined in "scan.l"
//

class stringtable {

  struct entry {
    char * string;
    entry * next;
  };

  entry * * entries;	// array of pointers to linked lists of entries

  int tablesize;

  // The hash function

  int slotnumber( char * s ) {
    char * c = s; int slot = 0;
    for ( ; *c != '\0' ; slot = ( (slot << 1) + *c++ ) % tablesize )
      ;
    return slot;
  }

public:
  stringtable( int size ) {
    tablesize = size;
    entries = new entry * [tablesize];

    for(  ; --size >= 0 ; )
      entries[size] = 0;
  }

  ~stringtable() {
    entry * e, * e1;

    for( int i = tablesize ; --i >= 0 ; )
      for ( e = entries[i] ; e ; e = e1 ) {
	e1 = e->next;
	delete[] e->string;
	delete[] e;
      }

    delete[] entries;
  }

  // copy the given string into the stringtable if it is necessary,
  // otherwise return the copy already present

  char * add( char * s ) {

    char * s1 = lookup(s);
    if ( s1 ) return s1;

    int slot = slotnumber( s );
    entry * e = new entry;
    e->string = new char[strlen(s) + 1];
    strcpy(e->string,s);
    e->next = entries[slot];
    entries[slot] = e;

#ifdef STDEBUG
    cout << "adding " << s << " to slot " << slot << '\n';
#endif

    return e->string;
  }

  // return the copy of the string from the stringtable, or NULL if it
  // was absent

  char * lookup( char * s ) {
    int slot = slotnumber( s );
    entry * e;
    for ( e = entries[slot] ;
	  e != 0 && strcmp(e->string, s) != 0 ;
	  e = e->next )
      ;
    return e ? e->string : 0;
  }

friend
  ostream & operator << ( ostream & s, stringtable & st );

};

//////////////////////////////
//
// The symbol table class
//
// Simple linked-list implementation of a stack and a context
// stack.  Each symbol table entry is a declaration
//

class symtab {

  struct entry {
    ASTdeclaration * decl;
    entry * next;
  };

  struct environ {
    entry * top;
    environ * next;
  };

  entry * tos;
  entry * oldtos;
  environ * topenviron;

public:
  symtab() { tos = 0; topenviron = 0; }

  void pushscope() { environ * e = new environ;
		     e->top = tos;		e->next = topenviron;
		     topenviron = e;
		     tos = 0;
		   }

  void popscope() { tos = topenviron->top;
		    environ * e = topenviron->next;
		    delete topenviron;
		    topenviron = e;
		  }

  void pushsym(ASTdeclaration * d) { entry * e = new entry;
				     e->decl = d;	e->next = tos;
				     tos = e;
				   }

  void pseudopush() { tos = oldtos; }
  void pseudopop() { oldtos = tos; tos = 0; }

  ASTdeclaration * findlocalsym(char * s) {
    for ( entry * e = tos ; e ; e = e->next )
      if ( e->decl->nameof() == s ) return e->decl;
    return 0;
  }

  ASTdeclaration * findglobalsym(char * s) {
    ASTdeclaration * d = findlocalsym(s);
    if ( d != 0 ) return d;
    for ( environ * en = topenviron ; en ; en = en->next )
      for ( entry * e = en->top ; e ; e = e->next )
	if ( e->decl->nameof() == s) return e->decl;
    return 0;
  }

};
