%{ // -*- C++ -*-

// yacc/bison grammar for Esterel

// Stephen Edwards

#include "est.h"

// The bison-generated parser uses malloc() and free(), defined in stdlib.h
#include <stdlib.h>

// Abstract Syntax Tree for the program being compiled

ASTmodule * theprogram;

// Symbol tables for the program, one for each name space

symtab variables;
symtab signals;		// inputs, outputs, inputoutputs, local signals
symtab exceptions;

// Make the error messages more interesting.  YYDEBUG must be non-zero for
// YYERROR_VERBOSE to work.  Debugging is not fully enabled unless yydebug = 1

#define YYDEBUG 1
#define YYERROR_VERBOSE

extern int yydebug;
extern int yylineno;

int nerrs = 0;

%}

%{ // Token definitions
%}

%token And Await Call Case Combine Constant Copymodule Do Each Else
%token Emit End Every Exit False Function Halt Handle If Immediate
%token In Input Inputoutput Loop Mod Module Not Nothing Or Output
%token Present Procedure Relation Repeat Sensor Signal Sustain Then Timeout
%token Times Trap True Type Upto Var Watching With

%token Integer Boolean

%token ColEq ExceptionVal SignalVal

%token ParComp

%token <i> IntLit
%token <s> ID

%token '+' '-' '*' '/' '<' '>' GE LE '.' '=' NE

%{ // Precedence relations
%}

%left Or
%left And
%left Not
%left '<' '>' LE GE '=' NE
%left '+' '-'
%left '*' '/' Mod
%right Uminus

%left ParComp
%left ';'


%union{
  char * s;
  int	i;
  ASTmodule		* ASTmodulet;
  ASTdeclaration	* ASTdeclarationt;
  ASTinstruction	* ASTinstructiont;
  ASTexpression		* ASTexpressiont;
  ASThandler		* ASThandlert;
  ASTpar1		* ASTpar1t;
  ASToccurrence		* ASToccurrencet;
  ASTcase		* ASTcaset;
}

%type <ASTmodulet>	file module
%type <ASTdeclarationt> declarations declaration1 declaration declarationlist
%type <ASTdeclarationt>	decl variablelist typedlist purelist vardecl

%type <ASTexpressiont>	expression

%type <ASTinstructiont> instruction instruction1 instruction2 
%type <ASTinstructiont> thenclause elseclause

%type <ASThandlert>	handler
%type <ASTpar1t>	parinst
%type <ASTcaset>	case cases
%type <ASToccurrencet> 	occurrence

%%

file :
    module		{ theprogram = $$ = $1; }
  | file module		{ $1->append($2); theprogram = $$ = $1; }
  ;

module :
    Module ID ':' declarations instruction '.'
	{ $$ = new ASTmodule(@1, $2,$4,$5); }
  ;

/*
 * Declarations
 */

declarations :
    { $$ = 0; }
  | declaration1
  ;

declaration1 :
    declaration			{ $$ = $1; }
  | declaration1 declaration	{ $1->append($2); $$ = $1; }
  ;

declaration :
    Input declarationlist ';'		{ $2->setflavor(INPUT); $$ = $2; }
  | Output declarationlist ';'		{ $2->setflavor(OUTPUT); $$ = $2; }
  | Inputoutput declarationlist ';'	{ $2->setflavor(INPUTOUTPUT); $$ = $2;}
  | Sensor declarationlist ';'		{ $2->setflavor(SENSOR); $$ = $2; }
  ;

declarationlist :
    decl			{ $$ = $1; }
  | declarationlist ',' decl	{ $1->append($3); $$ = $1; }
  ;

decl :
    ID			{ $$ = new ASTdeclaration(@1, $1); }
  | ID '(' Integer ')'  { $$ = new ASTdeclaration(@1, $1, INTEGER); }
  | ID '(' Boolean ')'  { $$ = new ASTdeclaration(@1, $1, BOOLEAN); }
  ;

/*
 * Expressions
 */

expression :
    IntLit		{ $$ = new ASTintlit( @1, $1 ); }
  | True		{ $$ = new ASTboollit( @1, TRUE ); }
  | False		{ $$ = new ASTboollit( @1, FALSE ); }
  | ID			{ $$ = new ASTvariableid( @1, $1 ); }
  | SignalVal ID	{ $$ = new ASTsignalid( @2, $2 ); }
  | ExceptionVal ID	{ $$ = new ASTexceptionid( @2, $2 ); }
  | '(' expression ')'  { $$ = $2; }
  | '-' expression %prec Uminus { $$ = new ASTunop( @1, opNEG, $2 ); }
  | expression '*' expression { $$ = new ASTbinop( @2, $1, opMULT, $3 ); }
  | expression '/' expression { $$ = new ASTbinop( @2, $1, opDIV , $3 ); }
  | expression Mod expression { $$ = new ASTbinop( @2, $1, opMOD , $3 ); }
  | expression '+' expression { $$ = new ASTbinop( @2, $1, opADD , $3 ); }
  | expression '-' expression { $$ = new ASTbinop( @2, $1, opSUB , $3 ); }
  | expression '<' expression { $$ = new ASTbinop( @2, $1, opLT  , $3 ); }
  | expression LE  expression { $$ = new ASTbinop( @2, $1, opLE  , $3 ); }
  | expression '>' expression { $$ = new ASTbinop( @2, $1, opGT  , $3 ); }
  | expression GE  expression { $$ = new ASTbinop( @2, $1, opGE  , $3 ); }
  | expression '=' expression { $$ = new ASTbinop( @2, $1, opEQ  , $3 ); }
  | expression NE  expression { $$ = new ASTbinop( @2, $1, opNE  , $3 ); }
  | Not expression	      { $$ = new ASTunop( @1, opNOT, $2 ); }
  | expression And expression { $$ = new ASTbinop( @2, $1, opAND , $3 ); }
  | expression Or  expression { $$ = new ASTbinop( @2, $1, opOR  , $3 ); }
  ;

/*
 * Instructions
 */

instruction :
    instruction1
  | parinst ParComp instruction1 { $1->append(new ASTpar1(@2, $3));
				   $$ = new ASTpar(@2, $1);
				 }
  ;

parinst :
    instruction1		 { $$ = new ASTpar1(@1, $1); }
  | parinst ParComp instruction1 { $1->append(new ASTpar1(@2, $3)); $$ = $1; }
  ;

instruction1 :
    instruction2		{ $$ = $1; }
  | instruction2 ';'		{ $$ = $1; }
  ;

instruction2 :
    Nothing				{ $$ = new ASTnothing(@1); }
  | Halt				{ $$ = new ASThalt(@1); }
  | Var variablelist In instruction End
				{ $$ = new ASTvariableblock( @1, $2, $4 ); }
  | Signal declarationlist In instruction End {
	 $2->setflavor(LOCALSIG);
	 $$ = new ASTsignalblock( @1, $2, $4);
    }
  | Trap decl In instruction End {
	$$ = new ASTexceptionblock(@1, $2,$4);
    }
  | Trap decl In instruction handler End {
	$$ = new ASTexceptionblock(@1, $2,$4,$5);
    }
  | Exit ID				{ $$ = new ASTexit(@1, $2); }
  | Exit ID '(' expression ')'		{ $$ = new ASTexit(@1, $2,$4); }
  | ID ColEq expression			{ $$ = new ASTassign(@2, $1, $3); }
  | Emit ID				{ $$ = new ASTemit(@1, $2); }
  | Emit ID '(' expression ')'		{ $$ = new ASTemit(@1, $2,$4); }
  | Sustain ID				{
	ASTinstruction * tmp = new ASTemit(@1, $2);
	ASToccurrence * tickoccurrence = new ASToccurrence(@2, TICKstring);
	tmp->append(new ASThalt(@2));
	$$ = new ASTawait(@1, tickoccurrence );
	$$->append(new ASTloop( @1, new ASTdowatching( @2,
						       tmp,
						       tickoccurrence )));
      }
  | Sustain ID '(' expression ')'	{
	ASTinstruction * tmp = new ASTemit(@1, $2, $4);
	ASToccurrence * tickoccurrence = new ASToccurrence(@2, TICKstring);
	tmp->append(new ASThalt(@2));
	$$ = new ASTawait(@1, tickoccurrence );
	$$->append(new ASTloop( @1, new ASTdowatching( @2,
						       tmp, tickoccurrence )));
      }
  | '[' instruction ']'			{ $$ = $2; }
  | instruction2 ';' instruction2	{ $1->append($3); $$ = $1; }
  | If expression thenclause elseclause End { $$ = new ASTif(@1, $2,$3,$4); }
  | Present ID thenclause elseclause End { $$ = new ASTpresent(@1, $2,$3,$4); }
  | Await occurrence			{ $$ = new ASTawait(@1, $2); }
  | Await occurrence Do instruction End  { $$ = new ASTawait(@1, $2);
					  $$->append($4);
					}
  | Await cases End			{ $$ = new ASTawaitcase(@1, $2); }
  | Loop instruction End		{ $$ = new ASTloop(@1, $2); }
  | Loop instruction Each occurrence	{
	$2->append(new ASThalt(@3));
	$$ = new ASTloop( @1, new ASTdowatching( @1, $2, $4) );
    }
  | Repeat expression Times instruction End { $$ = new ASTrepeat(@1, $2,$4); }
  | Do instruction Watching occurrence	{ $$ = new ASTdowatching(@1, $2,$4); }
  | Do instruction Watching occurrence Timeout instruction End
				{ $$ = new ASTdowatching(@1, $2,$4,$6); }
  | Do instruction Upto occurrence	{ $2->append(new ASThalt(@3));
					  $$ = new ASTdowatching(@1,$2,$4); }
  | Every occurrence Do instruction End  {
	$4->append(new ASThalt(@3));
	$$ = new ASTawait(@1, $2);
	$$->append(new ASTloop( @1, new ASTdowatching( @3, $4, $2 ) ) );
    }
  ;

variablelist :
    typedlist		 	{ $$ = $1; }
  | variablelist ',' typedlist	{ $1->append($3); $$ = $1; }
  ;

typedlist :
    purelist ':' Integer	{ $1->settype(INTEGER); $$ = $1; }
  | purelist ':' Boolean	{ $1->settype(BOOLEAN); $$ = $1; }
  ;

purelist :
    vardecl			{ $$ = $1; }
  | purelist ',' vardecl	{ $1->append($3); $$ = $1; }
  ;

vardecl :
    ID				{ $$ = new ASTdeclaration(@1, $1); }
  | ID ColEq expression		{ $$ = new ASTdeclaration(@2, $1, $3); }
  ;

handler :
    Handle ID Do instruction	{ $$ = new ASThandler(@1, $2,$4); }
  ;

thenclause :
    { $$ = 0; }
  | Then instruction { $$ = $2; }
  ;

elseclause :
    { $$ = 0; }
  | Else instruction { $$ = $2; }
  ;

occurrence :
    ID				{ $$ = new ASToccurrence(@1, $1); }
  | Immediate ID		{ $$ = new ASToccurrence(@2, $2,0); }
  | expression ID		{ $$ = new ASToccurrence(@2, $2,$1); }
  ;

cases :
    case			{ $$ = $1; }
  | cases case			{ $1->append($2); $$ = $1; }
  ;

case :
    Case occurrence			{ $$ = new ASTcase( @1, $2 ); }
  | Case occurrence Do instruction	{ $$ = new ASTcase( @1, $2, $4 ); }
  ;

%%

void
bail()
{
  cerr << "Too many errors---bailing out\n";
  exit(1);
}

void
yyerror( char * s, astnode * node = 0, char * t = 0 )
{
  if ( node != 0 ) { yylineno = node->locationof().first_line; }
  cerr.width(5);
  cerr << yylineno << ": " << s;
  if (t) cerr << t;
  cerr << '\n';
  if ( ++nerrs > MAXERRORS ) { bail(); }
}

void
yywarning( char * s, astnode * node = 0, char * t = 0 )
{
  if ( node != 0 ) { yylineno = node->locationof().first_line; }
  cerr.width(5);
  cerr << yylineno << ": " << s;
  if (t) cerr << t;
  cerr << '\n';
}
