// -*- C++ -*-

// Abstract Syntax Tree header file for the Esterel compiler

// Stephen Edwards

//////////////////////////////////////////////////////////////////////
//
// Forward class definitions
//

class sd;

class astnode;
class astlist;
class ASTdeclaration;
class ASTexpression;
class ASTintlit;
class ASTboollit;
class ASTvariableid;
class ASTsignalid;
class ASTexceptionid;
class ASTbinop;
class ASTunop;
class ASTinstruction;
class ASTnothing;
class ASThalt;
class ASTassign;
class ASTvariableblock;
class ASTsignalblock;
class ASThandler;
class ASTexceptionblock;
class ASTexit;
class ASTemit;
class ASTpar1;
class ASTpar;
class ASTif;
class ASTpresent;
class ASToccurrence;
class ASTawait;
class ASTcase;
class ASTawaitcase;
class ASTloop;
class ASTrepeat;
class ASTdowatching;
class ASTmodule;

class AMrequire;	// in am.h
class amprocess;

//////////////////////////////////////////////////////////////////////
//
// Structure and class definitions
//

enum sdtype { SDnone, SDreg, SDvar, SDsig, SDcntr, SDexc, SDconst };

// a source or destination
// register/variable/signal/counter/exception/constant

class sd {
  sdtype type;		// What type of sd this is
  int arg;		// register number, signal number, etc.
  int arg2;		// index of valued signal/exception

  ASTdeclaration * declaration;	// declaration which generated this, if any

  static array<sd *> variables;
  static array<sd *> signals;
  static array<sd *> counters;
  static array<sd *> exceptions;

  static int valuedsignals;
  static int valuedexceptions;

public:

  sd( sdtype t, ASTdeclaration * d = 0);
  sd( sdtype t, int a );

  sdtype sdtypeof() { return type; }

  int argof() { return arg; }
  int arg2of() { return arg2; }

  ASTdeclaration * declarationof() { return declaration; }

  int n_variables() { return variables.size(); }
  int n_signals() { return signals.size(); }
  int n_valuedsignals() { return valuedsignals; }
  int n_counters() { return counters.size(); }
  int n_exceptions() { return exceptions.size(); }
  int n_valuedexceptions() { return valuedexceptions; }

  void printsd( ostream & s );

friend
  ostream & operator << ( ostream & s, sd & st );

friend
  void dumpvartab( ostream & s, sd & thevars );
};

typedef int TARGET;
#define NOTARGET -1

// The definition of YYLTYPE actually comes from gram.h---used for location
// reports

#ifndef YYLTYPE
typedef
  struct yyltype
    {
      int timestamp;
      int first_line;
      int first_column;
      int last_line;
      int last_column;
      char *text;
   }
  yyltype;

#define YYLTYPE yyltype
#endif

#define PRINTFUNC void print( ostream & s, int i = -1 )
#define SEMANTFUNC void semant()
#define TRANSFUNC sd * translate( amprocess & s, int reg = 0 )
#define REQUIREFUNC void require( AMrequire & r )

#define ASTNODEFUNCS PRINTFUNC; SEMANTFUNC; TRANSFUNC

#define ASTEXPRESSIONFUNCS ASTNODEFUNCS; REQUIREFUNC

//////////////////////////////
//
// The Abstract Syntax Tree (AST) classes
//
// All are derived from either astnode (a single node) or
// astlist (a singly-linked list of nodes)
//
// All should have a virtual function "print"
//
// All should have a virtual function "semant" which does semantic checking
// on the node.

enum etype { PURE, INTEGER, BOOLEAN };
enum flavor { INPUT, OUTPUT, INPUTOUTPUT, SENSOR,
	      LOCALVAR, LOCALSIG, EXCEPTION, NONE };
enum boolean { TRUE = 1, FALSE = 0 };

enum binop { opMULT, opDIV, opMOD, opADD, opSUB,
	opLT, opLE, opGT, opGE, opEQ, opNE, opAND, opOR };

enum unop { opNEG, opNOT };

enum occur { SIMPLE, COUNTED, IMMEDIATE };

class astnode {
  astnode * parent;
  YYLTYPE location;
public:
  astnode(YYLTYPE & loc, astnode * p = 0)
  {
    location = loc;
    parent = p;
  }
  astnode * parentof() { return parent; }
  astnode * setparent( astnode * p ) { parent = p; return this; }
  astnode * attach(astnode * p) { parent = p; return this; }

  YYLTYPE & locationof() { return location; }

  virtual PRINTFUNC;
  virtual SEMANTFUNC;
  virtual TRANSFUNC;

friend
  ostream & operator << ( ostream & s, astnode & n);
};

class astlist : public astnode {
  astlist * next;
public:
  astlist( YYLTYPE & loc )
  : astnode(loc)
  {
    next = 0;
  }

  astlist * nextof() { return next; }
  astlist * append(astlist * n) { while(this->next) this = this->next;
				  this->next = n;
				  return n; }
  astlist * attach(astnode * p) { astlist * t = this;
				  while (this) { this->setparent(p);
						 this = this->next;
					       }
				  return t;
				}
};

//
// Declarations
//

class ASTdeclaration : public astlist {
  char * name;
  etype thetype;		// PURE, INTEGER, or BOOLEAN
  flavor theflavor;		// INPUT, OUTPUT, etc.
  ASTexpression * expression;	// Initializing expression, if any
  sd * oursd;			// the sd used by this signal/variable/etc..

public:
  ASTdeclaration( YYLTYPE & loc, char * s, etype t = PURE, flavor f = NONE)
  : astlist(loc)
  {
    name = s;
    thetype = t;
    theflavor = f;
    expression = 0;
    oursd = 0;
  }

  ASTdeclaration( YYLTYPE & loc, char * s, ASTexpression * e)
  : astlist(loc)
  {
    name = s;
    thetype = PURE;
    theflavor = NONE;
    expression = e;
    oursd = 0;
  }

  char * nameof() { return name; }
  etype etypeof() { return thetype; }
  flavor flavorof() { return theflavor; }
  ASTexpression * expressionof() { return expression; }

  void setsd( sd * s ) { oursd = s; }
  sd * sdof() { return oursd; }

  void setflavor( flavor f ) { theflavor = f;
			       while (this = 
				      (ASTdeclaration *)this->nextof() )
				 this->theflavor = f;
			     }
  void settype( etype t ) { thetype = t;
			    while ( this = (ASTdeclaration *) this->nextof() )
			      this->thetype = t;
			  }

  ASTNODEFUNCS;

};

//
// Expressions
//

class ASTexpression : public astnode {
  etype thetype;
public:
  ASTexpression( YYLTYPE & loc )
  : astnode(loc)
  {
    thetype = PURE;
  }

  void setetype(etype t) { thetype = t; }
  etype etypeof() { return thetype; }

  virtual REQUIREFUNC;

};

class ASTintlit : public ASTexpression {
  int value;
public:
  ASTintlit( YYLTYPE & loc, int v) 
  : ASTexpression(loc)
  {
    value = v;
    setetype(INTEGER);
  }

  int valueof() { return value; }

  ASTEXPRESSIONFUNCS;
};

class ASTboollit : public ASTexpression {
  boolean value;
public:
  ASTboollit( YYLTYPE & loc, boolean b)
  : ASTexpression(loc)
  {
    value = b;
    setetype(BOOLEAN);
  }

  boolean valueof() { return value; }

  ASTEXPRESSIONFUNCS;
};

class ASTvariableid : public ASTexpression {
  char * name;
  ASTdeclaration * decl;
public:
  ASTvariableid( YYLTYPE & loc, char * s )
  : ASTexpression( loc )
  {
    name = s;
    decl = 0;
  }

  char * nameof() { return name; }
  void setdecl(ASTdeclaration * d) { decl = d; setetype(d->etypeof()); }
  ASTdeclaration * declof() { return decl; }
  ASTEXPRESSIONFUNCS;
};

class ASTsignalid : public ASTexpression {
  char * name;
  ASTdeclaration * decl;
public:
  ASTsignalid( YYLTYPE & loc, char * s)
  : ASTexpression(loc)
  {
    name = s;
    decl = 0;
  }

  char * nameof() { return name; }
  void setdecl(ASTdeclaration * d) { decl = d; setetype(d->etypeof()); }
  ASTdeclaration * declof() { return decl; }
  ASTEXPRESSIONFUNCS;
};

class ASTexceptionid : public ASTexpression {
  char * name;
  ASTdeclaration * decl;
public:
  ASTexceptionid( YYLTYPE & loc, char * s)
  : ASTexpression( loc )
  {
    name = s;
  }

  char * nameof() { return name; }
  void setdecl(ASTdeclaration * d) { decl = d; setetype(d->etypeof()); }
  ASTdeclaration * declof() { return decl; }
  ASTEXPRESSIONFUNCS;
};

class ASTbinop : public ASTexpression {
  binop oper;
  ASTexpression * left, * right;
public:
  ASTbinop( YYLTYPE & loc, ASTexpression * l, binop o, ASTexpression * r)
  : ASTexpression(loc) 
  {
    oper = o;
    left = l;	l->attach(this);
    right = r;	r->attach(this);
  }

  ASTexpression * leftof() { return left; }
  ASTexpression * rightof() { return right; }
  binop opof() { return oper; }

  ASTEXPRESSIONFUNCS;
};

class ASTunop : public ASTexpression {
  unop oper;
  ASTexpression * right;
public:
  ASTunop( YYLTYPE & loc, unop o, ASTexpression * r)
  : ASTexpression(loc)
  {
    oper = o;
    right = r;	r->attach(this);
  }

  ASTexpression * rightof() { return right; }
  unop opof() { return oper; }

  ASTEXPRESSIONFUNCS;
};

//
// Instructions
//

class ASTinstruction : public astlist {
public:
  ASTinstruction( YYLTYPE & loc ) 
  : astlist(loc)
  {}

  PRINTFUNC;
  SEMANTFUNC;
};

class ASTnothing : public ASTinstruction {
public:
  ASTnothing( YYLTYPE & loc )
  : ASTinstruction(loc)
  {}

  PRINTFUNC;
  TRANSFUNC;
};

class ASThalt : public ASTinstruction {
public:
  ASThalt( YYLTYPE & loc )
  : ASTinstruction(loc)
  {}

  PRINTFUNC;
  TRANSFUNC;
};

class ASTassign : public ASTinstruction {
  char * name;
  ASTexpression * expression;
  ASTdeclaration * decl;
public:
  ASTassign( YYLTYPE & loc, char * s, ASTexpression * e)
  : ASTinstruction(loc)
  {
    name = s;
    expression = e;	e->attach(this);
    decl = 0;
  }

  char * nameof() { return name; }
  ASTexpression * expressionof() { return expression; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  ASTdeclaration * declof() { return decl; }
  
  ASTNODEFUNCS;
};

class ASTvariableblock : public ASTinstruction {
  ASTdeclaration * vars;
  ASTinstruction * instruction;
public:
  ASTvariableblock( YYLTYPE & loc, ASTdeclaration * v, ASTinstruction * i)
  : ASTinstruction( loc )
  {
    vars = v;		v->attach(this); v->setflavor(LOCALVAR);
    instruction = i;	i->attach(this);
  }

  ASTdeclaration * variablesof() { return vars; }
  ASTinstruction * instructionof() { return instruction; }

  ASTNODEFUNCS;
};

class ASTsignalblock : public ASTinstruction {
  ASTdeclaration * sigs;
  ASTinstruction * instruction;
public:
  ASTsignalblock( YYLTYPE & loc, ASTdeclaration * s, ASTinstruction * i)
  : ASTinstruction(loc)
  {
    sigs = s;		s->attach(this); s->setflavor(LOCALSIG);
    instruction = i;	i->attach(this);
  }

  ASTdeclaration * signalsof() { return sigs; }
  ASTinstruction * instructionof() { return instruction; }

  ASTNODEFUNCS;
};

class ASThandler : public ASTinstruction {
  char * name;
  ASTinstruction * instruction;
  ASTdeclaration * decl;
public:
  ASThandler( YYLTYPE & loc, char * s, ASTinstruction * i )
  : ASTinstruction(loc)
  {
    name = s;
    instruction = i;	i->attach(this);
    decl = 0;
  }

  char * nameof() { return name; }
  ASTinstruction * instructionof() { return instruction; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  ASTdeclaration * declof() { return decl; }
  PRINTFUNC;
  SEMANTFUNC;
};

class ASTexceptionblock : public ASTinstruction {
  ASTdeclaration * except;
  ASTinstruction * instruction;
  ASThandler * handler;
public:
  ASTexceptionblock( YYLTYPE & loc, ASTdeclaration * e, ASTinstruction * i,
		     ASThandler * h = 0)
  : ASTinstruction(loc)
  {
    except = e; e->setflavor(EXCEPTION); e->attach(this);
    instruction = i; i->attach(this);
    handler = h; if (h) h->attach(this);
  }

  ASTdeclaration * exceptionof() { return except; }
  ASTinstruction * instructionof() { return instruction; }
  ASThandler * handlerof() { return handler; }

  ASTNODEFUNCS;
};

class ASTexit : public ASTinstruction {
  char * name;
  ASTexpression * expression;
  ASTdeclaration * decl;
public:
  ASTexit( YYLTYPE & loc, char * s, ASTexpression * e = 0)
  : ASTinstruction(loc)
  {
    name = s;
    expression = e; if (e) e->attach(this);
    decl = 0;
  }

  char * nameof() { return name; }
  ASTexpression * expressionof() { return expression; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  ASTdeclaration * declof() { return decl; }

  ASTNODEFUNCS;
};

class ASTemit : public ASTinstruction {
  char * name;
  ASTexpression * expression;
  ASTdeclaration * decl;
public:
  ASTemit( YYLTYPE & loc, char * s, ASTexpression * e = 0)
  : ASTinstruction(loc)
  {
    name = s;
    expression = e; if (e) e->attach(this);
  }
  char * nameof() { return name; }
  ASTexpression * expressionof() { return expression; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  ASTdeclaration * declof() { return decl; }

  ASTNODEFUNCS;
};

class ASTpar1 : public ASTinstruction {
  ASTinstruction * instruction;
public:
  ASTpar1( YYLTYPE & loc, ASTinstruction * i )
  : ASTinstruction(loc)
  {
    instruction = i; i->attach(this);
  }

  ASTinstruction * instructionof() { return instruction; }
  PRINTFUNC;
  SEMANTFUNC;
};

class ASTpar : public ASTinstruction {
  ASTpar1 * instructions;
public:
  ASTpar( YYLTYPE & loc, ASTpar1 * i )
  : ASTinstruction(loc)
  {
    instructions = i; i->attach(this);
  }

  ASTpar1 * instructionsof() { return instructions; }

  ASTNODEFUNCS;
};

class ASTif : public ASTinstruction {
  ASTexpression * expression;
  ASTinstruction * theninst, * elseinst;
public:
  ASTif( YYLTYPE & loc, ASTexpression * e,
	 ASTinstruction * t, ASTinstruction * el)
  : ASTinstruction(loc)
  {
    expression = e; e->attach(this);
    theninst = t; if (t) t->attach(this);
    elseinst = el; if (el) el->attach(this);
  }

  ASTexpression * expressionof() { return expression; }
  ASTinstruction * thenof() { return theninst; }
  ASTinstruction * elseof() { return elseinst; }

  ASTNODEFUNCS;
};

class ASTpresent : public ASTinstruction {
  char * name;
  ASTinstruction * theninst, * elseinst;
  ASTdeclaration * decl;
public:
  ASTpresent( YYLTYPE & loc, char * n, ASTinstruction * t, ASTinstruction * el)
  : ASTinstruction(loc)
  {
    name = n;
    theninst = t; if (t) t->attach(this);
    elseinst = el; if (el) el->attach(this);
    decl = 0;
  }
  char * nameof() { return name; }
  ASTinstruction * thenof() { return theninst; }
  ASTinstruction * elseof() { return elseinst; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  ASTdeclaration * declof() { return decl; }

  ASTNODEFUNCS;
};

class ASToccurrence : public astnode {
  char * name;
  occur occurtype;
  ASTexpression * expression;
  ASTdeclaration * decl;
  sd * countsd;
public:
  ASToccurrence( YYLTYPE & loc, char * s )
  : astnode(loc)
  {
    name = s;
    occurtype = SIMPLE;
    expression = 0;
  }

  ASToccurrence( YYLTYPE & loc, char * s, int i )
  : astnode(loc)
  {
    name = s;
    occurtype = IMMEDIATE;
    expression = 0;
  }

  ASToccurrence( YYLTYPE & loc, char * s, ASTexpression * e )
  : astnode(loc)
  {
    name = s;
    occurtype = COUNTED;
    expression = e; e->attach(this);
  }

  char * nameof() { return name; }
  occur occurtypeof() { return occurtype; }
  ASTexpression * expressionof() { return expression; }
  void setdecl(ASTdeclaration * d) { decl = d; }
  void setcountsd( sd * s ) { countsd = s; }
  ASTdeclaration * declof() { return decl; }
  sd * countsdof() { return countsd; }
  PRINTFUNC;
  SEMANTFUNC;
};

class ASTawait : public ASTinstruction {
  ASToccurrence * occurrence;
public:
  ASTawait( YYLTYPE & loc, ASToccurrence * o )
  : ASTinstruction(loc)
  {
    occurrence = o; o->attach(this);
  }

  ASToccurrence * occurrenceof() { return occurrence; }

  ASTNODEFUNCS;
};

class ASTcase : public astlist {
  ASToccurrence * occurrence;
  ASTinstruction * instruction;
public:
  ASTcase( YYLTYPE & loc, ASToccurrence * o, ASTinstruction * i = 0)
  : astlist(loc)
  {
    occurrence = o; o->attach(this);
    instruction = i; if (i) i->attach(this);
  }

  ASToccurrence * occurrenceof() { return occurrence; }
  ASTinstruction * instructionof() { return instruction; }
  PRINTFUNC;
  SEMANTFUNC;
};

class ASTawaitcase : public ASTinstruction {
  ASTcase * cases;
public:
  ASTawaitcase( YYLTYPE & loc, ASTcase * c )
  : ASTinstruction(loc)
  {
    cases = c; c->attach(this);
  }

  ASTcase * casesof() { return cases; }

  ASTNODEFUNCS;
};

class ASTloop : public ASTinstruction {
  ASTinstruction * instruction;
public:
  ASTloop( YYLTYPE & loc, ASTinstruction * i )
  : ASTinstruction(loc)
  {
    instruction = i; i->attach(this);
  }

  ASTinstruction * instructionof() { return instruction; }

  ASTNODEFUNCS;
};

class ASTrepeat : public ASTinstruction {
  ASTexpression * expression;
  ASTinstruction * instruction;
  sd * repvar;
public:
  ASTrepeat( YYLTYPE & loc, ASTexpression * e, ASTinstruction * i )
  : ASTinstruction(loc)
  {
    expression = e; e->attach(this);
    instruction = i; i->attach(this);
  }

  ASTexpression * expressionof() { return expression; }
  ASTinstruction * instructionof() { return instruction; }

  void setrepvar( sd * s ) { repvar = s; }
  sd * repvarof() { return repvar; }

  ASTNODEFUNCS;
};

class ASTdowatching : public ASTinstruction {
  ASTinstruction * instruction, * timeout;
  ASToccurrence * occurrence;
public:
  ASTdowatching( YYLTYPE & loc, ASTinstruction * i, ASToccurrence * o,
		 ASTinstruction * t = 0 )
  : ASTinstruction(loc)
  {
    instruction = i;	i->attach(this);
    occurrence = o;	o->attach(this);
    timeout = t;	if (t) t->attach(this);
  }

  ASTinstruction * instructionof() { return instruction; }
  ASToccurrence * occurrenceof() { return occurrence; }
  ASTinstruction * timeoutof() { return timeout; }

  ASTNODEFUNCS;
};

//
// Module
//

class ASTmodule : public astlist {
  char * name;
  ASTdeclaration * declarations;
  ASTinstruction * instruction;
public:
  ASTmodule(YYLTYPE & loc, char * s, ASTdeclaration * d, ASTinstruction * i)
  : astlist(loc)
  {
    name = s;
    declarations = d;	d->attach(this);
    instruction = i;	i->attach(this);
  }
  char * nameof() { return name; }
  astlist * declarationsof() { return declarations; }
  astlist * instructionof() { return instruction; }

  ASTNODEFUNCS;
};

