// -*- C++ -*-

// AST-related functions for the Esterel compiler

// Stephen Edwards

#include "est.h"

#define PRINT(x) void x::print( ostream & s, int i = -1 )
#define SEMANT(x) void x::semant()
#define TRANSLATE(x) sd * x::translate( amprocess & s, int reg = 0 )
#define REQUIRE(x) void x::require( AMrequire & r )

#define NEXTINSTTARGET s.nextindex()

//////////////////////////////
//
// The "print" virtual function defined for all AST nodes:
//

ostream &
operator << ( ostream & s, astnode & n ) {
  n.print(s);
  return s;
}

PRINT(astnode) {
  s << "ERROR: print() called on an astnode!\n";
}

//
// Declarations
//

PRINT(ASTdeclaration) {
  switch ( flavorof() ) {
  case INPUT: case OUTPUT: case INPUTOUTPUT: case SENSOR:
    s.width(i); s << "";

    switch (flavorof()) {
    case INPUT :
      s << "input ";
      break;
    case OUTPUT :
      s << "output ";
      break;
    case INPUTOUTPUT :
      s << "inputoutput ";
      break;
    case SENSOR :
      s << "sensor ";
      break;
    }

    s << nameof();

    switch (etypeof()) {
    case INTEGER :
      s << " (integer)";
      break;
    case BOOLEAN :
      s << " (boolean)";
      break;
    default:
      break;
    }
    
    s << ";\n";
    
    if ( nextof() ) nextof()->print(s,i);
    break;

  case LOCALVAR:
    s << nameof();
    if (expressionof()) { s << " := "; expressionof()->print(s,i); }
    switch ( etypeof() ) {
    case BOOLEAN :
      s << " : boolean";
      break;
    case INTEGER :
      s << " : integer";
      break;
    }
    if ( nextof() ) { s << ",\n";
		      s.width(i); s << "";
		      nextof()->print(s,i); }
    break;

  case LOCALSIG: case EXCEPTION:
    s << nameof();
    switch ( etypeof() ) {
    case BOOLEAN :
      s << " (boolean)";
      break;
    case INTEGER :
      s << " (integer)";
      break;
    default:
      break;
    }

    if ( nextof() ) { s << ",\n";
		      s.width(i); s << "";
		      nextof()->print(s,i);
		    }
    break;
  }
}

//
// Expressions
//

PRINT(ASTintlit)
{
  s << valueof();
}

PRINT(ASTboollit)
{
  switch(valueof()) {
  case TRUE:
    s << "true";
    break;
  case FALSE:
    s << "false";
    break;
  }
}

PRINT(ASTvariableid)
{
  s << nameof();
}

PRINT(ASTsignalid)
{
  s << '?' << nameof();
}

PRINT(ASTexceptionid)
{
  s << "??" << nameof();
}


PRINT(ASTbinop)
{
  s << '(';

  leftof()->print(s);
  
  s << ' ';
  switch( opof() ) {
  case opMULT :
    s << '*';
    break;
  case opDIV :
    s << '/';
    break;
  case opMOD :
    s << "mod";
    break;
  case opADD :
    s << '+';
    break;
  case opSUB :
    s << '-';
    break;
  case opLT :
    s << '<';
    break;
  case opLE :
    s << "<=";
    break;
  case opGT :
    s << '>';
    break;
  case opGE :
    s << ">=";
    break;
  case opEQ :
    s << '=';
    break;
  case opNE :
    s << "<>";
    break;
  case opAND :
    s << "and";
    break;
  case opOR :
    s << "or";
    break;
  }

  s << ' ';

  rightof()->print(s,i);
  s << ')';
}

PRINT(ASTunop)
{
  switch( opof() ) {
  case opNEG :
    s << '-';
    break;
  case opNOT :
    s << "not";
    break;
  }
  s << "( ";
  rightof()->print(s,i);
  s << " )";
}

//
// Instructions
//

PRINT(ASTinstruction) {
  s << "instruction;\n";
}


PRINT(ASTnothing) {
  s.width(i); s << "" << "nothing";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}


PRINT(ASThalt) {
  s.width(i); s << "" << "halt";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTassign) {
  s.width(i); s << "" << nameof() << " := ";
  expressionof()->print(s,i);
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTvariableblock) {
  s.width(i); s << "" << "var ";
  variablesof()->print(s,i+4);
  s << " in\n";
  instructionof()->print(s,i+2);
  s << '\n'; s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTsignalblock) {
  s.width(i); s << "" << "signal ";
  signalsof()->print(s,i+7);
  s << " in\n";
  instructionof()->print(s,i+2);
  s << '\n'; s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASThandler) {
  s.width(i); s << "" << "handle " << nameof() << " do\n";
  instructionof()->print(s,i+2);
  if ( nextof() ) { s << '\n'; nextof()->print(s,i); }
}

PRINT(ASTexceptionblock) {
  s.width(i); s << "" << "trap ";
  exceptionof()->print(s,i+5);
  s << " in \n";
  instructionof()->print(s,i+2);
  if ( handlerof() ) { handlerof()->print(s,i); s << '\n'; }
  s << '\n'; s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTexit) {
  s.width(i); s << "" << "exit " << nameof();
  if ( expressionof() ) { s << "( "; expressionof()->print(s,i); s << " )"; }
  if ( nextof() ) { s << ";\n"; nextof()->print(s,i); }
}

PRINT(ASTemit) {
  s.width(i); s << "" << "emit " << nameof();
  if ( expressionof() ) { s << "( "; expressionof()->print(s,i); s << " )"; }
  if ( nextof() ) { s << ";\n"; nextof()->print(s,i); }
}

PRINT(ASTpar) {
  instructionsof()->print(s,i);
  if ( nextof() ) { s << '\n';
		    s.width(i); s << ""; s << ";\n";
		    nextof()->print(s,i);
		  }
}

PRINT(ASTpar1) {
  instructionof()->print(s,i+2);
  if ( nextof() ) { s << '\n';
		    s.width(i); s << "" << "||\n";
		    nextof()->print(s,i);
		  }
}

PRINT(ASTif) {
  s.width(i); s << "" << "if ";
  expressionof()->print(s,i); s << ' ';
  if ( thenof() ) { s << "then\n";
		    thenof()->print(s,i+2); s << '\n';
		    s.width(i); s << "";
		  }
  if ( elseof() ) { s << "else\n";
		    elseof()->print(s,i+2); s << '\n';
		    s.width(i); s << "";
		  }
  s << "end";
  if (nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTpresent) {
  s.width(i); s << "" << "present " << nameof() << ' ';
  if ( thenof() ) { s << "then\n";
		    thenof()->print(s,i+2); s << '\n';
		    s.width(i); s << "";
		  }
  if ( elseof() ) { s << "else\n";
		    elseof()->print(s,i+2); s << '\n';
		    s.width(i); s << "";
		  }
  s << "end";
  if (nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASToccurrence) {
  if ( expressionof() ) { expressionof()->print(s,i); s << ' '; }
  if ( occurtypeof() == IMMEDIATE ) { s << "immediate "; }
  s << nameof();
}

PRINT(ASTawait) {
  s.width(i); s << "" << "await ";
  occurrenceof()->print(s,i);
  if (nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTcase) {
  s.width(i); s << "" << "case ";
  occurrenceof()->print(s,i);
  if (instructionof()) { s << " do\n";
			 instructionof()->print(s,i+2);
		       }
  s << '\n';
  if ( nextof() ) nextof()->print(s,i);
}

PRINT(ASTawaitcase) {
  s.width(i); s << "" << "await\n";
  casesof()->print(s,i+2);
  s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTloop) {
  s.width(i); s << "" << "loop\n";
  instructionof()->print(s,i+2); s << '\n';
  s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTrepeat) {
  s.width(i); s << "" << "repeat "; expressionof()->print(s,i);
  s << " times\n";
  instructionof()->print(s,i+2); s << '\n';
  s.width(i); s << "" << "end";
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

PRINT(ASTdowatching) {
  s.width(i) ; s << "" << "do\n";
  instructionof()->print(s,i+2); s << '\n';
  s.width(i); s << "" << "watching "; occurrenceof()->print(s,i);
  if ( timeoutof() ) { s << '\n';
		       s.width(i) ; s << "" << "timeout\n";
		       timeoutof()->print(s,i+2); s << '\n';
		       s.width(i) ; s << "" << "end";
		     };
  if ( nextof() ) { s << " ;\n"; nextof()->print(s,i); }
}

//
// Module
//

PRINT(ASTmodule) {
  s.width(i); s << "" << "module " << nameof() << " :\n";
  if ( declarationsof() ) declarationsof()->print(s,i+2);
  s << '\n';
  if ( instructionof() ) instructionof()->print(s,i+2);
  s << "\n";
  s.width(i); s << "" << ".\n";
  if ( nextof() ) nextof()->print(s,i);
}

//////////////////////////////
//
// The "semant" virtual function defined for all AST nodes
//

SEMANT(astnode) {
  yyerror("Internal error: semant() called on pure AST node", this);
}

SEMANT(ASTintlit) {}
SEMANT(ASTboollit) {}

SEMANT(ASTvariableid) {
  ASTdeclaration * d = variables.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("undeclared variable ", this, nameof() );
  else
    setdecl(d);
}

SEMANT(ASTsignalid) {
  ASTdeclaration * d = signals.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("undeclared signal ",this, nameof() );
  else {
    if ( d->etypeof() == PURE )
      yyerror("attempt to access value of pure signal ",this, nameof() );
    setdecl(d);
  }
}

SEMANT(ASTexceptionid) {
  ASTdeclaration * d = exceptions.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("undeclared exception ",this, nameof() );
  else {
    if ( d->etypeof() == PURE )
      yyerror("attempt to access value of pure exception ",this, nameof());
    setdecl(d);
  }
}

SEMANT(ASTbinop) {
  leftof()->semant();
  rightof()->semant();

  switch ( opof() ) {

  case opMULT: case opDIV: case opMOD: case opADD: case opSUB:
    if ( (leftof()->etypeof() != rightof()->etypeof()) ||
	 (leftof()->etypeof() != INTEGER ) )
      yyerror("operands of arithmetic operator must both be integer",this);
    setetype( INTEGER );
    break;

  case opLT: case opLE: case opGT: case opGE:
    if ( (leftof()->etypeof() != rightof()->etypeof()) ||
	 (leftof()->etypeof() != INTEGER ) )
      yyerror("operands of relational operator must both be integer",this);
    setetype( BOOLEAN );
    break;

  case opEQ: case opNE:
    if ( leftof()->etypeof() != rightof()->etypeof() )
      yyerror("operands of binary equality operator have mismatched types",this);
    setetype( BOOLEAN );
    break;

  case opAND: case opOR:
    if ( (leftof()->etypeof() != rightof()->etypeof() ) ||
	 (leftof()->etypeof() != BOOLEAN ) )
      yyerror("operands of boolean operator must both be boolean",this);
    setetype( BOOLEAN );
    break;

  }
}

SEMANT(ASTunop) {
  rightof()->semant();

  switch ( opof() ) {

  case opNEG:
    if ( rightof()->etypeof() != INTEGER )
      yyerror("operand of unary negation operator must be integer",this);
    setetype( INTEGER );
    break;

  case opNOT:
    if ( rightof()->etypeof() != BOOLEAN )
      yyerror("operand of unary not operator must be boolean",this);
    setetype( BOOLEAN );
    break;

  }
}

// Declarations add symbols to the current scope, but they
// may have expressions which are outside the current scope

SEMANT(ASTdeclaration) {

  ASTdeclaration * d;

  switch( flavorof() ) {
  case INPUT: case OUTPUT: case INPUTOUTPUT: case SENSOR: case LOCALSIG:
    if ( flavorof() == SENSOR && etypeof() == PURE )
      yyerror("untyped declaration of sensor ", this, nameof() );
    if ( ( d = signals.findlocalsym(nameof()) ) != 0 ) {
      yywarning("redeclaration of signal/sensor ", this, nameof() );
      yyerror(" originally declared here ", d);
    }
    signals.pushsym( this );
    
    setsd( new sd(SDsig, this) );

    break;

  case LOCALVAR:
    if ( ( d = variables.findlocalsym(nameof()) ) != 0 ) {
      yywarning("redeclaration of variable ", this, nameof() );
      yyerror(" originally declared here ", d);
    }
    if (expressionof()) {

      // initialization expressions are outside the scope of the new variables

      variables.pseudopop();
      expressionof()->semant();
      variables.pseudopush();

      if ( etypeof() != expressionof()->etypeof() ) {
	yyerror("type of initialization expression "
		"differs from type of variable ",this, nameof() );
      }
    }

    variables.pushsym( this );

    setsd( new sd(SDvar, this) );
    break;

  case EXCEPTION:
    /*
    if ( ( d = exceptions.findlocalsym(nameof()) ) != 0 ) {
      yywarning("redeclaration of exception ",this, nameof() );
      yyerror(" originally declared here", d);
    }
    */

    exceptions.pushsym( this );

    setsd( new sd(SDexc, this) );

    break;

  case NONE:
    yyerror("Internal error: semant() called on an unflavored declaration",
	this);
    break; 
  }

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTinstruction) {
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTassign) {
  ASTdeclaration * d = variables.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("undeclared rhs variable ", this, nameof() );
  else {
    setdecl(d);
    expressionof()->semant();
    if ( expressionof()->etypeof() != d->etypeof() ) {
      yywarning("lhs does not match type of rhs variable ",this, nameof() );
      yyerror(" originally declared here", d);
    }
    if ( nextof() ) nextof()->semant();
  }
}

SEMANT(ASTvariableblock) {
  variables.pushscope();
  variablesof()->semant();
  instructionof()->semant();
  variables.popscope();
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTsignalblock) {
  signals.pushscope();
  signalsof()->semant();
  instructionof()->semant();
  signals.popscope();
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASThandler) {
  ASTdeclaration * d = exceptions.findlocalsym(nameof());
  if ( d == 0 )
    yyerror("handle refers to undeclared exception ",this, nameof() );
  else {
    setdecl(d);
  }

  instructionof()->semant();
}

SEMANT(ASTexceptionblock) {
  exceptions.pushscope();

  exceptionof()->semant();
  instructionof()->semant();

  if ( handlerof() ) {
    handlerof()->semant();
    if ( handlerof()->declof() != exceptionof() ) {
      yywarning("handle clause handling trap ",
	      handlerof(), handlerof()->nameof() );
      yyerror(" instead of trap started here with exception ",
	      this, exceptionof()->nameof() );
    }
  }

  exceptions.popscope();

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTexit) {
  ASTdeclaration * d = exceptions.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("exit with undeclared exception ",this, nameof() );
  else {
    setdecl(d);
    if ( expressionof() ) {
      expressionof()->semant();
      if ( expressionof()->etypeof() != d->etypeof() ) {
	yywarning("type of expression does not match type of exception ",
		  this, nameof() );
	yyerror(" declaration of exception ", d, d->nameof() );
      }
    } else {
      if ( d->etypeof() != PURE )
	yyerror("exit with valued exception has no expression",this);
    }
  }

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTemit) {
  ASTdeclaration * d = signals.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("emitting undeclared signal ", this, nameof() );
  else {
    setdecl(d);
    if ( d->flavorof() == INPUT ||
	 d->flavorof() == SENSOR ) {
      yywarning("emitting an input-only signal or sensor ",this, nameof() );
      yyerror(" originally declared here",d);
    }
    if ( expressionof() ) {
      expressionof()->semant();
      if ( expressionof()->etypeof() != d->etypeof() ) {
	yywarning("type of expression does not match type of signal ",
		  this, nameof());
	yyerror(" originally declared here",d);
      }
    } else {
      if ( d->etypeof() != PURE ) {
	yywarning("no expression with emit of valued signal ",this, nameof() );
	yyerror(" originally declared here",d);
      }
    }
  }

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTpar) {
  instructionsof()->semant();
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTpar1) {
  instructionof()->semant();
  if ( nextof() ) nextof()->semant();		// another ASTpar1
}

SEMANT(ASTif) {
  expressionof()->semant();
  if ( expressionof()->etypeof() != BOOLEAN )
    yyerror("condition of if must be boolean",this);
  if ( thenof() ) thenof()->semant();
  if ( elseof() ) elseof()->semant();
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTpresent) {
  ASTdeclaration * d = signals.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("present checks undeclared signal ",this, nameof() );
  else {
    setdecl(d);
    if ( thenof() ) thenof()->semant();
    if ( elseof() ) elseof()->semant();
    if ( nextof() ) nextof()->semant();
  }
}

SEMANT(ASToccurrence) {
  ASTdeclaration * d = signals.findglobalsym(nameof());
  if ( d == 0 )
    yyerror("occurrence contains undeclared signal ",this, nameof() );
  else {
    setdecl(d);
    if ( occurtypeof() == COUNTED ) {
      expressionof()->semant();
      if ( expressionof()->etypeof() != INTEGER )
	yyerror("expression in counted occurrence must be integer",this);
      setcountsd( new sd(SDcntr, d) );
    }
  }
}

SEMANT(ASTawait) {
  occurrenceof()->semant();
  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTcase) {
  occurrenceof()->semant();
  if ( instructionof() ) instructionof()->semant();
  if ( nextof() ) nextof()->semant();		// more ASTcases
}

SEMANT(ASTawaitcase) {
  casesof()->semant();

  for ( ASTcase * c = casesof() ; c ; c = (ASTcase *) c->nextof() )
    for ( ASTcase * c1 = (ASTcase *) c->nextof() ;
	  c1 ;
	  c1 = (ASTcase *) c1->nextof() )
      if ( c->occurrenceof()->declof() == c1->occurrenceof()->declof() )
	yyerror("multiple cases containing signal ", c1->occurrenceof(),
		c1->occurrenceof()->declof()->nameof() );

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTloop) {
  instructionof()->semant();

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTrepeat) {
  expressionof()->semant();
  if ( expressionof()->etypeof() != INTEGER )
    yyerror("repetition count of repeat must be integer",this);
  instructionof()->semant();

  setrepvar( new sd(SDvar) );

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTdowatching) {
  instructionof()->semant();
  if ( timeoutof() ) timeoutof()->semant();
  occurrenceof()->semant();

  if ( nextof() ) nextof()->semant();
}

SEMANT(ASTmodule) {
  signals.pushscope();
  if ( declarationsof() ) declarationsof()->semant();
  instructionof()->semant();
  signals.popscope();

  if ( nextof() ) nextof()->semant();
}

//////////////////////////////
//
// The "require" virtual function for ASTexpression nodes
//
// Each is called with the process being assembled "s"
// and adds a require statement for every signal in that expression
//

REQUIRE(ASTexpression) {}
REQUIRE(ASTintlit) {}
REQUIRE(ASTboollit) {}
REQUIRE(ASTvariableid) {}
REQUIRE(ASTsignalid) { r.addsignal(declof()->sdof()); }
REQUIRE(ASTexceptionid) {}
REQUIRE(ASTbinop) { leftof()->require(r); rightof()->require(r); }
REQUIRE(ASTunop) { rightof()->require(r); }

//////////////////////////////
//
// The "translate" virtual function defined for AST nodes
//
// Each is called with the process being assembled "s"
// and returns a destination of its result (used for expressions)

sd * translateexpr( amprocess & s, ASTexpression * e)
{
  AMrequire * r = new AMrequire;
  e->require(*r);
  if ( r->n_signals() )
    s , r;
  else
    delete r;

  return e->translate(s);
}

TRANSLATE(astnode) {
  return 0;
}

TRANSLATE(ASTdeclaration) {
  if ( expressionof() ) {

    sd * result = translateexpr(s,expressionof());

    s , new AMassign( sdof(), result );
  }
  
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTintlit) {
  sd * dest = new sd( SDreg, reg );
  s , new AMassign( dest, new sd( SDconst, valueof() ) );
  return dest;
}

TRANSLATE(ASTboollit) {
  sd * dest = new sd( SDreg, reg );
  s , new AMassign( dest, new sd( SDconst, valueof() ) );
  return dest;
}

TRANSLATE(ASTvariableid) {
  sd * dest = new sd( SDreg, reg );
  s , new AMassign( dest, declof()->sdof() );
  return dest;
}

TRANSLATE(ASTsignalid) {
  sd * dest = new sd( SDreg, reg );
  s , new AMassign( dest, declof()->sdof() );
  return dest;
}

TRANSLATE(ASTexceptionid) {
  sd * dest = new sd( SDreg, reg );
  s , new AMassign( dest, declof()->sdof() );
  return dest;
}

TRANSLATE(ASTbinop) {
  sd * src1 = leftof()->translate(s,reg);
  sd * src2 = rightof()->translate(s,reg+1);

  sd * tmp;

  sd * dest = new sd( SDreg, reg );

  amop o;

  switch( opof() ) {
  case opMULT: o = amMUL; break;
  case opDIV: o = amDIV; break;
  case opMOD: o = amMOD; break;
  case opADD: o = amADD; break;
  case opSUB: o = amSUB; break;
  case opLT: o = amLT; break;
  case opGT: o = amLT; tmp = src1; src1 = src2; src2 = tmp; break;
  case opLE: o = amLE; break;
  case opGE: o = amLE; tmp = src1; src1 = src2; src2 = tmp; break;
  case opEQ: o = amEQ; break;
  case opNE: o = amNE; break;
  case opAND: o = amAND; break;
  case opOR: o = amOR; break;
  default: o = amASSIGN; break;
  }

  s , new AMassign( dest, src1, o, src2 );

  return dest;
}

TRANSLATE(ASTunop) {
  sd * src = rightof()->translate(s,reg);

  sd * dest = new sd( SDreg, reg );

  amop o;
  switch( opof() ) {
  case opNEG: o = amNEG; break;
  case opNOT: o = amNOT; break;
  default: o = amASSIGN; break;
  }

  s , new AMassign( dest, src, o );

  return dest;

}

TRANSLATE(ASTnothing) {
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASThalt) {
  s , new AMhalt;
  return 0;
}

TRANSLATE(ASTassign) {
  sd * src = translateexpr(s,expressionof());
  s , new AMassign( declof()->sdof(), src );
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTvariableblock) {
  variablesof()->translate(s);
  instructionof()->translate(s);
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTsignalblock) {
  instructionof()->translate(s);
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

// trap e1 in i1 handle e2 i2 end
//  translates to
//
//   try
//    i1
//   handle e1 goto 1:
//   goto 2:
// 1: i2
// 2:

TRANSLATE(ASTexceptionblock) {

  // we always need a try

  AMtry * thetry = new AMtry;

  // add a branch of there is a handler

  AMbranch * bran = handlerof() ? new AMbranch : 0;

  // add the try and the branch to the process

  s , thetry , bran;

  // add the instruction to the collection of subprocesses

  amprocess * s1 = new amprocess; 
  instructionof()->translate(*s1);
  s1->addexit();
  thetry->appendsubprocess( s1 );

  thetry->appendhandle( new AMhandle( exceptionof()->sdof(), NEXTINSTTARGET ) );

  if ( handlerof() )
    handlerof()->instructionof()->translate(s);

  if ( bran )
    bran->settarget( NEXTINSTTARGET );

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTexit) {
  sd * dest = 0;
  if ( expressionof() ) {
    dest = translateexpr(s,expressionof());
  }

  s , new AMexit( declof()->sdof(), dest );

  return 0;
}

TRANSLATE(ASTemit) {
  sd * dest = 0;
  if ( expressionof() )
    dest = translateexpr(s,expressionof());

  s , new AMemit( declof()->sdof(), dest );

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTpar) {

  amprocess * s1;
  AMtry * thetry = new AMtry;

  s , thetry;

  for ( ASTpar1 * i = instructionsof() ; i ; i = (ASTpar1 *) i->nextof() ) {
    s1 = new amprocess;
    i->instructionof()->translate(*s1);
    s1->addexit();
    thetry->appendsubprocess( s1 );
  }

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTif) {

  sd * dest = translateexpr(s,expressionof());

  AMbranch * test;
  AMbranch * bran;

  if ( !elseof() && thenof() ) {
    s , (test = new AMbranch( 1, dest ));
    thenof()->translate(s);
    test->settarget( NEXTINSTTARGET );
  }

  if ( elseof() && !thenof() ) {
    s , (test = new AMbranch( 0, dest ));
    elseof()->translate(s);
    test->settarget( NEXTINSTTARGET );
  }

  if ( elseof() && thenof() ) {
    s, (test = new AMbranch( 1, dest ));
    thenof()->translate(s);
    s , (bran = new AMbranch);
    test->settarget( NEXTINSTTARGET );
    elseof()->translate(s);
    bran->settarget( NEXTINSTTARGET );
  }

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTpresent) {

  sd * sig = declof()->sdof();

  AMrequire * r = new AMrequire;
  r->addsignal(sig);

  s , r;

  AMbranch * test;
  AMbranch * bran;

  if ( !elseof() && thenof() ) {
    s , (test = new AMbranch( 1, sig ));
    thenof()->translate(s);
    test->settarget( NEXTINSTTARGET );
  }

  if ( elseof() && !thenof() ) {
    s , (test = new AMbranch( 0, sig ));
    elseof()->translate(s);
    test->settarget( NEXTINSTTARGET );
  }

  if ( elseof() && thenof() ) {
    s, (test = new AMbranch( 1, sig ));
    thenof()->translate(s);
    s , (bran = new AMbranch);
    test->settarget( NEXTINSTTARGET );
    elseof()->translate(s);
    bran->settarget( NEXTINSTTARGET );
  }
  
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTawait) {

  amprocess * hltseq = new amprocess;
  *hltseq , new AMhalt;

  AMtry * thetry = new AMtry;
  thetry->appendsubprocess(hltseq);

  switch( occurrenceof()->occurtypeof() ) {
  case SIMPLE:
    s , thetry;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET ) );
    break;

  case COUNTED:
    sd * count = translateexpr( s, occurrenceof()->expressionof() );
    s , new AMassign( occurrenceof()->countsdof(), count );
    s , thetry;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET,
				      occurrenceof()->countsdof() ) );
    break;

  case IMMEDIATE:
    AMbranch * bran = new AMbranch(0, occurrenceof()->declof()->sdof());
    AMrequire * req = new AMrequire;
    req->addsignal( occurrenceof()->declof()->sdof() );

    s , req , bran , thetry;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET ) );
    bran->settarget( NEXTINSTTARGET );
    break;
  }

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTawaitcase) {

  int i;

  ASTcase * c;
  sd * count;

  array<AMbranch *> immediatebranches;
  array<AMbranch *> watchbranches;

  AMtry * thetry = new AMtry;

  amprocess * hltseq = new amprocess;
  *hltseq , new AMhalt;
  hltseq->addexit();
  thetry->appendsubprocess(hltseq);

  // go through all the cases, adding any signals which are immediate to
  // a leading require

  AMrequire * r = new AMrequire;
  for ( c = casesof() ; c ; c = (ASTcase *) c->nextof() )
    if ( c->occurrenceof()->occurtypeof() == IMMEDIATE )
      r->addsignal( c->occurrenceof()->declof()->sdof() );

  if ( r->n_signals() )
    s , r;
  else
    delete r;

  // go through all the cases, adding a leading branch whenever there is
  // an immediate occurrence

  for( i=0, c = casesof() ; c ; i++, c = (ASTcase *) c->nextof() ) {
    if ( c->occurrenceof()->occurtypeof() == IMMEDIATE )
      s , ( immediatebranches[i] =
	    new AMbranch(0, c->occurrenceof()->declof()->sdof() ) );
    else
      immediatebranches[i] = 0;
  }

  // go through all the cases, adding a counter load whenever there is a
  // counted occurrence

  for ( c = casesof() ; c ; c = (ASTcase *) c->nextof() ) {
    if ( c->occurrenceof()->occurtypeof() == COUNTED ) {
      count = translateexpr( s, c->occurrenceof()->expressionof() );
      s , new AMassign( c->occurrenceof()->countsdof(), count );
    }
  }

  // add the try instruction

  s , thetry;

  // add the watches and the case instructions

  for( i=0, c = casesof() ; c ; i++, c = (ASTcase *) c->nextof() ) {

    if ( c->instructionof() )
      s , ( watchbranches[watchbranches.size()] = new AMbranch );

    switch ( c->occurrenceof()->occurtypeof() ) {
    case SIMPLE:

      if( c->instructionof() ) {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof(),
					  NEXTINSTTARGET ) );
	c->instructionof()->translate(s);
      } else {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof() ) );
      }

      break;

    case COUNTED:
      if ( c->instructionof() ) {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof(),
					  NEXTINSTTARGET,
					  c->occurrenceof()->countsdof() ) );
	c->instructionof()->translate(s);
      } else {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof(),
					  0,
					  c->occurrenceof()->countsdof() ) );
      }

      break;

    case IMMEDIATE:
      if ( c->instructionof() ) {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof(),
					  NEXTINSTTARGET ) );
	immediatebranches[i]->settarget( NEXTINSTTARGET );
	c->instructionof()->translate(s);
      } else {
	thetry->appendwatch( new AMwatch( c->occurrenceof()->declof()->sdof() ));
      }	
    }
  }


  // If any of the watch clauses in the try instruction did not have
  // targets, then they did not have instructions and should branch here

  for ( i=thetry->n_watches() ; --i >= 0 ; )
    if ( thetry->watch(i)->targetof() == NOTARGET )
      thetry->watch(i)->settarget( NEXTINSTTARGET );

  // all of the branches required by the instructions also need to have
  // their targets set here---so do it

  for ( i = watchbranches.size() ; --i >= 0 ; )
    watchbranches[i]->settarget( NEXTINSTTARGET );

  // all of immediate-induced branches which have no target must branch here

  for( i = immediatebranches.size() ; --i >= 0 ; )
    if ( immediatebranches[i] && immediatebranches[i]->targetof() == NOTARGET )
      immediatebranches[i]->settarget( NEXTINSTTARGET );

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTloop) {
  AMbranch * bran = new AMbranch;
  bran->settarget( NEXTINSTTARGET );

  instructionof()->translate(s);
  s , bran;

  return 0;
}

TRANSLATE(ASTrepeat) {
  
  sd * count = translateexpr(s, expressionof() );
  s , new AMassign( repvarof(), count );

  sd * tmpreg = new sd( SDreg, 0 );

  AMbranch * bran = new AMbranch(0, repvarof(), NEXTINSTTARGET );

  instructionof()->translate(s);

  // load the repvar into the temporary register, decrease it,
  // and store the result before using it to decide the branch

  s , new AMassign( tmpreg, repvarof() )
    , new AMassign( tmpreg, tmpreg, amDEC )
    , new AMassign( repvarof(), tmpreg )
    , bran;

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTdowatching) {

  // we always need a try

  AMtry * thetry = new AMtry;

  // If there is a timeout clause, we need a branch after the try

  AMbranch * bran = timeoutof() ? new AMbranch : 0;

  // There are three cases, depending on the type of the occurrence:

  switch( occurrenceof()->occurtypeof() ) {

    // do nothing watching A timeout nothing end
    //
    // 0: try
    //      nothing
    //    watching A goto 1:
    //    goto 2:
    // 1: nothing
    // 2:
    
  case SIMPLE:
    s , thetry , bran;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET ));
    break;

    // do nothing watching 5 A timeout nothing end
    //
    //  c1 := ...
    //    try
    //     nothing
    //    watching A c1 goto 1:
    //    goto 2:
    // 1: nothing
    // 2:

  case COUNTED:
    sd * count = translateexpr( s, occurrenceof()->expressionof() );
    s , new AMassign( occurrenceof()->countsdof(), count );
    s , thetry , bran;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET,
				      occurrenceof()->countsdof() ) );
    break;

    // do nothing watching immediate A timeout nothing end
    //
    //    if present A goto 1:
    //    try
    //      nothing
    //    watching A goto 1:
    //    goto 2:
    // 1: nothing
    // 2:

  case IMMEDIATE:
    AMbranch * bran0 = new AMbranch(0, occurrenceof()->declof()->sdof() );
    AMrequire * req = new AMrequire;
    req->addsignal( occurrenceof()->declof()->sdof() );
    s , req , bran0 , thetry , bran;
    thetry->appendwatch( new AMwatch( occurrenceof()->declof()->sdof(),
				      NEXTINSTTARGET ) );
    bran0->settarget( NEXTINSTTARGET );
    break;
  }

  // Make the instruction of the "do..watching" the subprocess for the try

  amprocess * s1 = new amprocess;
  instructionof()->translate(*s1);
  s1->addexit();
  thetry->appendsubprocess( s1 );

  // if there is a timeout instruction, place it after all of this

  if ( timeoutof() )
    timeoutof()->translate(s);

  if ( bran )
    bran->settarget( NEXTINSTTARGET );

  if ( nextof() ) nextof()->translate(s);
  return 0;
}

TRANSLATE(ASTmodule) {
  instructionof()->translate(s);
  if ( nextof() ) nextof()->translate(s);
  return 0;
}

//////////////////////////////
//
// class functions for the sd class
//

// static variable definitions

array<sd *> sd::variables;
array<sd *> sd::signals;
array<sd *> sd::counters;
array<sd *> sd::exceptions;
int sd::valuedsignals = 0;
int sd::valuedexceptions = 0;

sd::sd( sdtype t, ASTdeclaration * d = 0 )
{
  type = t;
  declaration = d;
  arg2 = -1;
  switch ( type ) {
  case SDvar:
    arg = variables.size();
    variables[arg] = this;
    break;
  case SDsig:
    arg = signals.size();
    signals[arg] = this;
    if ( d->etypeof() != PURE )
      arg2 = valuedsignals++;
    break;
  case SDcntr:
    arg = counters.size();
    counters[arg] = this;
    break;
  case SDexc:
    arg = exceptions.size();
    exceptions[arg] = this;
    if ( d->etypeof() != PURE )
      arg2 = valuedexceptions++;
    break;
  default:
    break;
  }

}

sd::sd( sdtype t, int a )
{
  type = t;
  arg = a;
  arg2 = 0;
  declaration = 0;
}

void sd::printsd( ostream & s )
{
  int i;
  s << "Local variables " << variables.size() << '\n';
  for ( i = 0 ; i < variables.size() ; i++ ) {
    s << 'v' << i << ": ";
    if ( variables[i]->declaration )
      s << variables[i]->declaration->nameof();
    s << '\n';
  }

  s << "Signals " << signals.size()
    << " ( " << valuedsignals << " valued )\n";
  for ( i = 0 ; i < signals.size() ; i++ ) {
    s << 's' << i << ": ";
    if (signals[i]->declaration)
      s << signals[i]->declaration->nameof();
    if ( signals[i]->arg2of() >= 0 )
      s << "(int)";
    s << '\n';
  }

  s << "Counters " << counters.size() << '\n';
    for ( i = 0 ; i < counters.size() ; i++ ) {
      s << 'c' << i << ": ";
      if ( counters[i]->declaration )
	s << counters[i]->declaration->nameof();
      s << '\n';
    }

  s << "Exceptions " << exceptions.size()
    << " ( " << valuedexceptions << " valued )\n";
  for ( i = 0 ; i < exceptions.size() ; i++ ) {
    s << 'e' << i << ": ";
    if ( exceptions[i]->declaration)
      s << exceptions[i]->declaration->nameof();
    if ( exceptions[i]->arg2of() >= 0 )
      s << "(int)";
    s << '\n';
  }

}
