// -*- C++ -*-

// Abstract Machine-related classes for the Esterel compiler

// Stephen Edwards

//////////////////////////////
//
// Forward class definitions
//

class amnode;
class amprocess;

class AMexit;
class AMhalt;
class AMassign;
class AMrequire;
class AMemit;
class AMbranch;
class AMwatch;
class AMhandle;
class AMtry;


//////////////////////////////
//
// Class definitions
//

enum amop { amASSIGN, amNEG, amNOT, amDEC,	// unary operators
	      amAND, amOR, amEQ, amNE, amLT, amLE,
	      amADD, amSUB, amMUL, amDIV, amMOD };

// Prints the node to the stream with the given indentation
#define AMPRINTFUNC void print( ostream & s )

// Prints the assembly-language manifestation of the node
#define AMASMFUNC void asmnode( ostream & s )

// Forms the potential set of a non-try instruction
#define AMPOFUNC void potentialof( setc & s )

#define AMNODEFUNCS AMPRINTFUNC; AMASMFUNC; AMPOFUNC

// Returns the number of program counters used by the instruction/process
#define COUNTPCFUNC int countpc( int pcin = 0 , int pcmax = 0 )

// Returns true when the node can terminate
#define TERMINATESFUNC int terminates()


// an instruction node

class amnode {
  int index;
  amprocess * process;
public:
  amnode() { index = -1; process = 0; }

  void setindex( int i ) { index = i; }
  int indexof() { return index; }

  void setprocess( amprocess * p ) { process = p; }
  amprocess * processof() { return process; }

  int processnumber();

  virtual AMPRINTFUNC;
  virtual AMASMFUNC;
  virtual AMPOFUNC;
  virtual COUNTPCFUNC { return pcmax; }
  virtual TERMINATESFUNC { return 1; }

  amnode * nextinst();

  int pcof();

friend
  ostream & operator << ( ostream & s, amnode & n );
};

// a sequence of instruction nodes, stored as an array

class amprocess : public array<amnode *> {

  amnode * parent;
  int number;
  int pc;

  static int pccount;		// global count of program counters

  static array<amprocess *> processes;	// global collection of all processes

  void newpc( int pcin ) { if ( pcin > pccount ) pccount = pcin; }

public:
  amprocess() { parent = 0;
		 number = processes.size();
		 processes[number] = this;
	       }

  void attach( amnode * p ) { parent = p; }

  void setpc(int p) { pc = p; }
  int pcof() { return pc; }

  amprocess & operator , ( amnode * n ) {
    if ( n ) {
      n->setindex(size());
      n->setprocess(this);
      (*this)[size()] = n;
    }
    return *this;
  }

  // index of next amnode in the process

  int nextindex() { return size(); }

  // parent (try instruction of) the process

  amnode * parentof() { return parent; }

  // global process number

  int processnumber() { return number; }

  // global number of processes in existence

  int n_processes() { return processes.size(); }

  // access to global process

  amprocess * process(int i) { return processes[i]; }

  // used to count the number of program counters required overall

  COUNTPCFUNC;

  // total number of program counters used by the entire program
  int n_pcs() { return pccount + 1; }

  // Adds an exit if the last instruction does not terminte

  void addexit();

  // Assembles this subprocess to the given output stream

  void assemble( ostream & s );

  // prints a one-line process description

  void printheader( ostream & s );

friend
  ostream & operator << ( ostream & s, amprocess & se );

};

class AMexit : public amnode {
  sd * excep;
  sd * reg;
public:
  AMexit( sd * e = 0, sd * r = 0 ) { excep = e; reg = r; }
  sd * exceptionof() { return excep; }
  sd * registerof() { return reg; }
  AMNODEFUNCS;
  TERMINATESFUNC { return 0; }
};

class AMhalt : public amnode {
public:
  AMNODEFUNCS;
  TERMINATESFUNC { return 0; }
};

class AMassign : public amnode {
  sd * dest;
  sd * src1;
  sd * src2;
  amop op;
public:
  AMassign( sd * d, sd * s1, amop o = amASSIGN, sd * s2 = 0) {
    dest = d;
    src1 = s1;
    src2 = s2;
    op = o;
  }

  sd * destof() { return dest; }
  sd * src1of() { return src1; }
  sd * src2of() { return src2; }
  amop opof() { return op; }

  AMNODEFUNCS;

};

class AMrequire : public amnode {
  array<sd *> signals;
public:
  int n_signals() { return signals.size(); }

  sd * signal(int i) { return signals[i]; }

  void addsignal(sd * s); 

  AMNODEFUNCS;
};

class AMemit : public amnode {
  sd * signal;
  sd * reg;
public:
  AMemit( sd * s, sd * r = 0 ) { signal = s; reg = r; }

  sd * signalof() { return signal; }
  sd * registerof() { return reg; }

  AMNODEFUNCS;

};

class AMbranch : public amnode {
  int sense;		// 0 = if non-zero, 1 = if zero
  sd * test;		// thing to test (signal, exception, etc)
  TARGET target;	// target of this branch
public:
  AMbranch( int s = 0, sd * tst = 0, TARGET targ = NOTARGET ) {
    sense = s;
    test = tst;
    target = targ;
  }

  int senseof() { return sense; }
  void settarget( TARGET t ) { target = t; }
  sd * testof() { return test; }
  TARGET targetof() { return target; }

  AMNODEFUNCS;
  TERMINATESFUNC { return test != 0; }
};

class AMwatch {
  sd * signal;
  TARGET target;
  sd * counter;
public:
  AMwatch( sd * s, TARGET t = NOTARGET, sd * c = 0 ) { signal = s;
						       target = t;
						       counter = c; }
  void settarget(TARGET t) { target = t; }
  sd * signalof() { return signal; }
  TARGET targetof() { return target; }
  sd * counterof() { return counter; }

  AMPRINTFUNC;
};

class AMhandle {
  sd * exception;
  TARGET target;
public:
  AMhandle( sd * e, TARGET t ) { exception = e; target = t; }

  sd * exceptionof() { return exception; }
  TARGET targetof() { return target; }

  AMPRINTFUNC;
};

class AMtry : public amnode {
  array<amprocess *> subprocesses;
  array<AMwatch *> watches;
  array<AMhandle *> handles;
public:
  void appendsubprocess( amprocess * s ) {
    subprocesses[subprocesses.size()] = s;
    s->attach(this);
  }

  void appendwatch( AMwatch * w ) {
    watches[watches.size()] = w;
  }

  void appendhandle( AMhandle * h ) {
    handles[handles.size()] = h;
  }

  amprocess * subprocess( int i ) { return subprocesses[i]; }
  AMwatch * watch( int i ) { return watches[i]; }
  AMhandle * handle( int i ) { return handles[i]; }

  int n_subprocesses() { return subprocesses.size(); }
  int n_watches() { return watches.size(); }
  int n_handles() { return handles.size(); }
  
  AMNODEFUNCS;
  COUNTPCFUNC;
};

inline
int amnode::processnumber() { return process->processnumber(); }

inline
amnode * amnode::nextinst() { return (*process)[index+1]; }

inline
int amnode::pcof() { return process->pcof(); }
