module Main where

import System.Environment (getArgs)
import Text.Read (readMaybe)

import MCTS
import TicTacToe
import Go5x5


defaultWorkers :: Int
defaultWorkers = 8

defaultIters :: Int
defaultIters = 20000

defaultChunkSize :: Int
defaultChunkSize = 1

main :: IO ()
main = do
  args <- getArgs

  case args of
    ("ttt":rest) -> runTTT rest
    ("go":rest)  -> runGo rest
    _            -> usage

-- TicTacToe

runTTT :: [String] -> IO ()
runTTT rest = do
  let (workers, chunk, iters) = parseParams rest
  putStrLn $ "[Game] Tic-Tac-Toe"
  putStrLn $ "[Config] workers=" ++ show workers ++
             ", chunk_size=" ++ show chunk ++
             ", iters/worker=" ++ show iters

  best <- mctsBestMoveParallel workers chunk iters emptyTTT
  putStrLn $ "Best move (0..8): " ++ show best

-- Go 5 x 5

runGo :: [String] -> IO ()
runGo rest = do
  let (workers, chunk, iters) = parseParams rest
  putStrLn $ "[Game] Go 5x5"
  putStrLn $ "[Config] workers=" ++ show workers ++
             ", chunk_size=" ++ show chunk ++
             ", iters/worker=" ++ show iters

  best <- mctsBestMoveParallel workers chunk iters emptyGo
  putStrLn $ "Best move: " ++ show best



parseParams :: [String] -> (Int, Int, Int)
parseParams xs =
  case xs of
    (a:b:c:_) ->
      ( readDef defaultWorkers a
      , readDef defaultChunkSize b
      , readDef defaultIters c
      )
    (a:b:_) ->
      ( readDef defaultWorkers a
      , readDef defaultChunkSize b
      , defaultIters
      )
    (a:_) ->
      ( readDef defaultWorkers a
      , defaultChunkSize
      , defaultIters
      )
    _ ->
      ( defaultWorkers
      , defaultChunkSize
      , defaultIters
      )

readDef :: Int -> String -> Int
readDef def s =
  case readMaybe s of
    Just x  -> x
    Nothing -> def

usage :: IO ()
usage = do
  putStrLn "Usage:"
  putStrLn "  ./Main [game] [workers] [chunk] [iters]"
  putStrLn ""
  putStrLn "Arguments:"
  putStrLn "  game    - 'ttt' (Tic-Tac-Toe) or 'go' (Go 5x5)"
  putStrLn "  workers - Number of parallel workers (default: 8)"
  putStrLn "  chunk   - Chunk size for batching (default: 2)"
  putStrLn "  iters   - Iterations per worker (default: 20000)"
  putStrLn ""
  putStrLn "Examples:"
  putStrLn "  ./Main ttt 8 2 10000"
  putStrLn "  ./Main go 4 1 20000 +RTS -N4 -s"
