{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE DeriveAnyClass #-}

module Types where

import GHC.Generics (Generic)
import Control.DeepSeq (NFData)
import qualified Data.Vector as V

-- | Cell states in the simulation
data CellState = Unburned | Burning | Burned
  deriving (Eq, Show, Generic, NFData)

-- | Terrain types affecting fire behavior
data TerrainType = Forest | Grassland | Water | Urban
  deriving (Eq, Show, Generic, NFData)

-- | Properties of each grid cell
data Cell = Cell
  { cellState      :: !CellState    -- Current state
  , cellPosition   :: !(Int, Int)   -- Position in grid (row, col)
  , fuelLevel      :: !Double       -- 0.0 (no fuel) to 1.0 (max fuel)
  , terrainType    :: !TerrainType  -- Type of terrain
  , elevation      :: !Double       -- Elevation for slope calculations
  , burnSteps      :: !Int          -- How many steps this cell has been burning
  } deriving (Eq, Show, Generic, NFData)

-- | Grid is a 2D vector of cells
type Grid = V.Vector (V.Vector Cell)

-- | Wind configuration
data Wind = Wind
  { windSpeed     :: !Double  -- 0.0 to 1.0
  , windDirection :: !Double  -- 0 to 360 degrees (0 = North, 90 = East)
  } deriving (Eq, Show, Generic, NFData)

-- | Simulation configuration
data SimConfig = SimConfig
  { gridRows        :: !Int
  , gridCols        :: !Int
  , wind            :: !Wind
  , maxTimeSteps    :: !Int
  , ignitionPoints  :: ![(Int, Int)]  -- Starting fire locations
  , baseProbability :: !Double        -- Base ignition probability
  , ignitionThreshold :: !Double      -- Threshold for ignition
  } deriving (Show, Generic, NFData)

-- | Statistics tracked during simulation
data SimStats = SimStats
  { timeStep       :: !Int
  , burningCells   :: !Int
  , burnedCells    :: !Int
  , totalBurned    :: !Int
  } deriving (Show, Generic, NFData)

-- | Complete simulation state
data SimState = SimState
  { simGrid   :: !Grid
  , simConfig :: !SimConfig
  , simStats  :: !SimStats
  } deriving (Show, Generic, NFData)