module Serial
  ( runSimulation
  , simulationStep
  , igniteInitialPoints
  ) where

import Types
import Grid
import FireRules
import qualified Data.Vector as V
import System.Random (mkStdGen, RandomGen, split)
import Control.DeepSeq (force)

-- | Run the complete simulation
runSimulation :: SimConfig -> Grid -> (Grid, [SimStats])
runSimulation config initialGrid =
  let gridWithFire = igniteInitialPoints (ignitionPoints config) initialGrid
      gen = mkStdGen 42  -- Fixed seed for reproducibility
  in runSteps config gridWithFire gen 0 []

-- | Ignite initial fire points
igniteInitialPoints :: [(Int, Int)] -> Grid -> Grid
igniteInitialPoints points grid =
  foldl ignitePoint grid points
  where
    ignitePoint g pos =
      case getCell g pos of
        Nothing -> g
        Just cell -> setCell g pos (cell { cellState = Burning, burnSteps = 0 })

-- | Run simulation steps recursively
runSteps :: RandomGen g => SimConfig -> Grid -> g -> Int -> [SimStats] -> (Grid, [SimStats])
runSteps config grid gen step stats
  | step >= maxTimeSteps config = (grid, reverse stats)
  | otherwise =
      let burning = countCellsByState grid Burning
          burned = countCellsByState grid Burned
          currentStats = SimStats step burning burned burned
          stats' = currentStats : stats
      in if burning == 0 && step > 0
         then (grid, reverse stats')  -- Fire extinguished
         else
           let (grid', gen') = simulationStep config grid gen
           in runSteps config grid' gen' (step + 1) stats'

-- | Perform one simulation step
simulationStep :: RandomGen g => SimConfig -> Grid -> g -> (Grid, g)
simulationStep config grid gen =
  let rows = V.length grid
      cols = if rows > 0 then V.length (grid V.! 0) else 0
      (newGrid, genFinal) = processAllCells config grid gen rows cols
  in (force newGrid, genFinal)

-- | Process all cells in the grid
processAllCells :: RandomGen g => SimConfig -> Grid -> g -> Int -> Int -> (Grid, g)
processAllCells config grid gen rows cols =
  foldl (processRow config grid cols) (grid, gen) [0..rows-1]

-- | Process all cells in a row
processRow :: RandomGen g => SimConfig -> Grid -> Int -> (Grid, g) -> Int -> (Grid, g)
processRow config originalGrid cols (currentGrid, gen) row =
  foldl (processCell config originalGrid (row, cols)) (currentGrid, gen) [0..cols-1]

-- | Process a single cell
processCell :: RandomGen g => SimConfig -> Grid -> (Int, Int) -> (Grid, g) -> Int -> (Grid, g)
processCell config originalGrid (row, cols) (currentGrid, gen) col =
  let pos = (row, col)
  in case getCell originalGrid pos of
    Nothing -> (currentGrid, gen)
    Just cell ->
      case cellState cell of
        -- Unburned cells might ignite
        Unburned ->
          if fuelLevel cell <= 0.0 || terrainType cell == Water
          then (currentGrid, gen)
          else
            let neighbors = getNeighbors8 originalGrid pos
                prob = calculateIgnitionProbability config cell neighbors originalGrid
                (ignites, gen') = shouldIgnite config prob gen
            in if ignites
               then (setCell currentGrid pos (cell { cellState = Burning, burnSteps = 0 }), gen')
               else (currentGrid, gen')
        
        -- Burning cells progress toward burned
        Burning ->
          let maxBurnSteps = case terrainType cell of
                Forest    -> 3
                Grassland -> 2
                _         -> 1
              updatedCell = updateCellState cell maxBurnSteps
          in (setCell currentGrid pos updatedCell, gen)
        
        -- Burned cells stay burned
        Burned -> (currentGrid, gen)