module Main where

import Types
import Grid
import Scenarios
import Serial
import Parallel
import System.CPUTime (getCPUTime)
import Text.Printf (printf)
import System.Environment (getArgs)

main :: IO ()
main = do
  args <- getArgs
  case args of
    ["serial"] -> runSerialTests
    ["parallel"] -> runParallelTests
    ["benchmark"] -> runBenchmarks
    _ -> runAllTests

-- | Run all tests (default)
runAllTests :: IO ()
runAllTests = do
  putStrLn "=== Wildfire Simulation - Phase 3: Parallel Implementation ==="
  putStrLn ""
  runSerialDemo
  putStrLn ""
  runParallelDemo
  putStrLn ""
  runBenchmarks

-- | Serial demonstration
runSerialDemo :: IO ()
runSerialDemo = do
  putStrLn "SERIAL IMPLEMENTATION"
  putStrLn $ replicate 60 '='
  putStrLn ""
  
  putStrLn "Test: Uniform Forest (20x20)"
  let scenario = uniformForestScenario 20 20
      config = scenarioConfig scenario
      grid = scenarioGrid scenario
      (finalGrid, stats) = runSimulation config grid
      finalStats = last stats
  
  putStrLn $ "Completed in " ++ show (timeStep finalStats) ++ " steps"
  putStrLn $ "Total burned: " ++ show (totalBurned finalStats) ++ " cells"

-- | Parallel demonstration
runParallelDemo :: IO ()
runParallelDemo = do
  putStrLn "PARALLEL IMPLEMENTATIONS"
  putStrLn $ replicate 60 '='
  putStrLn ""
  
  let scenario = uniformForestScenario 20 20
      config = scenarioConfig scenario
      grid = scenarioGrid scenario
  
  putStrLn "1. Spatial Decomposition:"
  let (finalGrid1, stats1) = runSimulationParallel config grid
      finalStats1 = last stats1
  putStrLn $ "   Completed in " ++ show (timeStep finalStats1) ++ " steps"
  putStrLn $ "   Total burned: " ++ show (totalBurned finalStats1) ++ " cells"
  putStrLn ""
  
  putStrLn "2. Red-Black Checkerboard:"
  let (finalGrid2, stats2) = runSimulationRedBlack config grid
      finalStats2 = last stats2
  putStrLn $ "   Completed in " ++ show (timeStep finalStats2) ++ " steps"
  putStrLn $ "   Total burned: " ++ show (totalBurned finalStats2) ++ " cells"

-- | Run only serial tests
runSerialTests :: IO ()
runSerialTests = do
  putStrLn "=== Serial Implementation Only ==="
  putStrLn ""
  let scenario = uniformForestScenario 50 50
  benchmarkScenario "Serial" (runSimulation) scenario

-- | Run only parallel tests
runParallelTests :: IO ()
runParallelTests = do
  putStrLn "=== Parallel Implementations Only ==="
  putStrLn ""
  let scenario = uniformForestScenario 50 50
  
  putStrLn "Spatial Decomposition:"
  benchmarkScenario "Parallel-Spatial" runSimulationParallel scenario
  putStrLn ""
  
  putStrLn "Red-Black Checkerboard:"
  benchmarkScenario "Parallel-RedBlack" runSimulationRedBlack scenario

-- | Comprehensive benchmarks
runBenchmarks :: IO ()
runBenchmarks = do
  putStrLn ""
  putStrLn $ replicate 70 '='
  putStrLn "PERFORMANCE BENCHMARKS"
  putStrLn $ replicate 70 '='
  putStrLn ""
  
  -- Small-Medium grid
  putStrLn "Grid Size: 100x100"
  putStrLn $ replicate 70 '-'
  let small = uniformForestScenario 100 100
  runComparison small
  
  putStrLn ""
  
  -- Medium grid
  putStrLn "Grid Size: 200x200"
  putStrLn $ replicate 70 '-'
  let medium = uniformForestScenario 200 200
  runComparison medium
  
  putStrLn ""
  
  -- Large grid
  putStrLn "Grid Size: 500x500"
  putStrLn $ replicate 70 '-'
  let large = uniformForestScenario 500 500
  runComparison large

-- | Run comparison of all three methods
runComparison :: Scenario -> IO ()
runComparison scenario = do
  let config = scenarioConfig scenario
      grid = scenarioGrid scenario
  
  -- Serial
  putStr "Serial:              "
  timeSerial <- benchmarkScenario' runSimulation scenario
  
  -- Parallel Spatial
  putStr "Parallel (Spatial):  "
  timeParallel <- benchmarkScenario' runSimulationParallel scenario
  
  -- Parallel Red-Black
  putStr "Parallel (RedBlack): "
  timeRedBlack <- benchmarkScenario' runSimulationRedBlack scenario
  
  putStrLn ""
  putStrLn "Speedup Analysis:"
  printf "  Spatial vs Serial:  %.2fx\n" (timeSerial / timeParallel :: Double)
  printf "  RedBlack vs Serial: %.2fx\n" (timeSerial / timeRedBlack :: Double)

-- | Benchmark a scenario and print results
benchmarkScenario :: String -> (SimConfig -> Grid -> (Grid, [SimStats])) -> Scenario -> IO ()
benchmarkScenario name simFunc scenario = do
  let config = scenarioConfig scenario
      grid = scenarioGrid scenario
  
  putStrLn $ "Running " ++ name ++ "..."
  putStrLn $ "Grid: " ++ show (gridRows config) ++ "x" ++ show (gridCols config)
  
  start <- getCPUTime
  let (finalGrid, stats) = simFunc config grid
  end <- getCPUTime
  
  let diff = fromIntegral (end - start) / (10^(12::Integer)) :: Double
      finalStats = last stats
  
  printf "Time:  %.4f seconds\n" diff
  printf "Steps: %d\n" (timeStep finalStats)
  printf "Burned: %d cells (%.1f%%)\n\n"
    (totalBurned finalStats)
    (100.0 * fromIntegral (totalBurned finalStats) / fromIntegral (gridRows config * gridCols config) :: Double)

-- | Benchmark and return time only
benchmarkScenario' :: (SimConfig -> Grid -> (Grid, [SimStats])) -> Scenario -> IO Double
benchmarkScenario' simFunc scenario = do
  let config = scenarioConfig scenario
      grid = scenarioGrid scenario
  
  start <- getCPUTime
  let (finalGrid, stats) = simFunc config grid
  end <- getCPUTime
  
  let diff = fromIntegral (end - start) / (10^(12::Integer)) :: Double
      finalStats = last stats
  
  printf "%.4f sec (%d steps, %d burned)\n"
    diff
    (timeStep finalStats)
    (totalBurned finalStats)
  
  return diff