/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Riju Dey, Rachinta Marpaung, Charles Chen, Sasha Isler
 * Modified from Stephen Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/ioctl.h>
#include "geo_dash.h"

// =============================================
// ===== geo_dash structures and constants =====
// =============================================

#define DRIVER_NAME "geo_dash"
// Assuming that we have 16-bit registers.
#define X_SHIFT(base)        ((base) + 0x02)  // 16-bit

#define TILEMAP(base)   ((base))
#define PALETTE(base)   ((base) + 0x2000) 
#define TILESET(base)   ((base) + 0x4000)

#define FLAGS(base)          ((base) + 0x0C)  // lower 8 bits used
#define OUTPUT_FLAGS(base)   ((base) + 0x0E)  // lower 8 bits used
#define SCROLL_OFFSET(base)  ((base) + 0x10)  // 16-bit  - added for tile scroll
#define PLAYER_Y_POS(base) ((base) + 0x3002)  // 16-bit register
#define SCROLL_OFFSET(base) ((base) + 0x3010)  // 16-bit register

/*
Information about our geometry_dash device. Acts as a mirror of hardware state.
*/

struct geo_dash_dev {
    struct resource res; /* Our registers. */
    void __iomem *virtbase; /* Where our registers can be accessed in memory. */
    short x_shift;
    short scroll_offset;   /* Added to keep track of scroll offset */
} geo_dash_dev;

static void write_tile(uint8_t *value, int row, int col)
{
    pr_info("writing %d to tile map at row %d, col %d", *value, row, col);
    void *tilemap_location = TILEMAP(geo_dash_dev.virtbase) + row * 32 + col;
    iowrite8(*value, tilemap_location);
}

static void write_palette(uint32_t *rgb, int color_index)
{
    pr_info("writing color %x at index %d to palette", *rgb, color_index);
    uint8_t r = (*rgb >> 16) & 0xFF;
    uint8_t g = (*rgb >> 8) & 0xFF;
    uint8_t b = (*rgb) & 0xFF;

    uintptr_t addr = (uintptr_t) PALETTE(geo_dash_dev.virtbase) + color_index * 4;
    
    iowrite8(r, addr);
    iowrite8(g, addr + 1);
    iowrite8(b, addr + 2);
    iowrite8(0x01, addr + 3);
}

static void write_tileset(uint8_t *value, int tile_no, int pixel_no)
{
    /*writing to the pixel in that specific tile. */
    //pr_info("writing to tile");
    iowrite8(*value, TILESET(geo_dash_dev.virtbase) + tile_no * 32 * 32 + pixel_no);
    
}

static uint8_t read_tile(int row, int col)
{
      uintptr_t tilemap_location = (uintptr_t) TILEMAP(geo_dash_dev.virtbase) + row * 40 + col;
      return ioread8(tilemap_location);
}

static uint32_t read_palette(int color_index)
{
      void *rgb_location = PALETTE(geo_dash_dev.virtbase) + 4 * color_index;
      return ioread32(rgb_location);
}

static uint8_t read_tileset(int tile_no, int pixel_no)
{
      void *pixel_location = TILESET(geo_dash_dev.virtbase) + tile_no * 32 * 32 + pixel_no;
      return ioread8(pixel_location);
}

static void write_player_y_position(uint8_t *value) {
	iowrite8(*value, PLAYER_Y_POS(geo_dash_dev.virtbase));
}

static void write_x_shift(unsigned short *value) {
    iowrite16(*value, X_SHIFT(geo_dash_dev.virtbase));
    geo_dash_dev.x_shift = *value;
}

static void write_flags(uint8_t *value) {
    iowrite16((uint16_t)(*value), FLAGS(geo_dash_dev.virtbase));
}

static void write_output_flags(uint8_t *value) {
    iowrite16((uint16_t)(*value), OUTPUT_FLAGS(geo_dash_dev.virtbase));
}

static void write_scroll_offset(uint8_t *value) {
	iowrite8(*value, SCROLL_OFFSET(geo_dash_dev.virtbase));
}

static long geo_dash_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
    geo_dash_arg_t vla;
	printk(KERN_INFO "geo_dash_ioctl called with cmd 0x%x\n", cmd);

    // Copy user struct into kernel space
    if (copy_from_user(&vla, (geo_dash_arg_t *) arg, sizeof(vla)))
        return -EFAULT;

    switch (cmd) {
        case WRITE_TILE:
            pr_info("writing %d to tilemap at %d, %d", vla.tile_value, vla.tilemap_row, vla.tilemap_col);
            write_tile(&vla.tile_value, vla.tilemap_row, vla.tilemap_col);
            break;
        
        case READ_TILE:
        {
            uint8_t tile_value = read_tile(vla.tilemap_row, vla.tilemap_col);
            vla.tile_value = tile_value;
            if (copy_to_user((geo_dash_arg_t *) arg, &vla, sizeof(vla)))
                  return -EFAULT;
            break;
        }
        case WRITE_PALETTE:
            pr_info("writing to palette\n");            
            write_palette(&vla.rgb, vla.color_index);
            break;

        case READ_PALETTE:
        {
            uint32_t rgb = read_palette(vla.color_index);
            vla.rgb = rgb;
            if (copy_to_user((geo_dash_arg_t *) arg, &vla, sizeof(vla)))
                return -EFAULT;
            break;
        }
        case WRITE_TILESET: ;
//            pr_info("writing to tile set\n");
            /* this should write 32x32 bytes to the location requested*/
            int i = 0;
            for (i = 0; i < 32; i++) {
                int j = 0;
                for (j = 0; j < 32; j++) {
                    write_tileset(&vla.tileset[i][j], vla.tile_no, 32*i+j);
                }
            }
            break;
        case READ_TILESET:
            {
              int i, j;

              for (i = 0; i < 32; i++) {
                for (j = 0; j < 32; j++) {
                  vla.tileset[i][j] = read_tileset(vla.tile_no, 32*i+j);
                } 
              }
              if (copy_to_user((geo_dash_arg_t *) arg, &vla, sizeof(vla)))
                return -EFAULT;
          }
          break;
        case WRITE_X_SHIFT:
            write_x_shift(&vla.x_shift);
            break;

        case WRITE_PLAYER_Y_POS:
            write_player_y_position(&vla.player_y);
            break;

        case WRITE_FLAGS:
            write_flags(&vla.flags);
            break;

        case WRITE_OUTPUT_FLAGS:
            write_output_flags(&vla.output_flags);
            break;
            
        case WRITE_SCROLL_OFFSET:
            write_scroll_offset(&vla.scroll_offset);
            break;

        default:
            return -EINVAL;  // Unknown command
    }

    return 0;
}

static const struct file_operations geo_dash_fops = {
    .owner = THIS_MODULE,
    .unlocked_ioctl = geo_dash_ioctl
};

static struct miscdevice geo_dash_misc_device = {
    .minor = MISC_DYNAMIC_MINOR,
    .name = DRIVER_NAME,
    .fops = &geo_dash_fops
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init geo_dash_probe(struct platform_device *pdev)
{
        //cat_invaders_color_t beige = { 0xf9, 0xe4, 0xb7 };
		//audio_t audio_begin = { 0x00, 0x00, 0x00 };
	int ret;
	pr_info("geo_dash: probe successful\n");

	/* Register ourselves as a misc device: creates /dev/geo_dash */
	ret = misc_register(&geo_dash_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &geo_dash_dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(geo_dash_dev.res.start, resource_size(&geo_dash_dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}
	
	/* Arrange access to our registers */
	geo_dash_dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (geo_dash_dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	return 0;

out_release_mem_region:
	release_mem_region(geo_dash_dev.res.start, resource_size(&geo_dash_dev.res));
out_deregister:
	misc_deregister(&geo_dash_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int geo_dash_remove(struct platform_device *pdev)
{
	iounmap(geo_dash_dev.virtbase);
	release_mem_region(geo_dash_dev.res.start, resource_size(&geo_dash_dev.res));
	misc_deregister(&geo_dash_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id geo_dash_of_match[] = {
	{ .compatible = "csee4840,vga_tiles-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, geo_dash_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver geo_dash_driver = {
	.probe = geo_dash_probe, // move it here
	.remove = __exit_p(geo_dash_remove),
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(geo_dash_of_match),
	},
};


// ============================================
// =========== module init and exit ===========
// ============================================

/* Called when the module is loaded: set things up */
static int __init geo_dash_init(void)
{
    int ret;

    pr_info(DRIVER_NAME ": init\n");

    ret = platform_driver_register(&geo_dash_driver);
    if (ret)
        return ret;

    return 0;
}

/* Called when the module is unloaded: release resources */
static void __exit geo_dash_exit(void)
{
	platform_driver_unregister(&geo_dash_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(geo_dash_init);
module_exit(geo_dash_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("geometry dash driver");
