#include "circuit.h"

#include <stdio.h>
#include <stdlib.h>
#include "gaussian.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#define POLL_INTERVAL_US 10000  // 10 ms



/* Read and print the input sent to the Gaussian */
void print_in(int gaussian_fd) {
  gaussian_arg_t vla;
  printf("%d\n", gaussian_fd);
  if (ioctl(gaussian_fd, GAUSSIAN_READ, &vla)) {
      perror("ioctl(GAUSSIAN_READ) failed");
      return;
  }
    printf("INPUT n=%u  GO=%u  RST=%u\n",
           vla.input.n,
           vla.input.g,
           vla.input.r);
}

/* Read and print the output obtained from Gaussian */
gaussian_out_t get_out(int gaussian_fd) {
    gaussian_arg_t vla;
    if (ioctl(gaussian_fd, GAUSSIAN_READ, &vla) < 0) {
        perror("ioctl(GAUSSIAN_READ) failed");
        exit(1);
    }
    printf("OUTPUT DONE=%u  E=%u  S=%u\n\n",
            vla.output.d,
            vla.output.e,
            vla.output.s);

    return vla.output;
}

/* Set martrix size and GO/RES flags */
void set_in(const gaussian_in_t *c, int gaussian_fd)
{
    gaussian_arg_t vla;
    memset(&vla, 0, sizeof(vla));
    vla.input = *c;
    if (ioctl(gaussian_fd, GAUSSIAN_WRITE, &vla)) {
        perror("ioctl(GAUSSIAN_WRITE) failed");
        return;
    }
}

/* This will become solve_system(n), the function called by newton.c */
int solve_system(int n, int gaussian_fd) {
    if (n <= 0 || n > 255) {
        fprintf(stderr, "Matrix size must be 1-255\n");
        return -1;
    }

    // kick off the hardware

    

    /* Kick off the computation: set size=n, GO=1, RST=0 */
    gaussian_in_t in = {
        .n = (uint8_t)n,
        .g = 1,    // GO
        .r = 0     // RESET
    };
    set_in(&in, gaussian_fd);

    print_in(gaussian_fd);

    /* Clear GO so we can detect future runs */
    in.g = 0;   
    /* Warning: GO may hold high for multiple cycles. Make sure HW can handle that */
    set_in(&in, gaussian_fd);

    print_in(gaussian_fd);

    /* poll until DONE = 1 */
    printf("Waiting for DONE flag...\n");
    while (1) {
        gaussian_out_t out = get_out(gaussian_fd);
        printf("  DONE=%u  E=%u  S=%u\r",
                out.d, out.e, out.s);
        fflush(stdout);
        if (out.d)
            break;
        usleep(POLL_INTERVAL_US);
    }

    printf("Gaussian Userspace program terminating\n");
        
    return 0;
}



void print_matrix(float m[][MAT_SIZE], float I[MAT_SIZE], int n) {
    for (int i = 0; i < n; i++) {
        for (int j = 0; j < n; j++)
            printf("%8.4g ", m[i][j]);
        printf(" | %8.4g\n", I[i]);
    }
    printf("\n");
}

void print_vector(float v[MAT_SIZE], int n) {
    for (int i = 0; i < n; i++)
        printf("%8.4g ", v[i]);
    printf("\n");
}
