// linear.c
#include "circuit.h"

// all linear components

float pulse(float t, SpicePulseParams params) {
    printf("pulse: t=%.6g  v0=%.6g v1=%.6g td=%.6g  tr=%.6g  tf=%.6g  pw=%.6g  per=%.6g\n", t, params.v0, params.v1, params.td, params.tr, params.tf, params.pw, params.per);
    float delay_adj_t = t - params.td;

    if (delay_adj_t < 0) return 0.0f;

    float period = params.per;

    float location_in_period = fmod(delay_adj_t, period);

    printf("pulse: location_in_period=%.6g\n", location_in_period);

    // check if on rising edge
    if (location_in_period < params.tr) {
        return params.v0 + (params.v1 - params.v0) * (location_in_period / params.tr);
    }

    // check if on high level
    if (location_in_period < params.tr + params.pw) {
        return params.v1;
    }

    // check if on falling edge
    if (location_in_period < params.tr + params.pw + params.tf) {
        return params.v1 - (params.v1 - params.v0) * ((location_in_period - params.tr - params.pw) / params.tf);
    }

    // check if on low level
    return params.v0;
}

// — Resistor (and Norton‐equiv Vsrc) stamp with -1 = ground —
void res_stamp(Component *c, float Gm[][MAT_SIZE], float I[]) {
    (void)I;
    Res *r = &c->u.res;
    float G = 1.0f / r->R;
    int   n1 = r->n1, n2 = r->n2;

    if (n1 != -1) Gm[n1][n1] += G;
    if (n2 != -1) Gm[n2][n2] += G;
    if (n1 != -1 && n2 != -1) {
        Gm[n1][n2] -= G;
        Gm[n2][n1] -= G;
    }
}

/* add an ideal voltage source; stamps a "virtual" node (ni) to the circuit.*/
void vsrc_stamp(Component *c, float Gm[][MAT_SIZE], float I[]) {
    VSrc *s = &c->u.vsrc;
    TransientSource *src = s->src;
    float E;
    if (src->type == SPICE_FUNC_SIN) {
        SpiceSinParams *params = &src->params.sin;
        E = params->vo + params->va * exp(-params->a * (t - params->td)) * sinf(2.0f * M_PI * params->fo * (t - params->td) + params->phase / 360.0f);
    } else if (src->type == SPICE_FUNC_PULSE) {
        // SpicePulseParams *params = &src->params.pulse;
        //TODO: implement pulse
        E = pulse(t, src->params.pulse);
        printf("vsrc_stamp: E=%.6g\n", E);
    } else if (src->type == SPICE_FUNC_DC) {
        E = src->params.dc_value;
    } else {
        printf("vsrc_stamp: unknown source type\n");
        E = 0.0f;
        // something went wrong if we're here
    }

    //printf("vsrc_stamp: E=%.6g\n", E);
    int n1 = s->n1, n2 = s->n2, ni = s->ni;

    if (n1 != -1) {
        Gm[n1][ni] += 1.0f;
        Gm[ni][n1] += 1.0f;  // symmetry for the voltage equation
    }
    if (n2 != -1) {
        Gm[n2][ni] -= 1.0f;
        Gm[ni][n2] -= 1.0f;
    }

    // Voltage equation: v(n1) - v(n2) = E
    // stamped as  +1·v(n1) -1*v(n2) +0*I_vs  = E
    I[ni] += E;
}

void isrc_stamp(Component *c, float Gm[][MAT_SIZE], float I[]) {
    (void)Gm; (void)I;
    ISrc *s = &c->u.isrc;
    TransientSource *src = s->src;
    float Ieq;
    if (src->type == SPICE_FUNC_SIN) {
        SpiceSinParams *params = &src->params.sin;
        Ieq = params->vo + params->va * exp(-params->a * (t - params->td)) * sinf(2.0f * M_PI * params->fo * (t - params->td) + params->phase / 360.0f);
    } else if (src->type == SPICE_FUNC_PULSE) {
        Ieq = pulse(t, src->params.pulse);
        //TODO: implement pulse
    } else if (src->type == SPICE_FUNC_DC) {
        Ieq = src->params.dc_value;
    } else {
        printf("vsrc_stamp: unknown source type\n");
        Ieq = 0.0f;
        // something went wrong if we're here
    }

    int   n1 = s->n1, n2 = s->n2;

    if (n1 != -1) I[n1] += Ieq;
    if (n2 != -1) I[n2] -= Ieq;
}

/*
 * Voltage‐Controlled Voltage Source (VCVS):
 *    v(n1)-v(n2) = mu * (v(np)-v(nn))
 * adds one extra current‐unknown at index ni.
 */
void vcvs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]) {
    (void)I;
    Vcvs *s = &c->u.vcvs;
    int n1   = s->n1,
        n2   = s->n2,
        np   = s->np,
        nn   = s->nn,
        ni   = s->ni;
    float mu = s->A;

    if (n1) Gm[n1][ni] +=  1.0f;
    if (n2) Gm[n2][ni] += -1.0f;

    if (n1) Gm[ni][n1] +=  1.0f;
    if (n2) Gm[ni][n2] += -1.0f;
    if (np) Gm[ni][np] += -mu;
    if (nn) Gm[ni][nn] +=  mu;

    //no direct RHS contribution for a pure dependent source
}

/*
 * Voltage‐Controlled Current Source (VCCS):
 *    i = gm * (v(np)-v(nn))
 * injects from n2 to n1.
 */
void vccs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]) {
    (void)I;
    Vccs *s = &c->u.vccs;
    int   n1 = s->n1,
          n2 = s->n2,
          np = s->np,
          nn = s->nn;
    float gm = s->A;

    if (n1 && np) Gm[n1][np] += -gm;
    if (n1 && nn) Gm[n1][nn] += gm;
    if (n2 && np) Gm[n2][np] += gm;
    if (n2 && nn) Gm[n2][nn] += -gm;
}


void cap_stamp_lin(Component *c, float Gm[][MAT_SIZE], float I[]) {
    Cap *p = &c->u.cap;
    float Gc  = p->C / p->dt;
    float i_prev = p->i_prev;

    int n1 = p->n1, n2 = p->n2;
    if (n1 != -1) Gm[n1][n1] += Gc;
    if (n2 != -1) Gm[n2][n2] += Gc;
    if (n1 != -1 && n2 != -1) {
        Gm[n1][n2] -= Gc;
        Gm[n2][n1] -= Gc;
    }
    if (n1 != -1) I[n1] -= i_prev; // Ieq is defined leaving n1, entering n2
    if (n2 != -1) I[n2] += i_prev;
}

void cap_update(Component *c) {
    Cap *p = &c->u.cap;
    float vc = (p->n1!=-1? v[p->n1]:0) - (p->n2!=-1? v[p->n2]:0);
    float Gc = p->C / p->dt;
    float Ieq = -Gc * vc;

    p->i_prev = Ieq;
    p->v_prev = vc;

    if (PRINT_LIN)
        printf("cap_update: i_prev=%.6g  v_prev=%.6g Ieq=%.6g\n", p->i_prev, p->v_prev, Ieq);
}


void ind_stamp_lin(Component *c, float Gm[][MAT_SIZE], float I[]) {
    Ind *p = &c->u.ind;
    float Gl  = p->dt / p->L;
    float i_prev = p->i_prev;

    int n1 = p->n1, n2 = p->n2;
    if (n1 != -1) Gm[n1][n1] += Gl;
    if (n2 != -1) Gm[n2][n2] += Gl;
    if (n1 != -1 && n2 != -1) {
        Gm[n1][n2] -= Gl;
        Gm[n2][n1] -= Gl;
    }
    if (n1 != -1) I[n1] -= i_prev;
    if (n2 != -1) I[n2] += i_prev;
}

void ind_update(Component *c) {
    Ind *p = &c->u.ind;
    float vl = (p->n1!=-1? v[p->n1]:0) - (p->n2!=-1? v[p->n2]:0);
    float Gl = p->dt / p->L;
    float Ieq = Gl * vl;

    p->i_prev = p->i_prev + Ieq;
    p->v_prev = vl;

    if (PRINT_LIN)
        printf("ind_update: i_prev=%.6g  v_prev=%.6g\n", p->i_prev, p->v_prev);
}