/*
 * Userspace program that communicates with the gaussian device driver
 * through ioctls
 *
 * Ming Gong
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include <stdlib.h>
#include "gaussian.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#define POLL_INTERVAL_US 10000  // 10 ms


int gaussian_fd;

/* Read and print the input sent to the Gaussian */
void print_in() {
  gaussian_arg_t vla;
  
  if (ioctl(gaussian_fd, GAUSSIAN_READ, &vla)) {
      perror("ioctl(GAUSSIAN_READ) failed");
      return;
  }
    printf("INPUT n=%u  GO=%u  RST=%u\n",
           vla.input.n,
           vla.input.g,
           vla.input.r);
}

/* Read and print the output obtained from Gaussian */
gaussian_out_t get_out(void) {
    gaussian_arg_t vla;
    if (ioctl(gaussian_fd, GAUSSIAN_READ, &vla) < 0) {
        perror("ioctl(GAUSSIAN_READ) failed");
        exit(1);
    }
    printf("OUTPUT DONE=%u  E=%u  S=%u\n\n",
           vla.output.d,
           vla.output.e,
           vla.output.s);

    return vla.output;
}

/* Set martrix size and GO/RES flags */
void set_in(const gaussian_in_t *c)
{
    gaussian_arg_t vla;
    memset(&vla, 0, sizeof(vla));
    vla.input = *c;
    if (ioctl(gaussian_fd, GAUSSIAN_WRITE, &vla)) {
        perror("ioctl(GAUSSIAN_WRITE) failed");
        return;
    }
}

/* This will become solve_system(n), the function called by newton.c */
int main() {
    int n;
    if (n <= 0 || n > 255) {
        fprintf(stderr, "Matrix size must be 1-255\n");
        return -1;
    }
    static const char filename[] = "/dev/gaussian";

    printf("Gaussian Userspace program started\n");

    if ((gaussian_fd = open(filename, O_RDWR)) == -1) {
        fprintf(stderr, "could not open %s\n", filename);
        return -1;
    }

    // kick off the hardware

    print_in();

    /* Kick off the computation: set size=n, GO=1, RST=0 */
    gaussian_in_t in = {
        .n = (uint8_t)n,
        .g = 1,    // GO
        .r = 0     // RESET
    };
    set_in(&in);

    /* Clear GO so we can detect future runs */
    in.g = 0;   
    /* Warning: GO may hold high for multiple cycles. Make sure HW can handle that */
    set_in(&in);

    /* poll until DONE = 1 */
    printf("Waiting for DONE flag...\n");
    while (1) {
        gaussian_out_t out = get_out();
        printf("  DONE=%u  E=%u  S=%u\r",
               out.d, out.e, out.s);
        fflush(stdout);
        if (out.d)
            break;
        usleep(POLL_INTERVAL_US);
    }

    printf("Gaussian Userspace program terminating\n");

    if (gaussian_fd >= 0)
        close(gaussian_fd);
        
    return 0;
}
