/* * Device driver for the matrix solver
 *
 * A Platform device implemented using the misc subsystem
 *
 * Ming Gong
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod gaussian.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree gaussian.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "gaussian.h"

#define DRIVER_NAME "gaussian"

/* Device registers */
#define N(x) 	  ((x)+0)
#define IFLAGS(x) ((x)+1)
#define OFLAGS(x) ((x)+2)

/*
 * Information about our device
 */
struct gaussian_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        gaussian_in_t in;
		gaussian_out_t out;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_in(gaussian_in_t *in)
{
	iowrite8(in->n, N(dev.virtbase) );
	u8 iflags;
	iflags =  (in->g & 0x1)        /* bit 0 */
				| ((in->r & 0x1) << 1); /* bit 1 */
	iowrite8(iflags, IFLAGS(dev.virtbase));
	dev.in = *in;
}

static void read_out(gaussian_out_t *out)
{
    uint8_t oflags = ioread8(OFLAGS(dev.virtbase));
    out->d = (oflags >> 7) & 1;
    out->e = (oflags >> 6) & 1;
    out->s = (oflags >> 5) & 1;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long gaussian_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	gaussian_arg_t vla;

	switch (cmd) {
	case GAUSSIAN_WRITE:
		if (copy_from_user(&vla, (gaussian_arg_t *) arg,
				   sizeof(gaussian_arg_t)))
			return -EACCES;
		//write_background(&vla.background);
		write_in(&vla.input);
		break;

	case GAUSSIAN_READ:
		read_out(&vla.output);
		if (copy_to_user((gaussian_arg_t *) arg, &vla,
						 sizeof(gaussian_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations gaussian_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = gaussian_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice gaussian_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &gaussian_fops,
};

/*
 * Initialization code: get resources (registers)
 */
static int __init gaussian_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/gaussian */
	ret = misc_register(&gaussian_misc_device);
    if (ret) {
        pr_err("gaussian: misc_register failed (%d)\n", ret);
        return ret;
    }

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
        pr_err("gaussian: region busy @ %pa\n", &dev.res.start);
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
	    pr_err("gaussian: iomap failed\n");
		ret = -ENOMEM;
		goto out_release_mem_region;
	}

	/* Welcome banner */
    pr_info("gaussian: probed @ phys %pa, virt %p\n",
            &dev.res.start, dev.virtbase);
    pr_info("gaussian: ready!  Welcome to the Gaussian driver.\n");
        
	// clear all inputs
    memset(&dev.in, 0, sizeof(dev.in));
    write_in(&dev.in);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&gaussian_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int gaussian_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&gaussian_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id gaussian_of_match[] = {
	{ .compatible = "csee4840,gaussian-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, gaussian_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver gaussian_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(gaussian_of_match),
	},
	.remove	= __exit_p(gaussian_remove),
};

/* Called when the module is loaded: set things up */
static int __init gaussian_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&gaussian_driver, gaussian_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit gaussian_exit(void)
{
	platform_driver_unregister(&gaussian_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(gaussian_init);
module_exit(gaussian_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Ming Gong, Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("Gaussian driver");
