/* circuit.h */
#ifndef CIRCUIT_H
#define CIRCUIT_H

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#ifdef __cplusplus
extern "C" {
#endif

#define MAT_SIZE     128    // maximum number of nodes
#define MAX_COMPS    100   // maximum number of components

#define MAX_NR_ITER 20
#define NR_TOL      1e-4f
#define DV_MAX      100.0f
#define VT_DEFAULT 0.025875f

#define PRINT_LIN   0
#define PRINT_NL    0
#define PRINT_MAT   0

#define FPGA_AXI_BASE 0xC0000000 // base address for RAM on FPGA
#define FPGA_AXI_SPAN 0x00020000 // 131072 bytes

// — Global nodal matrix and RHS —
extern float G[MAT_SIZE][MAT_SIZE];
extern float Ivec[MAT_SIZE];
extern float v[MAT_SIZE], v_prev[MAT_SIZE];
extern float t;

typedef struct {
    double v0;
    double v1;
    double td;
    double tr;
    double tf;
    double pw;
    double per;
} SpicePulseParams;

typedef struct {
    double vo;
    double va;
    double fo;
    double td;
    double a;
    double phase;
} SpiceSinParams;

typedef enum {
    SPICE_FUNC_PULSE,
    SPICE_FUNC_SIN,
    SPICE_FUNC_DC
} SpiceFuncType;

typedef struct {
    SpiceFuncType type;
    union {
        SpicePulseParams pulse;
        SpiceSinParams sin;
        double dc_value;
    } params;
} TransientSource;

// — Component data types —
// sources
typedef struct { 
    int n1, n2, ni;             // VSrc has an internal node
    TransientSource *src;
} VSrc;   
typedef struct { 
    int n1, n2;
    TransientSource *src;
} ISrc;
// dependent sources
typedef struct { 
    int n1, n2;
    int np, nn, ni;   
    float A;         // gain    
} Vcvs;
typedef struct { 
    int n1, n2;
    int np, nn;
    float A; 
} Vccs;
typedef struct { int n1, n2, np, nn, ni; float A; } Ccvs;
typedef struct { int n1, n2, np, nn; float A; } Cccs;

// linear components
typedef struct { 
    int n1, n2;
    float R; 
} Res;

// time-varying components
typedef struct { 
    int n1, n2;                 // nodes
    float C, dt;                // device constants
    float i_prev, v_prev;       // previous values
} Cap;
typedef struct { 
    int n1, n2; 
    float L, dt;
    float i_prev, v_prev; 
} Ind;

// nonlinear components
typedef struct { 
    int n1, n2; 
    float Is, Vt;
    float v_prev; 
} Diode;
typedef struct { 
    int ng, nd, ns; 
    float beta, Vt, lambda;
    float vgs_prev, vds_prev; 
} Nmos;
typedef struct { 
    int ng, nd, ns; 
    float beta, Vt, lambda;
    float vsg_prev, vsd_prev; 
} Pmos;

// generic component structure
typedef enum { STA_T, LIN_T, NL_T } CompType;
typedef struct Component {
    CompType type;
    void (*stamp_lin)(struct Component*, float[][MAT_SIZE], float[]);
    void (*stamp_nl) (struct Component*, float[][MAT_SIZE], float[]);
    void (*update)   (struct Component*);
    union {
        VSrc   vsrc;
        ISrc   isrc;
        Vcvs   vcvs;
        Vccs   vccs;
        Ccvs   ccvs;
        Cccs   cccs;
        Res    res;
        Cap    cap;
        Ind    ind;
        Diode  dio;
        Nmos   nmos;
        Pmos   pmos;
    } u;
} Component;

// component registry
extern Component comps[MAX_COMPS];
extern int       ncomps; // number of components in the circuit
extern int       nnodes; // number of nodes in the circuit

// stamper and solver
void clear_system(void);
void clear_system_sta(void);
void stamp_static(void);
int solve_system(int n, int fd);
void update_all(float t);

void print_components(void);
void print_matrix(float m[][MAT_SIZE], float I[MAT_SIZE], int n);
void print_vector(float v[MAT_SIZE], int n);


void res_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void vsrc_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void isrc_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void vcvs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void vccs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void ccvs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);
void cccs_stamp(Component *c, float Gm[][MAT_SIZE], float I[]);

void cap_stamp_lin(Component *c, float Gm[][MAT_SIZE], float I[]);
void cap_update(Component *c);

void ind_stamp_lin(Component *c, float Gm[][MAT_SIZE], float I[]);
void ind_update(Component *c);

void dio_stamp_nl(Component *c, float Gm[][MAT_SIZE], float I[]);
void dio_update(Component *c);

void nmos_stamp_nl(Component *c, float Gm[][MAT_SIZE], float I[]);
void nmos_update(Component *c);

void pmos_stamp_nl(Component *c, float Gm[][MAT_SIZE], float I[]);
void pmos_update(Component *c);


// registration & top‑level
void add_vsrc(int n1, int n2, int ni, TransientSource *src);
void add_isrc(int n1, int n2, TransientSource *src);
void add_vcvs(int n1, int n2, int np, int nn, int ni, float A);
void add_vccs(int n1, int n2, int np, int nn, float A);
void add_ccvs(int n1, int n2, int np, int nn, int ni, float A);
void add_cccs(int n1, int n2, int np, int nn, float A);

void add_res (int n1, int n2, float R);
void add_cap (int n1, int n2, float C, float dt, float v0);
void add_ind (int n1, int n2, float L, float dt, float i0);
void add_diode(int n1, int n2, float Is, float Vt);
void add_nmos(int ng, int nd, int ns, float beta, float Vt, float lambda);
void add_pmos(int ng, int nd, int ns, float beta, float Vt, float lambda);

void add_not(int vin, int vout, int vDD);
void add_nand(int vina, int vinb, int vout, int vn, int vDD);

#ifdef __cplusplus
} // extern "C"
#endif

#endif // CIRCUIT_H
