# circuit-sim

Reference: https://www.ecircuitcenter.com/SpiceTopics/Overview/Overview.htm

# Compilation
## File List
- `driver.c`: Driver that loads test cases from the `tests` folder
    - `test_lrc.c`: simple linear LRC circuit
    - `test_doubler.c`: diode voltage doubler
    - `test_4th.c`: 4th-order Sallen-Key filter
    - `test_not.c`: simple CMOS NOT gate
    - `test_dff.c`: master-slave D flip flop, edge triggered
- `add_component.c`: user level API to register the components
- `input_funcs.c`: temporary files that define input voltage functions

- `newton.c`: Newton-Raphson loop to solve and simulate the circuit
- `linear.c`, `nonlinear.c`: stamp and update functions for various components
- `solver.c`: Gaussian Matrix solver

## Makefile
Run `make all` to compile all test cases
- `make test_*` will only compile `tests/test_*.c` to `./test_*.exe`

# API
Circuit-Sim maintains a global array of components and their corresponding constants and variables. The user can add components and specify their behaviors.
## User level
Currently, the user calls the adder functions define in `add_component.c`. They generally have the following format:
```c
void add_component(int n1, int n2, float constant1, ...);
```
- `n1`, `n2` are the two nodes the component connects to 
    - Voltage sources have an additional internal node `ni`. This is another row in the matrix needed [Modified Nodal Analysis](https://lpsa.swarthmore.edu/Systems/Electrical/mna/MNA2.html#Example_3) and [controlled sources] (https://qucs.sourceforge.net/tech/node60.html)
    - Dependent sources will have extra nodes for input
    - The upper level needs to make sure that all nodes, including internal ones, are allocated and connected properly
- Constants: device parameters (resistance, capacitance, etc.)
- Independent sources take a `float (*func)(float)` to generate custom inputs (DC, AC, step, etc.) 

Finally, the user sets `n_nodes` to the number of nodes. Don't forget this!

### Examples
Here's `add_cap()`:
```c
void add_cap(int n1, int n2, float C, float dt) {
    // allocate a new component
    Component *c = &comps[ncomps++];

    // hard-coded section
    c->type      = LIN_T;
    c->stamp_lin = cap_stamp_lin;
    c->stamp_nl  = NULL;
    c->update    = cap_update;

    // user-defined section
    c->u.cap = (Cap){
        .n1     = n1,       // positive node
        .n2     = n2,       // negative node
        .C      = C,        // capacitance
        .dt     = dt,       // timestep, can be hard-coded to a const
        .i_prev = 0.0f,     // initial Norton current, can be hard-coded to 0
        .v_prev = 0.0f      // initial voltage , can be hard-coded to 0
    };
}
```
`dt` can be a hard-coded global variable. We only need `n1`, `n2`, `C` from the user

Here's `add_vsrc()`:
```c
void add_vsrc(int n1,int n2, int ni, float (*volt)(float)) {
    Component *c = &comps[ncomps++];
    c->type      = LIN_T;
    c->stamp_lin = vsrc_stamp;
    c->stamp_nl  = NULL;
    c->update    = NULL;

    // user-defined section
    c->u.vsrc = (VSrc){
        .n1     = n1,
        .n2     = n2, 
        .ni     = ni,   // ni for an internal node for an extra row in the equation.
        .volt   = volt  // voltage function
    };
}
```
below are some sample `volt` functions, defined in `main.c`
```c
float dc0(float t) { return 0.0f; }

float sin5(float t) { return 5.0f * sinf(2.0f * M_PI * 1000.0f * t); }
```

For AC, the user will input two more parameters, amplitude and frequency. They are currently hard-coded as 5.0 and 1000.0, but we need to make it configurable.

### Simulation results
The node voltage at each time step can be saved to a csv
- Intermediate convergence models will not be saved
- You can also save the device parameters (E. inductor current) by reading their values at each time step.

## Hardware level
`newton.c` takes care of all the simulation, calling each component's update function if needed. In each step, it calls `solve_matrix()`, which we will implement in hardware
1. Loads all the matrix entries (`G` and `Ivec`)
2. Call `solve_matrix()`
3. `solve_matrix()` returns the values in `v`

I defined a Gaussian prototype `solve_matrix()` in `solver.c`. We need to implement it in HW.



# Algorithm
## Overview 
We define a generic `struct Component`, with the following attributes
```c
typedef struct Component {
    DevType type;
    void (*stamp_lin)(struct Component*, float[][MAT_SIZE], float[], float);
    void (*stamp_nl) (struct Component*, float[][MAT_SIZE], float[]);
    // float (*residue) (struct Component*);
    void (*update)   (struct Component*);

    // device values, and memory (if applicable)
    union {         
        VSrc   vsrc;
        ISrc   isrc;
        Vcvs   vcvs;
        Vccs   vccs;
        Ccvs   ccvs;
        Cccs   cccs;
        Res    res;
        Cap    cap;
        Ind    ind;
        Diode  dio;
        Nmos   nmos;
        Pmos   pmos;
    } u;
} Component;
```
This is `struct Cap`, for instance:
```c
typedef struct { 
    int n1, n2;                 // nodes
    float C, dt;                // constants
    float i_prev, v_prev, Ieq;  // previous memory
} Cap;
```
- `u`: device metadata: capacitance, last voltage, last current, etc.
- Stamper (`stamp_lin` or `stamp_nl`): stamp the old metadata into the matrix
- Updater (`update`): update device metadata based on new simulation

This is a generic routing. For some components, not all steps are required. 
- We can save a copy of all static components (E. resistor, dependent source), so we don't have to re-stamp them every time, saving some computation.
## 0. DC Operating point analysis
Find the initial conditions (all 0)


## 1. Compute linear conductance for nonlinear/time-varying devices
### [Diodes](https://www.ecircuitcenter.com/SpiceTopics/Non-Linear%20Analysis/Non-Linear%20Analysis.htm)
$$i_D = a(e^{bv_D} - 1)$$
where a, b are constants

Companion model: turn diode into conductance $G_{eq}$ in parallel with current source $I_{eq}$ (not the same as $i_D$)
$$G_{eq} = \frac{di_D}{dv_D} = abe^{bv_D}$$
$$I_{eq} = I_{eq, prev} - G_{eq} v_D$$

### MOSFETS
Very similar ideas with diodes, but this time it's a piecewise-multivariable function. I used the same linear approximation planes
- There is a small discontinuity near the cutoff region, which may lead to convergence issues, but that is very very rare in practice.
- We can modify the models a bit to ensure continuity


### Inductors
!! The model below is outdated. I used [this](https://electronics.stackexchange.com/questions/272012/companion-capacitor-model-in-circuit-simulation)


[Companion model](https://circsimproj.blogspot.com/2009/07/companion-models.html):
$$G_{eq} = \frac{\Delta t}{2L}$$
$$I_{eq} = I_{eq, prev}  + G_{eq} v_L$$
Note that for inductors, $G_{eq}$ is constant, if the timestep $\Delta t$ is constant

### Capacitors
Similar companion model as the inductor

## 2. Load conductance into [nodal matrix](https://www.ecircuitcenter.com/SpiceTopics/Nodal%20Analysis/Nodal%20Analysis.htm)
$$G \cdot \vec{v} = \vec{I}$$

This is the same as $v = IR$. Conductance $G = 1/R$, the inverse of resistance

## 3. Solve the linear matrix
Done in hardware
- Gaussian elimination, or
- LRU decomposition  

Probably need pipelining!!


## 4. (if nonlinear) Update matrix for [nonlinear](https://www.ecircuitcenter.com/SpiceTopics/Non-Linear%20Analysis/Non-Linear%20Analysis.htm) components
!! Software or hardware?

For diodes, the only function we need is $\exp$. If a diode connects 2 nodes, we'll need to update the corresponding entry in $G$ and $\vec{I}$ after every iteration
- I'm thinking doing it in hardware
- After matrix solve, we'll design a module that takes a memory address of $G_{mn}$, and updates 
$$G_{mn, nxt} = a + be^{\vec{c}\vec{v}}$$
where a, b, **c** are constants, and written by software
- We need space to store a, b, and **c** for every $G_{mn}$ that we want to update
- This should't be too much, as long as there aren't too many diodes

Need to **repeat** 1-4 until the nonlinear estimate converges

## 5. Update matrix for time-varying components
The idea is [here](https://www.ecircuitcenter.com/SpiceTopics/Transient%20Analysis/Transient%20Analysis.htm) (similar as step 4), but I think this [trapezoidal formula](https://circsimproj.blogspot.com/2009/07/companion-models.html) is easier
- This timestep needs to be chosen *very* carefully
- Again, should the update part go in SW or HW?
    - Since $G_{eq}$ for L and C are constants, this update step would be easier than (4).

Working example: `companion-sim.c`
- Simulates an LRC circuit (linear) with companion models for L and C.
- Results consistent with Falstad

Then, advance the timestep

