module signed_adder_subtractor_wrapper (
    input  logic        clk,
    input  logic        rst,
    input  logic        start,
    output logic [16:0] result_out
);

    // Parameters
    localparam WIDTH = 16;

    // Internal signals
    logic signed [WIDTH-1:0] dataa;
    logic signed [WIDTH-1:0] datab;
    logic                    add_sub;
    logic [WIDTH:0]          result;

    // Instantiate the signed adder/subtractor
    signed_adder_subtractor #(
        .WIDTH(WIDTH)
    ) dut (
        .dataa(dataa),
        .datab(datab),
        .add_sub(add_sub),
        .clk(clk),
        .result(result)
    );

    // FSM states
    typedef enum logic [1:0] {
        IDLE,
        LOAD_INPUT,
        WAIT_RESULT
    } state_t;

    state_t state;

    // Input generation counter
    logic [3:0] counter;

    always_ff @(posedge clk or posedge rst) begin
        if (rst) begin
            state      <= IDLE;
            dataa      <= 0;
            datab      <= 0;
            add_sub    <= 0;
            result_out <= 0;
            counter    <= 0;
        end else begin
            case (state)
                IDLE: begin
                    if (start)
                        state <= LOAD_INPUT;
                end

                LOAD_INPUT: begin
                    dataa   <= counter - 5;           // Signed values: -5 to +10
                    datab   <= 15 - counter;          // Varies inversely
                    add_sub <= counter[0];            // Alternate add/sub
                    state   <= WAIT_RESULT;
                end

                WAIT_RESULT: begin
                    result_out <= result;
                    counter    <= counter + 1;
                    state      <= IDLE;  // Can loop back or go to LOAD_INPUT
                end
            endcase
        end
    end

endmodule
