module SatGen
  ( generateSatisfiableCNF, 
    cnfToDimacs, 
    CNF, 
    Clause, 
    Literal
  ) where

import System.Random
import Control.Monad
import qualified Data.Set as Set

type Literal = Int
type Clause = [Literal]
type CNF = [Clause]

randomLiteral :: Int -> IO Literal
randomLiteral numVars = do
  var <- randomRIO (1, numVars)
  sign <- randomRIO (False, True)
  return $ if sign then var else -var

generateClauseWithSat :: Int -> Int -> Literal -> IO Clause
generateClauseWithSat numVars len satLit = go (Set.singleton satLit)
  where
    go used
      | Set.size used == len = pure $ Set.toList used
      | otherwise = do
          lit <- randomLiteral numVars
          if Set.member lit used || Set.member (-lit) used
            then go used
            else go (Set.insert lit used)

generateSatisfiableCNF :: Int -> Int -> Int -> IO CNF
generateSatisfiableCNF numVars numClauses clauseLen =
  do
    randVals <- replicateM numVars (randomRIO (0, 1) :: IO Int)
    -- convert from numbers to booleans
    let assignment = map (==1) randVals
    let satisfying = zipWith (\v b -> if b then v else -v) [1..numVars] assignment
    
    clauses <- replicateM numClauses $ do
      -- pick a random satisfying literal
      satLit <- (satisfying !!) <$> randomRIO (0, numVars - 1)
      -- genrate the clause using this literal
      generateClauseWithSat numVars clauseLen satLit
      
    -- remove duplicates
    pure $ Set.toList $ Set.fromList clauses

cnfToDimacs :: Int -> CNF -> String
cnfToDimacs numVars cnf = 
  let
    header = "p cnf " ++ show numVars ++ " " ++ show (length cnf)
    
    clauseToString :: Clause -> String
    clauseToString clause = 
        let numbers = map show clause -- convert numbers to strings
            joined = unwords numbers -- join with spaces
        in joined ++ " 0"
        
    clauseStrings = map clauseToString cnf
    allLines = header : clauseStrings
    in
      unlines allLines