module SatBruteForce (solveSATParallel) where

import Control.Parallel.Strategies
import Control.Monad (replicateM)
import Control.Applicative (Alternative(..))
import Data.Maybe (fromJust)
import Data.List.Split (chunksOf)
import SatGen (CNF, Clause, Literal)

type Assignment = [(Int, Bool)]

generateAllAssignments :: Int -> [[Assignment]]
generateAllAssignments n = 
  let allAssignments = [ zip [1..n] bools | bools <- replicateM n [False, True] ]
      chunkSize = 128
  in chunksOf chunkSize allAssignments

evaluateLiteral :: Assignment -> Literal -> Bool
evaluateLiteral assignment lit = 
  let variable = abs lit
      value = fromJust (lookup variable assignment)
  in if lit > 0 then value else not value 

evaluateClause :: Assignment -> Clause -> Bool
evaluateClause assignment clause = any (evaluateLiteral assignment) clause

evaluateCNF :: Assignment -> CNF -> Bool
evaluateCNF assignment cnf = all (evaluateClause assignment) cnf

evaluateChunk :: CNF -> [Assignment] -> Maybe Assignment
evaluateChunk cnf assignments = 
    findFirstSatisfying assignments
  where
    findFirstSatisfying [] = Nothing
    findFirstSatisfying (assign:rest) 
        | evaluateCNF assign cnf = Just assign
        | otherwise = findFirstSatisfying rest

solveSATParallel :: CNF -> Int -> Maybe Assignment
solveSATParallel cnf numVars = 
  let chunks = generateAllAssignments(numVars)
      results = parMap rdeepseq (evaluateChunk cnf) chunks
  in foldr (<|>) Nothing results