module DPLL.ParallelDpll (
    SatSolver(..),
    parallelSolveOne,
    parallelSolveQueue,
    parallelSolveDynamicQ
) where

import Control.Concurrent
import Control.Concurrent.STM
import Control.Parallel.Strategies
import Data.Maybe (mapMaybe, listToMaybe)
import qualified Data.IntMap.Strict as IM
import qualified Data.IntSet as IS
import System.Random (StdGen, randomRs)
import Control.Monad (replicateM_, foldM)
import DPLL.DpllSolver
import DPLL.Literal
import DPLL.Clause

parallelSolveDynamicQ :: Int -> StdGen -> SatSolver -> IO (Maybe SatSolver)
parallelSolveDynamicQ  numThreads gen solver = do
    taskQueue <- newTQueueIO
    resultsVar <- newEmptyMVar

    let vars = selectRandomVars gen solver
    let subproblems = generateSubproblems vars solver
    atomically $ mapM_ (writeTQueue taskQueue) subproblems

    replicateM_ numThreads $ forkIO $ worker taskQueue resultsVar
    takeMVar resultsVar
  where
    worker taskQueue resultsVar = do
        maybeTask <- atomically $ tryReadTQueue taskQueue
        case maybeTask of
            Nothing -> return ()  -- No more work
            Just subproblem -> do
                case solve subproblem of
                    Just solution -> putMVar resultsVar (Just solution)
                    Nothing -> do
                        -- find & add new subproblem dynamically
                        let newSubproblems = splitSubproblem solver
                        atomically $ mapM_ (writeTQueue taskQueue) newSubproblems
                        worker taskQueue resultsVar  -- Continue working

splitSubproblem :: SatSolver -> [SatSolver]
splitSubproblem solver =
    let variable = selectBranchVar solver
    in [ solver { bindings = IM.insert variable True (bindings solver) },
         solver { bindings = IM.insert variable False (bindings solver) }
       ]

parallelSolveQueue :: Int -> StdGen -> SatSolver -> IO (Maybe SatSolver)
parallelSolveQueue numThreads gen solver = do
    taskQueue <- newTQueueIO -- shared work queue
    resultsVar <- newEmptyMVar -- result

    let vars = selectRandomVars gen solver
    let subproblems = generateSubproblems vars solver
    
    -- add the subproblems to the queue. *atomically* used for atomic transaction
    atomically $ mapM_ (writeTQueue taskQueue) subproblems
    
    -- start parallel processing
    replicateM_ numThreads $ forkIO $ worker taskQueue resultsVar
    takeMVar resultsVar -- blocked until a result is added
  where
    worker taskQueue resultsVar = do
        maybeTask <- atomically $ tryReadTQueue taskQueue -- read a task
        case maybeTask of
            Nothing -> return () -- exit with no left work
            Just subproblem -> do
                let result = solve subproblem
                case result of
                    Just solution -> putMVar resultsVar (Just solution)
                    Nothing -> worker taskQueue resultsVar

parallelSolveOne :: StdGen -> SatSolver -> Maybe SatSolver
parallelSolveOne gen solver =
    let vars = selectRandomVars gen solver
        subproblems = generateSubproblems vars solver
        results = parMap rdeepseq solve subproblems
    in listToMaybe (mapMaybe id results) -- return first solution

selectRandomVars :: StdGen -> SatSolver -> [Var]
selectRandomVars gen solver =
    let allVars = IS.toList $ IS.fromList 
                  [var lit | clause <- clauses solver, lit <- literals clause]
        indices = take 5 $ randomRs (0, length allVars - 1) gen -- take 5 random vars
    in map (allVars !!) indices

generateSubproblems :: [Var] -> SatSolver -> [SatSolver]
generateSubproblems vars solver =
    -- some assignments may fail due to conflicts, filter them
    mapMaybe (`applyAssignment` solver) (generateAssignments vars)

generateAssignments :: [Var] -> [[Lit]]
generateAssignments vars =
    [[mkLit v val | (v, val) <- zip vars vals] | vals <- sequence (replicate (length vars) [True, False])]

applyAssignment :: [Lit] -> SatSolver -> Maybe SatSolver
applyAssignment lits baseSolver =
    foldM (\solver lit -> guess lit solver) baseSolver lits