module Main (main) where

import Control.DeepSeq (force)
import qualified DPLL.Clause as DClause
import qualified DPLL.DpllSolver as Solver
import qualified DPLL.Literal as DLiteral
import qualified DPLL.ParallelDpll as ParallelSolver
import qualified Data.IntMap as IM
import Data.Maybe (mapMaybe)
import GHC.Conc (getNumCapabilities)
import SatGen (CNF, cnfToDimacs, generateSatisfiableCNF)
import SatBruteForce (solveSATParallel)
import System.Random (newStdGen)
import System.Directory (doesFileExist)

convertCNF :: CNF -> [DClause.Clause]
convertCNF cnf =
  map (\lits -> DClause.mkClause False (map litFromInt lits)) cnf
  where
    litFromInt x
      | x > 0 = DLiteral.mkLit x False
      | otherwise = DLiteral.mkLit (-x) True

saveCNFAsDimacs :: FilePath -> Int -> CNF -> IO ()
saveCNFAsDimacs path numVars cnf = do
  let dimacs = cnfToDimacs numVars cnf
  writeFile path dimacs
  putStrLn $ "CNF saved in DIMACS format to " ++ path

loadDimacs :: FilePath -> IO (Maybe CNF)
loadDimacs path = do
  content <- readFile path
  let parseClause line =
        let literals = takeWhile (/= 0) . map read . words $ line
         in if null literals then Nothing else Just literals
  let cnf = mapMaybe parseClause . filter (not . null) . filter ((/= 'p') . head) . lines $ content
  return $ if null cnf then Nothing else Just cnf

generateAndSaveCNF :: FilePath -> Int -> Int -> Int -> IO ()
generateAndSaveCNF dimacsPath numVars numClauses clauseLen = do
  putStrLn $ "Generating CNF with " ++ show numVars ++ " variables, " ++ show numClauses ++ " clauses, clause length " ++ show clauseLen
  cnf <- generateSatisfiableCNF numVars numClauses clauseLen
  saveCNFAsDimacs dimacsPath numVars cnf

solveBruteForceCNF :: FilePath -> IO ()
solveBruteForceCNF path = do
  maybeCnf <- loadDimacs path 
  case maybeCnf of
    Nothing -> putStrLn "Failed to load CNF from DIMACS file."
    Just cnf -> do
      let numVars = countVariables cnf  
      let forcedCnf = force cnf 
      case solveSATParallel forcedCnf numVars of
        Nothing -> putStrLn "Brute Force Solver returned UNSATISFIABLE."
        Just result -> do
          putStrLn "Brute Force Solver returned SATISFIABLE."
          putStrLn $ "Satisfying Assignment: " ++ show result

countVariables :: CNF -> Int
countVariables cnf = 
  maximum [abs lit | clause <- cnf, lit <- clause]

solveParallelCNF :: FilePath -> IO ()
solveParallelCNF path = do
  maybeCnf <- loadDimacs path
  case maybeCnf of
    Nothing -> putStrLn "Failed to load CNF from DIMACS file."
    Just cnf -> do
      let clauses = force $ convertCNF cnf
      let solver = force $ Solver.newSatSolver {Solver.clauses = clauses}

      gen <- newStdGen
      case ParallelSolver.parallelSolveOne gen solver of
        Nothing -> putStrLn "Parallel Solver returned UNSATISFIABLE"
        Just result -> do
          putStrLn "Parallel Solver returned SATISFIABLE"
          let solverBindings = Solver.bindings result
          validateSolution cnf solverBindings

solveParallelQueueCNF :: FilePath -> IO ()
solveParallelQueueCNF path = do
  maybeCnf <- loadDimacs path
  case maybeCnf of
    Nothing -> putStrLn "Failed to load CNF from DIMACS file."
    Just cnf -> do
      let clauses = force $ convertCNF cnf
      let solver = force $ Solver.newSatSolver {Solver.clauses = clauses}

      gen <- newStdGen
      numThreads <- getNumCapabilities
      parallelResult <- ParallelSolver.parallelSolveQueue numThreads gen solver
      case parallelResult of
        Nothing -> putStrLn "Parallel Solver returned UNSATISFIABLE"
        Just result -> do
          putStrLn "Parallel Solver returned SATISFIABLE"
          let solverBindings = Solver.bindings result
          validateSolution cnf solverBindings

validateSolution :: CNF -> IM.IntMap Bool -> IO ()
validateSolution cnf bindings = do
  let checkClause clause =
        any
          ( \lit ->
              (lit > 0 && IM.findWithDefault False lit bindings)
                || (lit < 0 && not (IM.findWithDefault False (-lit) bindings))
          )
          clause
  let allSatisfied = all checkClause cnf
  if allSatisfied
    then putStrLn "Solution is valid"
    else putStrLn "Solution is INVALID"

main :: IO ()
main = do
  numThreads <- getNumCapabilities
  putStrLn $ "Number of threads available: " ++ show numThreads

  let dimacsPath = "generated.cnf"
  fileExists <- doesFileExist dimacsPath
  if not fileExists
    then do
      -- if file exists, use the current file. else generate new one
      putStrLn "Generating CNF..."
      -- uncomment for brute force data
      -- generateAndSaveCNF dimacsPath 25 75 5
      generateAndSaveCNF dimacsPath 100 50000 5
    else putStrLn "CNF file already exists. Using the existing CNF."

  putStrLn "\nSolving Parallel CNF:"
  -- solveBruteForceCNF "generated.cnf"
  -- solveParallelCNF "generated.cnf"
  solveParallelQueueCNF "generated.cnf"