module SolverSequential
( bestMoveSequential
) where

import Board
import Data.Maybe (fromMaybe)
import Data.List (maximumBy)
import Data.Ord (comparing)

bestMoveSequential :: Board -> Player -> Maybe (Int,Int)
bestMoveSequential board player =
    let moves = availableMoves board
    in if null moves
       then Nothing
       else Just $ fst $ maximumBy (comparing snd)
            [(m, alphabeta (fromMaybe board (makeMove board player m)) 
                         (next player) 
                         (-infinity) 
                         infinity
                         False) | m <- moves]
  where
    infinity = 1000
    
    next PlayerX = PlayerO
    next PlayerO = PlayerX

    -- Alpha-beta pruning algorithm
    -- isMax: True if maximizing player, False if minimizing
    -- alpha: best value for maximizing player
    -- beta: best value for minimizing player
    alphabeta :: Board -> Player -> Int -> Int -> Bool -> Int
    alphabeta b curPlayer alpha beta isMax =
        case checkWin b of
          Just p -> score p
          Nothing -> 
            if isFull b
            then 0
            else if null (availableMoves b) 
                 then 0
                 else if isMax 
                      then maximizeAB b curPlayer alpha beta
                      else minimizeAB b curPlayer alpha beta

    -- Maximize with alpha-beta pruning
    maximizeAB :: Board -> Player -> Int -> Int -> Int
    maximizeAB b curPlayer alpha beta = 
        let moves = availableMoves b
            nextP = next curPlayer
        in foldr (maxLoop nextP) alpha moves
      where
        maxLoop nextP move alpha' =
            if alpha' >= beta 
            then alpha'  -- Beta cutoff - no need to search further
            else max alpha' value
          where
            newBoard = fromMaybe b (makeMove b curPlayer move)
            value = alphabeta newBoard nextP alpha' beta False

    -- Minimize with alpha-beta pruning
    minimizeAB :: Board -> Player -> Int -> Int -> Int
    minimizeAB b curPlayer alpha beta =
        let moves = availableMoves b
            nextP = next curPlayer
        in foldr (minLoop nextP) beta moves
      where
        minLoop nextP move beta' =
            if alpha >= beta'
            then beta'  -- Alpha cutoff - no need to search further
            else min beta' value
          where
            newBoard = fromMaybe b (makeMove b curPlayer move)
            value = alphabeta newBoard nextP alpha beta' True

    -- Scoring function
    score :: Player -> Int
    score PlayerX = 1    -- Maximizing player
    score PlayerO = -1   -- Minimizing player