module SolverParallel
( bestMoveParallel
) where

import Board
import Control.Parallel.Strategies
import Data.Maybe (fromMaybe)
import Data.List (maximumBy)
import Data.Ord (comparing)

bestMoveParallel :: Board -> Player -> Maybe (Int,Int)
bestMoveParallel board player =
    let moves = availableMoves board
    in if null moves
       then Nothing
       else 
         let scoredMoves = runEval $ do
               let nextP = next player
               -- Parallel evaluation of first level moves
               parList rdeepseq [ (m, alphabeta (fromMaybe board (makeMove board player m)) 
                                              nextP
                                              (-infinity) 
                                              infinity 
                                              False) | m <- moves]
         in Just $ fst $ maximumBy (comparing snd) scoredMoves
  where
    infinity = 1000
    
    next PlayerX = PlayerO
    next PlayerO = PlayerX

    -- Alpha-beta pruning algorithm
    -- isMax: True if maximizing player, False if minimizing
    -- alpha: best value for maximizing player
    -- beta: best value for minimizing player
    alphabeta :: Board -> Player -> Int -> Int -> Bool -> Int
    alphabeta b curPlayer alpha beta isMax =
        case checkWin b of
          Just p -> score p
          Nothing -> 
            if isFull b 
            then 0
            else if null (availableMoves b)
                 then 0
                 else if isMax
                      then maximizeAB b curPlayer alpha beta
                      else minimizeAB b curPlayer alpha beta

    -- Maximize with alpha-beta pruning
    -- Uses chunking for parallel evaluation
    maximizeAB :: Board -> Player -> Int -> Int -> Int
    maximizeAB b curPlayer alpha beta =
        let moves = availableMoves b
            nextP = next curPlayer
            -- Evaluate chunks of moves in parallel
            moveChunks = chunksOf 4 moves  -- Adjust chunk size based on your system
            evaluateChunk chunk = runEval $ do
                values <- parList rdeepseq 
                           [alphabeta (fromMaybe b (makeMove b curPlayer move))
                                    nextP alpha beta False | move <- chunk]
                return $ maximum values
            chunkResults = map evaluateChunk moveChunks
        in if null moves
           then alpha
           else maximum (runEval $ parList rdeepseq chunkResults)

    -- Minimize with alpha-beta pruning
    -- Uses chunking for parallel evaluation
    minimizeAB :: Board -> Player -> Int -> Int -> Int
    minimizeAB b curPlayer alpha beta =
        let moves = availableMoves b
            nextP = next curPlayer
            -- Evaluate chunks of moves in parallel
            moveChunks = chunksOf 4 moves  -- Adjust chunk size based on your system
            evaluateChunk chunk = runEval $ do
                values <- parList rdeepseq 
                           [alphabeta (fromMaybe b (makeMove b curPlayer move))
                                    nextP alpha beta True | move <- chunk]
                return $ minimum values
            chunkResults = map evaluateChunk moveChunks
        in if null moves
           then beta
           else minimum (runEval $ parList rdeepseq chunkResults)

    -- Helper function to split list into chunks
    chunksOf :: Int -> [a] -> [[a]]
    chunksOf _ [] = []
    chunksOf n xs = take n xs : chunksOf n (drop n xs)

    -- Scoring function
    score :: Player -> Int
    score PlayerX = 1    -- Maximizing player
    score PlayerO = -1   -- Minimizing player