{-# LANGUAGE RecordWildCards #-}

module Board 
( Player(..)
, Cell(..)
, Board(..)
, emptyBoard
, makeMove
, availableMoves
, checkWin
, isFull
, boardSize
) where

import Data.List (transpose, isInfixOf)

data Player = PlayerX | PlayerO deriving (Eq, Show)
data Cell = Empty | Mark Player deriving (Eq, Show)
data Board = Board { n :: Int
                   , k :: Int
                   , cells :: [[Cell]] 
                   } deriving (Show, Eq)

-- Create an empty NxN board with given K for win condition
emptyBoard :: Int -> Int -> Board
emptyBoard size winK = Board size winK (replicate size (replicate size Empty))

boardSize :: Board -> Int
boardSize Board{..} = n

-- Make a move at (r,c) for a given Player if Empty
makeMove :: Board -> Player -> (Int, Int) -> Maybe Board
makeMove b@Board{..} p (r,c) = 
    if r < 0 || c < 0 || r >= n || c >= n then Nothing else
    case (cells !! r !! c) of
      Empty -> Just $ b { cells = replace r (replace c (Mark p) (cells !! r)) cells }
      _     -> Nothing
  where
    replace i e xs = take i xs ++ [e] ++ drop (i+1) xs

-- List of available moves (row,col)
availableMoves :: Board -> [(Int,Int)]
availableMoves Board{..} = [(r,c) | r <- [0..n-1], c <- [0..n-1], cells !! r !! c == Empty]

-- Check if the board is full
isFull :: Board -> Bool
isFull Board{..} = all (/= Empty) (concat cells)

-- Check for a win condition for a given player
checkWin :: Board -> Maybe Player
checkWin Board{..} =
   if isWinner PlayerX then Just PlayerX
   else if isWinner PlayerO then Just PlayerO
   else Nothing
  where
    linesToCheck = rows ++ cols ++ diags
    rows = cells
    cols = transpose cells
    diags = diagonals cells ++ diagonals (map reverse cells)
    diagonals m = [ [m !! i !! (i+j) | i <- [0..n-1], i+j >=0, i+j<n] | j <- [-(n-1)..(n-1)] ]
    isWinner p = any (hasKConsecutive (Mark p)) linesToCheck

    hasKConsecutive mark line = 
      let marks = map (\x -> if x == mark then 'X' else '.') line
          pattern = replicate k 'X'
      in pattern `isInfixOf` marks

