module DIMACSParserSpec (tests) where

import Test.HUnit
import DIMACSParser(parseDIMACS)
import Lib(Clause(..), Literal(..))

-- Unit tests for parseDIMACS
tests :: Test
tests = TestList
  [ testSatisfiableInput
  , testUnsatisfiableInput
  , testTooLittleClauses
  , testTooManyClauses
  , testMissingProblemLine
  , testInvalidProblemLine
  , testInvalidProblemLine2
  , testExtraComments
  , testInvalidLiteral
  , testInvalidLiteral2
  ]

testSatisfiableInput :: Test
testSatisfiableInput = TestCase $
  let input = unlines
                [ "c Example satisfiable DIMACS CNF input file"
                , "p cnf 3 3"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0" ]
      expected :: Either String (Int, [Clause])
      expected = Right
                   (3,
                   [ Clause [LiteralPos 0, LiteralNeg 1, LiteralPos 2]
                   , Clause [LiteralNeg 0, LiteralPos 1]
                   , Clause [LiteralNeg 1, LiteralPos 2] ])
  in assertEqual "Valid satisfiable DIMACS input should parse correctly" expected (parseDIMACS input)

testUnsatisfiableInput :: Test
testUnsatisfiableInput = TestCase $
  let input = unlines
                [ "c Example unsatisfiable DIMACS CNF input file"
                , "p cnf 1 2"
                , "1 0"
                , "-1 0" ]
      expected :: Either String (Int, [Clause])
      expected = Right
                   (1,
                   [ Clause [LiteralPos 0]
                   , Clause [LiteralNeg 0] ])
  in assertEqual "Valid unsatisfiable DIMACS input should parse correctly" expected (parseDIMACS input)

testTooLittleClauses :: Test
testTooLittleClauses = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file with too little clauses"
                ,"p cnf 3 2"
                , "-2 3 0"]
      expected = Left "Error: expected 2 clauses, found 1"
  in assertEqual "Mismatch in expected vs. actual number of clauses should return an error" expected (parseDIMACS input)

testTooManyClauses :: Test
testTooManyClauses = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file with too many clauses"
                ,"p cnf 3 2"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0"]
      expected = Left "Error: expected 2 clauses, found 3"
  in assertEqual "Mismatch in expected vs. actual number of clauses should return an error" expected (parseDIMACS input)

testMissingProblemLine :: Test
testMissingProblemLine = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file with missing problem line"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0" ]
      expected = Left "Error: invalid or missing problem line"
  in assertEqual "Missing problem line should return an error" expected (parseDIMACS input)

testInvalidProblemLine :: Test
testInvalidProblemLine = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file with invalid problem line"
                , "p foo"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0" ]
      expected = Left "Error: invalid or missing problem line"
  in assertEqual "Missing problem line should return an error" expected (parseDIMACS input)

testInvalidProblemLine2 :: Test
testInvalidProblemLine2 = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file with invalid problem line"
                , "p cnf 3 y"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0" ]
      expected = Left "Error: invalid or missing problem line"
  in assertEqual "Missing problem line should return an error" expected (parseDIMACS input)
  
testExtraComments :: Test
testExtraComments = TestCase $
  let input = unlines
                [ "c Example DIMACS CNF input file"
                , "c Additional comment line"
                , "c Another additional comment line"
                , "p cnf 3 3"
                , "1 -2 3 0"
                , "-1 2 0"
                , "-2 3 0" ]
      expected :: Either String (Int, [Clause])
      expected = Right
                   (3,
                   [ Clause [LiteralPos 0, LiteralNeg 1, LiteralPos 2]
                   , Clause [LiteralNeg 0, LiteralPos 1]
                   , Clause [LiteralNeg 1, LiteralPos 2] ])
  in assertEqual "Extra comment lines should be ignored" expected (parseDIMACS input)

testInvalidLiteral :: Test
testInvalidLiteral = TestCase $
  let input = unlines
                [ "c Example with invalid literal"
                , "p cnf 3 2"
                , "1 -2 4 0" -- only 3 variables declared
                , "-1 3 0" ]
      expected :: Either String (Int, [Clause])
      expected = Left "Error: invalid literal 4 for 3 variables"
  in assertEqual "Reference to more variables than declared should return an error" expected (parseDIMACS input)

testInvalidLiteral2 :: Test
testInvalidLiteral2 = TestCase $
  let input = unlines
                [ "c Example with invalid literal"
                , "p cnf 3 2"
                , "1 -2 -4 0" -- only 3 variables declared
                , "-1 3 0" ]
      expected :: Either String (Int, [Clause])
      expected = Left "Error: invalid literal 4 for 3 variables"
  in assertEqual "Reference to more variables than declared should return an error" expected (parseDIMACS input)
