module DIMACSParser (parseDIMACS) where

import Lib(Clauses(..), Literal(..))
import qualified Data.List as L
import Data.Char (isDigit)

-- Parse given DIMACS input file into `(numVars, [Clause])`
parseDIMACS :: String -> Either String (Int, [Clauses])
parseDIMACS content = do
    let (_, remainingLines) = span (L.isPrefixOf "c") (lines content)
    case remainingLines of
         (problemLine:clauseLines) -> do
           (numVars, numClauses) <- parseProblemLine problemLine
           clauses <- parseClauses numVars numClauses clauseLines
           return (numVars, clauses)
         _ -> Left "Error: missing problem line and clauses"

-- Parse expected problem line
parseProblemLine :: String -> Either String (Int, Int)
parseProblemLine line =
  case words line of
    ["p", "cnf", vars, clauses] | all (all isDigit) [vars, clauses] ->
      Right (read vars, read clauses)
    _ -> Left "Error: invalid or missing problem line"

-- Parse and validate clause lines
parseClauses :: Int -> Int -> [String] -> Either String [Clauses]
parseClauses numVars numClauses clauseLines
  | length clauseLines /= numClauses = Left $ "Error: expected " ++ show numClauses ++ " clauses, found " ++ show (length clauseLines)
  | otherwise = mapM (parseClause numVars) clauseLines

-- Parse a single clause
parseClause :: Int -> String -> Either String Clauses
parseClause numVars line = do
  let literalStrings = init $ words line -- Drop trailing '0'
  literals <- mapM parseLiteral literalStrings
  return $ Clause literals
  where
    -- Parse a single literal within a clause
    parseLiteral :: String -> Either String Literal
    parseLiteral lString
      | head lString == '-' && (all isDigit) (tail lString) = validateBounds ((read (tail lString) :: Int)) LiteralNeg
      | (all isDigit) lString = validateBounds ((read lString :: Int)) LiteralPos
      | otherwise = Left $ "Error: invalid literal " ++ lString

    -- Ensure a clause doesn't contain more variables than declared in problem line
    validateBounds :: Int -> (Int -> Literal) -> Either String Literal
    validateBounds lInt constructor
      | lInt < 1 || lInt > numVars = Left $ "Error: invalid literal " ++ show lInt ++ " for " ++ show numVars ++ " variables"
      | otherwise = Right $ constructor (lInt-1)

