module Utils (parseLineOfMoves, logMsg, LogLevel(..)) where

import Data.Char (isSpace)
import Data.List (dropWhileEnd)
import RubikCube

-- Helper functions to parse the input file
parseMove :: String -> Move
parseMove str = case dropSpaces str of
  "F"  -> F
  "Fi" -> Fi
  "R"  -> R
  "Ri" -> Ri
  "U"  -> U
  "Ui" -> Ui
  "B"  -> B
  "Bi" -> Bi
  "L"  -> L
  "Li" -> Li
  "D"  -> D
  "Di" -> Di
  _    -> error $ "Unknown move: " ++ str

dropSpaces :: String -> String
dropSpaces = dropWhileEnd isSpace . dropWhile isSpace

-- A line of moves might look like:
-- "U, R, F, B, L, D"
-- We split by comma and trim spaces, then parse each move.
parseLineOfMoves :: String -> [Move]
parseLineOfMoves line =
    let parts = map dropSpaces (splitBy ',' line)
    in map parseMove parts

splitBy :: Char -> String -> [String]
splitBy delim s = case break (== delim) s of
  (chunk, [])     -> [chunk]
  (chunk, _:rest) -> chunk : splitBy delim rest


-- Define log levels
data LogLevel = DEBUG | INFO | ERROR
  deriving (Eq, Ord, Show, Read)

-- Logging function
logMsg :: LogLevel -> LogLevel -> String -> IO ()
logMsg currentLevel msgLevel msg =
  -- Only print if msgLevel <= currentLevel (lower or equal means more important message)
  if msgLevel >= currentLevel
    then putStrLn msg
    else return ()