{-# OPTIONS_GHC -Wall #-}
module RubikCube (
    Cube(..),
    Face,
    Color,
    Move(..),
    printCube,
    allMoves,
    initCube,
    applyMove,
    applyMoves,
) where 

import Data.List (transpose)

type Color = Char -- 'R', 'G', 'B', 'Y', 'O', 'W' for colors

-- A face is a 2D array (list of lists) of colors
type Face = [[Color]]

-- The Cube is a list of faces
data Cube = Cube {
    up    :: Face,
    down  :: Face,
    left  :: Face,
    right :: Face,
    front :: Face,
    back  :: Face
} deriving (Eq, Show)

initCube :: Int -> Cube
initCube n = Cube {
    up = replicate n (replicate n 'W'),
    down = replicate n (replicate n 'Y'),
    left = replicate n (replicate n 'O'),
    right = replicate n (replicate n 'R'),
    front = replicate n (replicate n 'G'),
    back = replicate n (replicate n 'B')
}

-- Function to display a face
printFace :: Face -> IO ()
printFace face = mapM_ print face

-- Function to display the entire cube
printCube :: Cube -> IO ()
printCube cube = do
    putStrLn "Up Face:"
    printFace (up cube)
    putStrLn "Front Face:"
    printFace (front cube)
    putStrLn "Right Face:"
    printFace (right cube)
    putStrLn "Back Face:"
    printFace (back cube)
    putStrLn "Left Face:"
    printFace (left cube)
    putStrLn "Down Face:"
    printFace (down cube)


data Move
    = F | Fi | R | Ri | U | Ui | B | Bi | L | Li | D | Di
    deriving (Eq, Show)

-- Helper functions needed to perform moves...

-- Rotate the faces...

-- Rotate a face 90 degrees clockwise
-- Original face:
-- 1 2 3
-- 4 5 6
-- 7 8 9
-- After transpose:
-- 1 4 7
-- 2 5 8
-- 3 6 9
-- After map reverse:
-- 7 4 1
-- 8 5 2
-- 9 6 3
rotateFaceClockwise :: Face -> Face
rotateFaceClockwise = map reverse . transpose

-- Rotate a face 90 degrees counterclockwise
-- Original face:
-- 1 2 3
-- 4 5 6
-- 7 8 9
-- After map reverse:
-- 3 2 1
-- 6 5 4
-- 9 8 7
-- After transpose:
-- 3 6 9
-- 2 5 8
-- 1 4 7
rotateFaceCounterClockwise :: Face -> Face
rotateFaceCounterClockwise = transpose . map reverse


-- Replace the rows/columns in a face...

-- Replaces the row at index idx in face with newRow.

-- Example:
-- face = [
--     ['A', 'B', 'C'],  -- Row 0
--     ['D', 'E', 'F'],  -- Row 1
--     ['G', 'H', 'I']   -- Row 2
-- ]
-- idx = 1
-- newRow = ['X', 'Y', 'Z']

-- Steps:
-- 1. take 1 face => [['A', 'B', 'C']]
-- 2. [newRow] => [['X', 'Y', 'Z']]
-- 3. drop (1 + 1) face => [['G', 'H', 'I']]
-- 4. Concatenate: [['A', 'B', 'C']] ++ [['X', 'Y', 'Z']] ++ [['G', 'H', 'I']]

-- Result:
-- [
--   ['A', 'B', 'C'],
--   ['X', 'Y', 'Z'],
--   ['G', 'H', 'I']
-- ]
replaceRow :: Face -> Int -> [Color] -> Face
replaceRow face idx newRow = take idx face ++ [newRow] ++ drop (idx + 1) face

-- Replaces the column at index idx in face with newCol.

-- Example:
-- face = [
--     ['A', 'B', 'C'],  -- Row 0
--     ['D', 'E', 'F'],  -- Row 1
--     ['G', 'H', 'I']   -- Row 2
-- ]
-- idx = 0
-- newCol = ['X', 'Y', 'Z']

-- Steps:
-- 1. ['A', 'B', 'C'] 'X' => [] ++ ['X'] ++ ['B', 'C'] => ['X', 'B', 'C']
-- 2. ['D', 'E', 'F'] 'Y' => [] ++ ['Y'] ++ ['E', 'F'] => ['Y', 'E', 'F']
-- 3. ['G', 'H', 'I'] 'Z' => [] ++ ['Z'] ++ ['H', 'I'] => ['Z', 'H', 'I']

-- Result:
-- [
--     ['X', 'B', 'C'],
--     ['Y', 'E', 'F'],
--     ['Z', 'H', 'I']
-- ]

replaceCol :: Face -> Int -> [Color] -> Face
replaceCol face idx newCol = zipWith (\row newVal -> take idx row ++ [newVal] ++ drop (idx + 1) row) face newCol

-- Get the column at index idx in the face
getCol :: Face -> Int -> [Color]
getCol face idx = map (!! idx) face


-- NOTE: although our cube defined above is nxn, the following code currently only works on 2x2 or 3x3

-- A 2x2 or 3x3 cube has 12 possible moves. A 4x4 or larger cube will have more!
allMoves :: [Move]
allMoves = [F, Fi, R, Ri, U, Ui, B, Bi, L, Li, D, Di]

-- Function to map a Move to its corresponding Cube -> Cube function
applyMove :: Move -> Cube -> Cube
applyMove F  = moveF
applyMove Fi = moveFi
applyMove R  = moveR
applyMove Ri = moveRi
applyMove U  = moveU
applyMove Ui = moveUi
applyMove B  = moveB
applyMove Bi = moveBi
applyMove L  = moveL
applyMove Li = moveLi
applyMove D  = moveD
applyMove Di = moveDi

-- Function to apply a list of Moves sequentially to a Cube
applyMoves :: [Move] -> Cube -> Cube
applyMoves moves cube = foldl (\c m -> applyMove m c) cube moves


-- Implement all these moves...

-- Perform a move that rotates the front face clockwise
-- 1. front: rotate the front face
-- 2. up: update the last row of the up face with reversed values from the last column of the left face.
-- 3. left: update the last column of the left face with the first row of the down face.
-- 4. down: update the first row of the down face with reversed values from the first column of the right face.
-- 5. right: update the first column of the right face with the last row of the up face.
-- 6. back: no update required
moveF :: Cube -> Cube
moveF cube = cube {
    front = rotateFaceClockwise (front cube),
    up = replaceRow (up cube) (n-1) (reverse (getCol (left cube) (n-1))),
    left = replaceCol (left cube) (n-1) (downFirstRow),
    down = replaceRow (down cube) 0 (reverse (getCol (right cube) 0)),
    right = replaceCol (right cube) 0 (upLastRow)
}
  where
    n = length (front cube)
    upLastRow = (up cube) !! (n-1)
    downFirstRow = (down cube) !! 0


-- For time reasons, explanations of the rest of the moves are not provided.
-- If you understand how moveF works, the implementation below is basically self-explanatory.

-- Perform a move that rotates the front face counter-clockwise
moveFi :: Cube -> Cube
moveFi cube = cube {
    front = rotateFaceCounterClockwise (front cube),
    up = replaceRow (up cube) (n - 1) (getCol (right cube) 0),
    right = replaceCol (right cube) 0 (reverse downFirstRow),
    down = replaceRow (down cube) 0 (getCol (left cube) (n - 1)),
    left = replaceCol (left cube) (n - 1) (reverse upLastRow)
}
  where
    n = length (front cube)
    upLastRow = (up cube) !! (n - 1)
    downFirstRow = (down cube) !! 0

-- Perform a move that rotates the right face clockwise
moveR :: Cube -> Cube
moveR cube = cube {
    right = rotateFaceClockwise (right cube),
    up = replaceCol (up cube) (n - 1) (getCol (front cube) (n - 1)),
    front = replaceCol (front cube) (n - 1) (getCol (down cube) (n - 1)),
    down = replaceCol (down cube) (n - 1) (reverse (getCol (back cube) 0)),
    back = replaceCol (back cube) 0 (reverse (getCol (up cube) (n - 1)))
}
  where
    n = length (front cube)

-- Perform a move that rotates the right face counter-clockwise
moveRi :: Cube -> Cube
moveRi cube = cube {
    right = rotateFaceCounterClockwise (right cube),
    up = replaceCol (up cube) (n - 1) (reverse (getCol (back cube) 0)),
    back = replaceCol (back cube) 0 (reverse (getCol (down cube) (n - 1))),
    down = replaceCol (down cube) (n - 1) (getCol (front cube) (n - 1)),
    front = replaceCol (front cube) (n - 1) (getCol (up cube) (n - 1))
}
  where
    n = length (front cube)

-- Perform a move that rotates the up face clockwise
moveU :: Cube -> Cube
moveU cube = cube {
    up = rotateFaceClockwise (up cube),
    front = replaceRow (front cube) 0 (rightFirstRow),
    right = replaceRow (right cube) 0 (backFirstRow),
    back = replaceRow (back cube) 0 (leftFirstRow),
    left = replaceRow (left cube) 0 (frontFirstRow)
}
  where
    frontFirstRow = (front cube) !! 0
    rightFirstRow = (right cube) !! 0
    backFirstRow = (back cube) !! 0
    leftFirstRow = (left cube) !! 0

-- Perform a move that rotates the up face counter-clockwise
moveUi :: Cube -> Cube
moveUi cube = cube {
    up = rotateFaceCounterClockwise (up cube),
    front = replaceRow (front cube) 0 (leftFirstRow),
    left = replaceRow (left cube) 0 (backFirstRow),
    back = replaceRow (back cube) 0 (rightFirstRow),
    right = replaceRow (right cube) 0 (frontFirstRow)
}
  where
    frontFirstRow = (front cube) !! 0
    rightFirstRow = (right cube) !! 0
    backFirstRow = (back cube) !! 0
    leftFirstRow = (left cube) !! 0

-- Perform a move that rotates the back face clockwise
moveB :: Cube -> Cube
moveB cube = cube {
    back = rotateFaceClockwise (back cube),
    up = replaceRow (up cube) 0 (getCol (right cube) (n - 1)),
    right = replaceCol (right cube) (n - 1) (reverse downLastRow),
    down = replaceRow (down cube) (n - 1) (getCol (left cube) 0),
    left = replaceCol (left cube) 0 (reverse upFirstRow)
}
  where
    n = length (front cube)
    upFirstRow = (up cube) !! 0
    downLastRow = (down cube) !! (n - 1)

-- Perform a move that rotates the back face counter-clockwise
moveBi :: Cube -> Cube
moveBi cube = cube {
    back = rotateFaceCounterClockwise (back cube),
    up = replaceRow (up cube) 0 (reverse (getCol (left cube) 0)),
    left = replaceCol (left cube) 0 (downLastRow),
    down = replaceRow (down cube) (n - 1) (reverse (getCol (right cube) (n - 1))),
    right = replaceCol (right cube) (n - 1) (upFirstRow)
}
  where
    n = length (front cube)
    upFirstRow = (up cube) !! 0
    downLastRow = (down cube) !! (n - 1)

-- Perform a move that rotates the left face clockwise
moveL :: Cube -> Cube
moveL cube = cube {
    left = rotateFaceClockwise (left cube),
    up = replaceCol (up cube) 0 (reverse (getCol (back cube) (n - 1))),
    back = replaceCol (back cube) (n - 1) (reverse (getCol (down cube) 0)),
    down = replaceCol (down cube) 0 (getCol (front cube) 0),
    front = replaceCol (front cube) 0 (getCol (up cube) 0)
}
  where
    n = length (front cube)

-- Perform a move that rotates the left face counter-clockwise
moveLi :: Cube -> Cube
moveLi cube = cube {
    left = rotateFaceCounterClockwise (left cube),
    up = replaceCol (up cube) 0 (getCol (front cube) 0),
    front = replaceCol (front cube) 0 (getCol (down cube) 0),
    down = replaceCol (down cube) 0 (reverse (getCol (back cube) (n - 1))),
    back = replaceCol (back cube) (n - 1) (reverse (getCol (up cube) 0))
}
  where
    n = length (front cube)

-- Perform a move that rotates the down face clockwise
moveD :: Cube -> Cube
moveD cube = cube {
    down = rotateFaceClockwise (down cube),
    front = replaceRow (front cube) (n - 1) (leftLastRow),
    left = replaceRow (left cube) (n - 1) (backLastRow),
    back = replaceRow (back cube) (n - 1) (rightLastRow),
    right = replaceRow (right cube) (n - 1) (frontLastRow)
}
  where
    n = length (front cube)
    frontLastRow = (front cube) !! (n - 1)
    rightLastRow = (right cube) !! (n - 1)
    backLastRow = (back cube) !! (n - 1)
    leftLastRow = (left cube) !! (n - 1)

-- Perform a move that rotates the down face counter-clockwise
moveDi :: Cube -> Cube
moveDi cube = cube {
    down = rotateFaceCounterClockwise (down cube),
    front = replaceRow (front cube) (n - 1) (rightLastRow),
    right = replaceRow (right cube) (n - 1) (backLastRow),
    back = replaceRow (back cube) (n - 1) (leftLastRow),
    left = replaceRow (left cube) (n - 1) (frontLastRow)
}
  where
    n = length (front cube)
    frontLastRow = (front cube) !! (n - 1)
    rightLastRow = (right cube) !! (n - 1)
    backLastRow = (back cube) !! (n - 1)
    leftLastRow = (left cube) !! (n - 1)