{-# LANGUAGE BangPatterns #-}
{-# OPTIONS_GHC -Wall #-}
module Main where

import System.Environment (getArgs)
import System.Exit (exitFailure)
import Control.Monad (forM_, when)

import RubikCube
import PDB
import IDAStar (heuristic, isGoal, idaStar)
import Utils (parseLineOfMoves, logMsg, LogLevel(..))


main :: IO ()
main = do
    args <- getArgs
    if length args < 3 || length args > 4
      then do
        putStrLn "Usage: ./IDAStarLinear <pdb_file> <n> <scramble_file> [log_level]"
        putStrLn "log_level can be DEBUG, INFO, ERROR"
        exitFailure
      else return ()

    let pdbFile       = args !! 0
    let nStr          = args !! 1
    let scrambleFile  = args !! 2

    -- Parse n
    let n = read nStr :: Int
    if n /= 2 && n /= 3
      then do
        putStrLn "Error: n must be 2 or 3."
        exitFailure
      else return ()

    -- Parse optional log level
    let logLevel = if length args == 4
                     then read (args !! 3) :: LogLevel
                     else INFO  -- default to INFO if not provided

    -- Load the PDB
    pdb <- loadPDB pdbFile

    -- Initialize solved cube
    let solvedCube = initCube n

    -- Read scramble file
    contents <- readFile scrambleFile
    let linesOfMoves = lines contents -- each line is a scramble sequence

    forM_ (zip ([1..] :: [Int]) linesOfMoves) $ \(idx, line) -> do
      logMsg logLevel INFO $ "\nSolving cube #" ++ show idx

      let scrambleMoves = parseLineOfMoves line
      let scrambledCube = applyMoves scrambleMoves solvedCube

      logMsg logLevel DEBUG "Scrambled Cube:"
      when (logLevel <= DEBUG) $ printCube scrambledCube

      logMsg logLevel INFO "Starting IDA* search..."
      maybeSolution <- idaStar scrambledCube (heuristic pdb) allMoves
      case maybeSolution of
        Just solMoves -> do
          -- Apply the solution moves to scrambledCube and verify it's solved:
          let verifiedCube = applyMoves solMoves scrambledCube
          logMsg logLevel DEBUG "After applying solution moves, cube state:"
          when (logLevel <= DEBUG) $ printCube verifiedCube
          if isGoal verifiedCube
            then logMsg logLevel INFO $ "Solution for cube #" ++ show idx ++ ": "++ unwords (map show solMoves)
            else logMsg logLevel ERROR $ "Error: no solution found for cube #" ++ show idx ++ "."

        Nothing -> logMsg logLevel ERROR $ "Error: no solution found for cube #" ++ show idx ++ "."