{-# LANGUAGE BangPatterns #-}
{-# OPTIONS_GHC -Wall #-}
module Main where

import System.Environment (getArgs)
import System.Exit (exitFailure)
import Control.Concurrent.Async (forConcurrently_)

import RubikCube
import PDB
import IDAStar (heuristic, isGoal, idaStar)
import Utils (parseLineOfMoves)


main :: IO ()
main = do
    args <- getArgs
    if length args < 3 || length args > 4
      then do
        putStrLn "Usage: ./IDAStarBatch <pdb_file> <n> <scramble_file>"
        exitFailure
      else return ()

    let pdbFile       = args !! 0
    let nStr          = args !! 1
    let scrambleFile  = args !! 2

    -- Parse n
    let n = read nStr :: Int
    if n /= 2 && n /= 3
      then do
        putStrLn "Error: n must be 2 or 3."
        exitFailure
      else return ()

    -- Load the PDB
    pdb <- loadPDB pdbFile

    -- Initialize solved cube
    let solvedCube = initCube n

    -- Read scramble file
    contents <- readFile scrambleFile
    let linesOfMoves = lines contents -- each line is a scramble sequence

    -- Solve each cube in parallel:
    forConcurrently_ (zip ([1..] :: [Int]) linesOfMoves) $ \(idx, line) -> do

      let scrambleMoves = parseLineOfMoves line
      let scrambledCube = applyMoves scrambleMoves solvedCube

      maybeSolution <- idaStar scrambledCube (heuristic pdb) allMoves
      case maybeSolution of
        Just solMoves -> do
          -- Apply the solution moves to scrambledCube and verify it's solved:
          let verifiedCube = applyMoves solMoves scrambledCube
          if isGoal verifiedCube
            then putStrLn $ "Solution for cube #" ++ show idx ++ ": "++ unwords (map show solMoves)
            else putStrLn $ "Error: no solution found for cube #" ++ show idx ++ "."

        Nothing -> putStrLn $ "Error: no solution found for cube #" ++ show idx ++ "."