{-# OPTIONS_GHC -Wall #-}
module Main where

import qualified Data.Vector.Unboxed as V
import qualified Data.Map.Strict as Map
import Text.Read (readMaybe)
import System.Environment (getArgs)
import Control.Monad (forM_)

import Data.Time (getCurrentTime, diffUTCTime)

import PDB (generatePDB, savePDB, loadPDB, Word8Vector(..))


-- Function to save a PDB to a file
savePDBToFile :: FilePath -> Int -> Int -> IO ()
savePDBToFile filePath n depthLimit = do
    putStrLn $ "Generating PDB with cube size " ++ show n ++ " and depth limit " ++ show depthLimit
    let pdb = generatePDB n depthLimit
    savePDB filePath pdb
    putStrLn $ "PDB saved to " ++ filePath

-- Function to load a PDB from a file and display its size and some entries in the PDB
loadPDBFromFile :: FilePath -> IO ()
loadPDBFromFile filePath = do
    putStrLn $ "Loading PDB from " ++ filePath
    pdb <- loadPDB filePath
    let entriesCount = Map.size pdb
    putStrLn $ "PDB contains " ++ show entriesCount ++ " entries."

    -- Print the first few entries (up to 10)
    let fewEntries = take 10 (Map.toList pdb)
    putStrLn "Some entries in the PDB (key as list of Word8 and depth):"
    forM_ fewEntries $ \(Word8Vector vec, depth) -> do  -- Unwrap Word8Vector
        let keyList = V.toList vec  -- Access the underlying vector
        putStrLn $ "Key: " ++ show keyList ++ ", Depth: " ++ show depth


-- Main function to take parameters filePath, n, and depthLimit from the command line
main :: IO ()
main = do
    args <- getArgs
    startTime <- getCurrentTime  -- Start time
    case args of
        [filePath, nStr, depthLimitStr] -> do
            -- Parse input arguments
            let maybeN = readMaybe nStr :: Maybe Int
            let maybeDepthLimit = readMaybe depthLimitStr :: Maybe Int
            case (maybeN, maybeDepthLimit) of
                (Just n, Just depthLimit) -> do
                    savePDBToFile filePath n depthLimit
                    loadPDBFromFile filePath
                _ -> putStrLn "Error: Both n and depthLimit must be valid integers."
        _ -> putStrLn "Usage: TestPDB <filePath> <n> <depthLimit>"

    endTime <- getCurrentTime  -- End time
    putStrLn $ "\nTotal time consumed: " ++ show (diffUTCTime endTime startTime)