module MDEngine
  ( forceMatrix,
    velocityVerlet,
    isf,
    mdIsf,
    mdTraj
  ) where

import Control.Parallel.Strategies (using, parList, rseq)

import MDVector

-- Simulation Parameters

epsilon :: Double
epsilon = 1.0

sigma :: Double
sigma = 2.0 ** (-1.0/6.0) -- Chosen so that r_{min} = 1

mass :: Double
mass = 1.0

dt :: Double
dt = 1e-3

-- Computes list of forces on all particles given a configuration
forceMatrix :: [MDVector] -> Double -> [MDVector]
forceMatrix rs boxLength =
  map totalForce rs `using` parList rseq
  where
    -- Gets force acting on particle at r1 due to particle at r2
    forceVector r1 r2
      | r1 == r2 = zeroVector
      | otherwise = vectorMultiply flj (unitVector r12)
        where r12 = displacement r2 r1 boxLength
              d12 = vectorNorm r12
              sor = sigma / d12
              flj = 24.0 * epsilon * (2 * (sor ** 12.0) - (sor ** 6.0)) / d12
    -- Computes total force on particle at r due to all other particles
    totalForce r = foldr vectorAdd zeroVector $ map (forceVector r) rs

-- Updates positions, velocities, and forces using velocity Verlet
velocityVerlet :: [MDVector] -> [MDVector] -> [MDVector] -> Double -> ([MDVector],[MDVector],[MDVector])
velocityVerlet rt1 vt1 ft1 boxLength =
  ( rt2 , vt2 , ft2 )
  where rt2 = addMatrixList rt1 [ (matrixMultiply dt vt1) , (matrixMultiply c1 ft1) ]
        ft2 = forceMatrix rt2 boxLength
        vt2 = addMatrixList vt1 [ (matrixMultiply c2 ft1) , (matrixMultiply c2 ft2) ]
        c1 = (dt ** 2.0) / (2.0 * mass)
        c2 = dt / (2.0 * mass)

-- Computes intermediate scattering function value between two configurations
isf :: MDVector -> [MDVector] -> [MDVector] -> Double
isf k r0 rt =
  let diffMatrix = zipWith vectorSubtract rt r0 in
  let dotMatrix = map (dotProduct k) diffMatrix in
  let cosMatrix = map cos dotMatrix in
  (sum cosMatrix) / (fromIntegral (length cosMatrix))

-- Given initial configuration, velocities, number of timesteps to execute
-- and a k-vector of interest, computes the self-ISF trajectory.
mdIsf :: [MDVector] -> [MDVector] -> Int -> Double -> MDVector -> [Double]
mdIsf r0 v0 timesteps boxLength k =
  let mdIsfHelper rt vt ft steps
        | steps == 0 = []
        | otherwise  = (isf k r0 rt) : (mdIsfHelper rt2 vt2 ft2 (steps - 1))
          where (rt2,vt2,ft2) = velocityVerlet rt vt ft boxLength
  in mdIsfHelper r0 v0 f0 timesteps
     where f0 = forceMatrix r0 boxLength

-- Given initial configuration, velocities, and number of timesteps to execute,
-- computes the trajectory of the first particle.
mdTraj :: [MDVector] -> [MDVector] -> Int -> Double -> [MDVector]
mdTraj r0 v0 timesteps boxLength =
  let mdTrajHelper rt vt ft steps
        | steps == 0 = []
        | otherwise  = (head rt) : (mdTrajHelper rt2 vt2 ft2 (steps - 1))
          where (rt2,vt2,ft2) = velocityVerlet rt vt ft boxLength
  in mdTrajHelper r0 v0 f0 timesteps
     where f0 = forceMatrix r0 boxLength
